/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCFuture.h
 *    Implementation of RPCFuture.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2010/11/11
 */

#ifndef _MIRA_RPCFUTURE_H_
#define _MIRA_RPCFUTURE_H_

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#include <boost/thread/future.hpp>
#endif

#include <rpc/AbstractRPCClient.h>
#include <rpc/JSONRPCResponse.h>
#include <rpc/RPCError.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Wrapper for boost::unique_future that is specialized for RPC processing.
 * This is a base class for the concrete derived RPCFuture classes.
 */
template <typename R>
class RPCFutureCommon : boost::noncopyable
{
public:

	RPCFutureCommon() : mClient(NULL), mCallId("") {}

	RPCFutureCommon(AbstractRPCClient* client, const std::string& callId) :
		mClient(client), mCallId(callId) {}

	~RPCFutureCommon()
	{
		// remove call upon destruction of this future
		if(mClient!=NULL)
			mClient->onDestructFuture(mCallId);
	}

public:

	/// query call ID
	const std::string& callId() { return mCallId; }

	// delegate to boost future

	/// Checks to see if the result of the RPC call associated with this future is set
	bool isReady() const { return mFuture.is_ready(); }

	/**
	 * Returns true if the RPC call associated with this future has finished
	 * with an exception rather than a return value.
	 */
	bool hasException() const { return mFuture.has_exception(); }

	/**
	 * Returns true if the RPC call associated with this future has finished
	 * with a return value value rather than an exception.
	 */
	bool hasValue() const { return mFuture.has_value(); }

	/**
	 * Waits and blocks the current thread until the result of the associated
	 * RPC call is ready.
	 * This is an interruption point.
	 */
	void wait() const { mFuture.wait(); }

	/**
	 * Waits and blocks the current thread until the result of the associated
	 * RPC call is ready, or the time duration specified by waitDuration has
	 * elapsed.
	 * This is an interruption point.
	 * @return false if the call is returning because the time specified was reached, 
	 * true otherwise. 
	 */
	template<typename Duration>
	bool timedWait(Duration const& relTime) const {
		return mFuture.timed_wait(relTime);
	}

	/**
	 * Waits and blocks the current thread until the result of the associated
	 * RPC call is ready, or the time point specified by waitDuration has passed.
	 * This is an interruption point.
	 * @return false if the call is returning because the time specified was reached, 
	 * true otherwise.
	 */
	bool timedWaitUntil(boost::system_time const& absTime) const {
		return mFuture.timed_wait_until(absTime);
	}

	/// Swaps ownership of the asynchronous results associated with other and *this.
	void swap(RPCFutureCommon<R>& other) {
		mFuture.swap(other.mFuture);
		std::swap(mClient, other.mClient);
		std::swap(mCallId, other.mCallId);
	}


protected:
	AbstractRPCClient* mClient;
	std::string mCallId;
	boost::unique_future<R> mFuture;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 *
 * An RPCFuture is a proxy for the result of an asynchronous RPC call.
 * It can be used to block the calling thread until the RPC call has finished
 * and the result is available. The use of
 * futures dramatically reduces latency in RPC calls, since the caller can
 * decide whether it wants to wait for the result or not and where it wants to
 * wait for the result.
 *
 * The RPCFuture cannot be shared, however, the ownership can be transferred
 * between different instances using the move constructor or move-assignment
 * operator, but at most one instance holds a reference to a given asynchronous
 * result. When the result is ready, it can be returned using get() by
 * rvalue-reference to allow the result to be moved or copied as appropriate
 * for the type.
 *
 * @see @ref ServiceRPCPage
 */
template <typename R>
class RPCFuture : public RPCFutureCommon<R>
{
public:
	RPCFuture() {}

	/// internally used by RPCClient
	RPCFuture(boost::unique_future<R> other, AbstractRPCClient* client,
	          const std::string& callId) :
		RPCFutureCommon<R>(client, callId)
	{
		this->mFuture.swap(other);
	}

	/// move constructor
	RPCFuture(RPCFuture&& other) noexcept {
		this->swap(other);
	}

	/// move assignment operator
	RPCFuture& operator=(RPCFuture && other) noexcept {
		this->swap(other);
		return *this;
	}

public:

	/// The rvalue return value, which essentially can be thought of R&&

#if BOOST_VERSION / 100 % 1000 > 56
	typedef typename boost::detail::shared_state<R>::move_dest_type ReturnValue;
#else
	typedef typename boost::detail::future_traits<R>::move_dest_type ReturnValue;
#endif
	/**
	 * Casts the return value of the RPC call associated to this future to R&&.
	 * @note This cast will block until the result becomes available.
	 * @see wait(), timedWait(), hasValue()
	 */
	operator ReturnValue() { return this->mFuture.get(); }

	/**
	 * Obtains the return value of the RPC call associated to this future as
	 * rvalue R&&.
	 * @param[in] throwXRPC throw XRPC instead of original exception that occured in the
	 *                      called function (default = true, for backward compatibility)
	 * @param[in] recursive recurse to innermost exception
	 *                      (for nested XRPC, only if throwXRPC=false)
	 * @note This method will block until the result becomes available.
	 * @note This method will throw if an exception occurred on the
	 *       server side while processing the call.
	 * @see wait(), timedWait(), hasValue(), hasException()
	 */
	ReturnValue get(bool throwXRPC = true, bool recursive = true)
	{
		if (throwXRPC)
			return this->mFuture.get();

		try {
			return this->mFuture.get();
		}
		catch (XRPC& ex) {
			if (ex.hasOrigException())
				ex.raiseOrigException(recursive);

			throw; // throw XRPC if it has no original exception stored
		}
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * This is a specialization for RPC calls with the return type of void, where
 * the get() method does not return a value.
 * @see RPCFuture, RPCFuture<R>
 */
template <>
class RPCFuture<void> : public RPCFutureCommon<void>
{
public:
	RPCFuture() {}

	/// internally used by RPCClient
	RPCFuture(boost::unique_future<void> other, AbstractRPCClient* client,
	          const std::string& callId) :
		RPCFutureCommon<void>(client, callId)
	{
		this->mFuture.swap(other);
	}

	/// move constructor
	RPCFuture(RPCFuture&& other) noexcept {
		this->swap(other);
	}

	/// move assignment operator
	RPCFuture& operator=(RPCFuture && other) noexcept {
		this->swap(other);
		return *this;
	}

	/**
	 * Waits until the call has finished and will raise an exception if the
	 * call has failed.
	 * @param[in] throwXRPC throw XRPC instead of original exception that occured in the
	 *                      called function (default = true, for backward compatibility)
	 * @param[in] recursive recurse to innermost exception
	 *                      (for nested XRPC, only if throwXRPC=false)
	 * @note This method will block until the call has finished.
	 * @note This method will throw if an exception occurred on the
	 *       server side while processing the call.
	 * @note Depending on configuration of boost future, it may be illegal to
	 *       call this more than once on the same object.
	 * @see wait(), timedWait(), hasValue(), hasException()
	 */
	void get(bool throwXRPC = true, bool recursive = true)
	{
		if (throwXRPC) {
			this->mFuture.get();
		} else {
			try {
				this->mFuture.get();
			}
			catch (XRPC& ex) {
				if (ex.hasOrigException())
					ex.raiseOrigException(recursive);

				throw; // throw XRPC if it has no original exception stored
			}
		}
	}

};

//////////////////////////////////////////////////////////////////////////////

/**
 * This is a specialization for JSON RPC calls.
 * @see RPCFuture, RPCFuture<R>
 */
template <>
class RPCFuture<JSONRPCResponse> : public RPCFutureCommon<JSONRPCResponse>
{
public:
	RPCFuture() {}

	/// internally used by RPCClient
	RPCFuture(boost::unique_future<JSONRPCResponse> other, AbstractRPCClient* client,
	          const std::string& callId) :
		RPCFutureCommon<JSONRPCResponse>(client, callId)
	{
		this->mFuture.swap(other);
	}

	/// move constructor
	RPCFuture(RPCFuture&& other) noexcept {
		this->swap(other);
	}

	/// move assignment operator
	RPCFuture& operator=(RPCFuture && other) noexcept {
		this->swap(other);
		return *this;
	}

	/**
	 * Waits until the call has finished and will raise an exception if the
	 * call has failed.
	 * @param[in] throwXRPC throw XRPC instead of original exception that occurred in the
	 *                      called function (default = true, for backward compatibility)
	 * @param[in] recursive recurse to innermost exception
	 *                      (for nested XRPC, only if throwXRPC=false)
	 * @note This method will block until the call has finished.
	 * @note This method will throw if an exception occurred on the
	 *       server side while processing the call.
	 * @see wait(), timedWait(), hasValue(), hasException()
	 */
	json::Value get(bool throwXRPC = true, bool recursive = true)
	{
		if (throwXRPC)
			return this->mFuture.get().getResult();

		try {
			return this->mFuture.get().getResult();
		}
		catch (XRPC& ex) {
			if (ex.hasOrigException())
				ex.raiseOrigException(recursive);

			throw; // throw XRPC if it has no original exception stored
		}
	}

	/**
	 * Waits until the call has finished and returns the complete json rpc answer
	 * containing call id, result or in case of an error the error message and code.
	 * @note This method will block until the call has finished.
	 * @see wait(), timedWait(), hasValue()
	 */
	json::Value getAnswer()
	{
		return this->mFuture.get().response;
	}

	bool hasException()
	{
		if (!this->mFuture.has_value())
			return false;
		try
		{
			get();
		}
		catch(...)
		{
			return true;
		}
		return false;
	}
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
