/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AbstractRPCHandler.h
 *    Abstract interface for RPCHandler(s).
 *
 * @author Erik Einhorn
 * @date   2010/11/23
 */

#ifndef _MIRA_ABSTRACTRPCHANDLER_H_
#define _MIRA_ABSTRACTRPCHANDLER_H_

#include <thread/Thread.h>

#include <rpc/AbstractDeferredInvoker.h>


namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup RPCModule
 * Abstract interface for derived RPCHandler(s).
 * The actual derived RPCHandler should manage an internal list of pending
 * requests. Additionally, the RPCHandler usually will have its own thread
 * for processing those pending requests.
 * The requests are given as AbstractDeferredInvokerPtr. They can be processed
 * by calling the AbstractDeferredInvoker::invoke() method.
 */
class AbstractRPCHandler
{
public:
	virtual ~AbstractRPCHandler() {}

public:

	/**
	 * Queues a new request that is given as AbstractDeferredInvokerPtr.
	 * The actual requests can be invoked by calling
	 */
	virtual void queueRequest(AbstractDeferredInvokerPtr invoker) = 0;

	/**
	 * Should return the thread id of the thread that processes the rpc
	 * requests. The RPCManager uses this thread id to determine, if a rpc
	 * call is requested within the same thread that is used to process it.
	 * In order to avoid deadlocks the rpc method is directly called in this
	 * case, instead of calling queueRequest().
	 */
	virtual boost::thread::id getThreadID() const = 0;

};

typedef boost::shared_ptr<AbstractRPCHandler> AbstractRPCHandlerPtr;


//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_RPCHANDLER_H_ */
