/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Typename.h
 *    Get compiler and platform independent typenames.
 *
 * @author Tim Langner
 * @date   2010/10/19
 */

#ifndef _MIRA_TYPENAME_H_
#define _MIRA_TYPENAME_H_

#include <typeinfo>
#include <string>
#include <sstream>

#include <platform/Platform.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

typedef std::string Typename;

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

namespace Private
{
Typename MIRA_BASE_EXPORT demangle(const std::type_info& t);

template<typename T>
inline Typename cvqualify(const Typename& tn) {
	bool needDelimiter = (tn.back() != '*') && (tn.back() != '>') && (tn.back() != ')');

	std::stringstream ss;
	ss << tn;

	if (std::is_const<T>()) {
		if (needDelimiter)
			ss << " ";
		ss << "const";
		needDelimiter = true;
	}
	if (std::is_volatile<T>()) {
		if (needDelimiter)
			ss << " ";
		ss << "volatile";
	}

	return ss.str();
}

}

///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * Returns a compiler and platform independent typename of T.
 * @return Typename of T
 */
template<typename T>
inline Typename typeName(bool cvqualify = true)
{
	if (cvqualify)
		return Private::cvqualify<T>(Private::demangle(typeid(T)));
	else
		return Private::demangle(typeid(T));
}

/**
 * Returns a compiler and platform independent typename of t (not necessarily type T).
 * * @return Typename of t
 */
template<typename T>
inline Typename typeName(T& t, bool cvqualify = true)
{
	if (cvqualify)
		return Private::cvqualify<T>(Private::demangle(typeid(t)));
	else
		return Private::demangle(typeid(t));
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
