/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Color.h
 *	This file contains color classes for the Img class. Every class derived
 *	from the ColorBase class has a method toRGB() for conversion.
 *
 * @author Erik Einhorn
 * @date   2010/10/21
 */

#ifndef _MIRA_COLOR_H_
#define _MIRA_COLOR_H_

#include <math.h>
#include <opencv2/core/core.hpp>

#include <platform/Platform.h>
#include <factory/Factory.h>
#include <serialization/PropertyHint.h>
#include <serialization/ReflectorInterface.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward decl for RGB
namespace Color {
class MIRA_BASE_EXPORT RGB;
}

///////////////////////////////////////////////////////////////////////////////

/** @brief base interface for all colors in different color spaces
 * The class provides a base interface for all color (space) classes.
 * Each derived class must implement the pure-virtual method toRGB().
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT ColorBase: public Object {
	MIRA_OBJECT(ColorBase)
public:
	virtual ~ColorBase() {
	}

public:
	/// converts the color of every color space to a RGB color.
	virtual Color::RGB toRGB() const = 0;

public:
	/**@brief converts color to cv::Scalar
	 * Converts color to cv::Scalar by first converting to RGB and then
	 * creating a 4 channel cv::Scalar(B,G,R,255);
	 */
	virtual operator cv::Scalar() const;
};

///////////////////////////////////////////////////////////////////////////////

namespace Color {

///////////////////////////////////////////////////////////////////////////////
/// The different color spaces

/**
 * Color in RGB color space
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT RGB: public ColorBase {
	MIRA_OBJECT(RGB)
public:

	/**
	 * Constructs RGB color
	 * R,G and B: range from 0.0 to 1.0
	 */
	RGB(float ir, float ig, float ib) :
			r(ir), g(ig), b(ib) {
	}

	/// Default constructor that creates a black color.
	RGB() :
			r(0), g(0), b(0) {
	}

	/// copy constructor
	RGB(const RGB& other) {
		r = other.r;
		g = other.g;
		b = other.b;
	}

	template<typename Reflector>
	void reflect(Reflector& reflector) {
		reflector.property("R", r, "The red component",
		                   PropertyHints::limits(0., 1.));
		reflector.property("G", g, "The green component",
		                   PropertyHints::limits(0., 1.));
		reflector.property("B", b, "The blue component",
		                   PropertyHints::limits(0., 1.));
	}

	/// no conversion is done here, because it is already RGB
	virtual Color::RGB toRGB() const {
		return *this;
	}

	/**Returns true, if all rgb components are within valid ranges, i.e.
	 * between 0.0 and 1.0.
	 */
	bool isInRange() const {
		return r >= 0.0f && r <= 1.0f && g >= 0.0f && g <= 1.0f && b >= 0.0f
				&& b <= 1.0f;
	}

public:
	float r, g, b;
};

/**
 * Color in RGBA color space
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT RGBA: public RGB {
	MIRA_OBJECT(RGBA)
public:

	/**
	 * Constructs RGBA color
	 * R,G,B and A: range from 0.0 to 1.0
	 */
	RGBA(float ir, float ig, float ib, float ia) :
			RGB(ir, ig, ib), a(ia) {
	}

	/// Default constructor that creates a black color.
	RGBA() :
			a(1.0f) {
	}

	/// copy constructor
	RGBA(const RGBA& other) :
			RGB(other.r, other.g, other.b), a(other.a) {
	}

	/** Constructs RGBA color from RGB color and alpha value
	 *  (that is 1.0f per default).
	 */
	RGBA(const RGB& other, float ia = 1.0f) :
			RGB(other.r, other.g, other.b), a(ia) {
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, RGB);
		r.property("A", a, "The alpha component",
		           PropertyHints::limits(0., 1.));
	}

	virtual operator cv::Scalar() const {
		return cv::Scalar(b * 255.0f, g * 255.0f, r * 255.0f, a * 255.0f);
	}

public:
	float a;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Color in HSV color space
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT HSV: public ColorBase {
	MIRA_OBJECT(HSV)
public:

	/**
	 * Constructs color in HSV color space.
	 * H,S and V: range from 0.0 to 1.0
	 */
	HSV(float hue, float saturation, float value) :
			h(hue), s(saturation), v(value) {
	}
	/**
	 * Default constructor
	 * initializes H,S, and V to 0.0
	 */
	HSV() :
			h(0.0f), s(0.0f), v(0.0f) {
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("H", h, "The hue component", PropertyHints::limits(0., 1.));
		r.property("S", s, "The saturation component",
		           PropertyHints::limits(0., 1.));
		r.property("V", v, "The value component",
		           PropertyHints::limits(0., 1.));
	}

	/**
	 * Converts the HSV color to RGB.
	 * If the HSV components are within their valid ranges, the resulting
	 * RGB color components will be within their valid ranges, too.
	 */
	virtual Color::RGB toRGB() const {
		float f = fmod(h * 6.0f, 6.0f);
		int hi = ((int) f);
		f = f - hi; // get the fraction

		float p = v * (1 - s);
		float q = v * (1 - s * f);
		float t = v * (1 - s * (1 - f));

		switch (hi) {
		case 0:
			return RGB(v, t, p);
		case 1:
			return RGB(q, v, p);
		case 2:
			return RGB(p, v, t);
		case 3:
			return RGB(p, q, v);
		case 4:
			return RGB(t, p, v);
		case 5:
			return RGB(v, p, q);
		}

		return RGB(0, 0, 0);
	}

public:
	float h, s, v;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Color in CIE 1931 XYZ color space
 * The CIE XYZ color space is the master for the derived CIE Lab color space,
 * which is perceptually uniform and hence will be used more often for
 * visualization purposes. @see mira::Color::Lab
 * @note Since the CIE XYZ color space is larger than the RGB space,
 * some XYZ colors cannot be represented by a corresponding RGB color.
 * Hence, a conversation to a RGB color may result in a RGB color that is
 * out of bounds.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT XYZ: public ColorBase {
	MIRA_OBJECT(XYZ)
public:

	/**
	 * Constructs color in CIE 1931 XYZ color space
	 * X: from 0 to  95.047
	 * Y: from 0 to 100.000
	 * Z: from 0 to 108.883
	 */
	XYZ(float ix, float iy, float iz) :
			x(ix), y(iy), z(iz) {
	}
	/**
	 * Default constructor
	 * Initializes X,Y, and Z with 0.0
	 */
	XYZ() :
			x(0.0f), y(0.0f), z(0.0f) {
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("X", x, "The x component",
		           PropertyHints::limits(0., 98.047));
		r.property("Y", y, "The y component",
		           PropertyHints::limits(0., 100.000));
		r.property("Z", z, "The z component",
		           PropertyHints::limits(0., 108.883));
	}

	/**
	 * Converts the CIE XYZ color to RGB.
	 * @note Since the CIE XYZ color space is larger than the RGB space,
	 * the components of the resulting RGB color may be out of their valid
	 * ranges, i.e. some components may be negative or larger than 1.0. This
	 * happens, if a XYZ color cannot be represented by a RGB color. The
	 * range of the returned RGB color can be checked using the isInRange()
	 * method.
	 */
	virtual Color::RGB toRGB() const {
		// formulas for conversation were taken from
		// http://www.easyrgb.com/index.php
		// Observer = 2 deg, Illuminant = D65
		float X = x / 100.0f;
		float Y = y / 100.0f;
		float Z = z / 100.0f;

		float R = X * 3.2406f + Y * -1.5372f + Z * -0.4986f;
		float G = X * -0.9689f + Y * 1.8758f + Z * 0.0415f;
		float B = X * 0.0557f + Y * -0.2040f + Z * 1.0570f;

		if (R > 0.0031308f)
			R = 1.055f * pow(R, 1.0f / 2.4f) - 0.055f;
		else
			R = 12.92f * R;

		if (G > 0.0031308f)
			G = 1.055f * pow(G, 1.0f / 2.4f) - 0.055f;
		else
			G = 12.92f * G;

		if (B > 0.0031308f)
			B = 1.055f * pow(B, 1.0f / 2.4f) - 0.055f;
		else
			B = 12.92f * B;

		return RGB(R, G, B);
	}

public:
	float x, y, z;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Color in CIE Lab color space
 * The CIE Lab color space is derived from the XYZ color space and is designed
 * to approximate human vision. It aspires to perceptual uniformity, and its
 * L component closely matches human perception of lightness. It can thus be
 * used to make accurate color balance corrections by modifying output curves
 * in the a and b components, or to adjust the lightness contrast using the
 * L component. The Lab color space is perceptually uniform, i.e. a change of
 * the same amount in a color value should produce a change of about the same
 * visual importance.
 *
 * For more details see http://en.wikipedia.org/wiki/Lab_color_space.
 *
 * @note Since the CIE Lab color space (gamut) is larger than the RGB space,
 * some CIE Lab colors cannot be represented by a corresponding RGB color.
 * Hence, a conversation to a RGB color may result in a RGB color that is
 * out of bounds.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT Lab: public ColorBase {
	MIRA_OBJECT(Lab)
public:
	/**
	 * Constructs color in CIE Lab color space.
	 * L: from 0 to 100
	 * a: from -150 to 100
	 * b: from -100 to 150
	 */
	Lab(float iL, float ia, float ib) :
			L(iL), a(ia), b(ib) {
	}
	/**
	 * Default constructor
	 * Initializes L,a, and b with 0.0
	 */
	Lab() :
			L(0.0f), a(0.0f), b(0.0f) {
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.property("L", L, "The L component",
		           PropertyHints::limits(0., 100.));
		r.property("A", a, "The a component",
		           PropertyHints::limits(-150., 100.));
		r.property("B", b, "The b component",
		           PropertyHints::limits(-100., 150.));
	}

	/**
	 * Converts CIE Lab to CIE XYZ.
	 * @note The CIE Lab color space may differ from the XYZ color space, hence
	 * some CIE Lab colors can result in XYZ colors that are out of bounds.
	 */
	XYZ toXYZ() const {

		// formulas for conversation were taken from
		// http://www.easyrgb.com/index.php
		float Y = (L + 16.0f) / 116.0f;
		float X = a / 500.0f + Y;
		float Z = Y - b / 200.0f;

		float Y3 = Y * Y * Y;
		float X3 = X * X * X;
		float Z3 = Z * Z * Z;

		if (Y3 > 0.008856f)
			Y = Y3;
		else
			Y = (Y - 16.0f / 116.0f) / 7.787f;
		if (X3 > 0.008856f)
			X = X3;
		else
			X = (X - 16.0f / 116.0f) / 7.787f;
		if (Z3 > 0.008856f)
			Z = Z3;
		else
			Z = (Z - 16.0f / 116.0f) / 7.787f;

		// Reference color space:
		// Observer= 2 deg, Illuminant= D65

		float refX = 95.047f;
		float refY = 100.000f;
		float refZ = 108.883f;

		X *= refX;
		Y *= refY;
		Z *= refZ;

		return XYZ(X, Y, Z);
	}

	/**
	 * Converts CIE Lab to RGB
	 * @note Since the CIE Lab color space is larger than the RGB space,
	 * the components of the resulting RGB color may be out of their valid
	 * ranges, i.e. some components may be negative or larger than 1.0. This
	 * happens, if a CIE Lab color cannot be represented by a RGB color. The
	 * range of the returned RGB color can be checked using the isInRange()
	 * method.
	 */
	virtual Color::RGB toRGB() const {
		return toXYZ().toRGB();
	}

public:
	float L, a, b;
};

///////////////////////////////////////////////////////////////////////////////

//
// The following code (YUV color class) is disabled, since the conversion
// to/from is not clear at the moment. There are different formulas avialable:
// http://en.wikipedia.org/wiki/YUV and http://www.fourcc.org/fccyvrgb.php
//
// We decided to remove the YUV color class until somebody has some time of
// investigate this into more detail.
//
///**
// * Color in YUV color space
// *
// * conversion formulas to/from RGB taken from http://en.wikipedia.org/wiki/YUV
// *
// * @ingroup ImageModule
// */
//class MIRA_BASE_EXPORT YUV: public ColorBase {
//	MIRA_OBJECT(YUV)
//public:
//
//	/**
//	 * Constructs color in YUV color space.
//	 * Y:    0.0 to 1.0
//	 * U: -0.436 to 0.436
//	 * V: -0.615 to 0.615
//	 */
//	YUV(float iy, float iu, float iv) :
//			y(iy), u(iu), v(iv) {
//	}
//
//	/**
//	 * Default constructor
//	 * initializes Y,U, and V to 0.0
//	 * equivalent to RGB = (0,0,0)
//	 */
//	YUV() :
//			y(0.0f), u(0.0f), v(0.0f) {
//	}
//
//	YUV(const RGB& rgb) :
//		y( 0.299f   * rgb.r +0.587f   * rgb.g  +0.114f   * rgb.b),
//		u(-0.14713f * rgb.r -0.28886f * rgb.g  +0.436f   * rgb.b),
//		v( 0.615f   * rgb.r -0.51499f * rgb.g  -0.10001f * rgb.b) {
//	}
//
//	template<typename Reflector>
//	void reflect(Reflector& r) {
//		r.property("Y", y, "The luminance component", PropertyHints::limits(0., 1.));
//		r.property("U", u, "The chrominance component 1 (blue-luminance difference)",
//		           PropertyHints::limits(-0.436f, 0.436f));
//		r.property("V", v, "The chrominance component 2 (red-luminance difference)",
//		           PropertyHints::limits(-0.615f, 0.615f));
//	}
//
//	/**
//	 * Converts the YUV color to RGB.
//	 */
//	virtual Color::RGB toRGB() const {
//
//		float r =  1.0f * y +              + 1.13983f * v;
//		float g =  1.0f * y - 0.39465f * u - 0.58060f * v;
//		float b =  1.0f * y + 2.03211f * u;
//
//		return RGB(r, g, b);
//	}
//
//public:
//	float y, u, v;
//};

///////////////////////////////////////////////////////////////////////////////

// some important color predefs
const RGB Red(1.0f, 0.0f, 0.0f);
const RGB Green(0.0f, 1.0f, 0.0f);
const RGB Blue(0.0f, 0.0f, 1.0f);
const RGB Cyan(0.0f, 1.0f, 1.0f);
const RGB Magenta(1.0f, 0.0f, 1.0f);
const RGB Yellow(1.0f, 1.0f, 0.0f);
const RGB White(1.0f, 1.0f, 1.0f);
const RGB Black(0.0f, 0.0f, 0.0f);

///////////////////////////////////////////////////////////////////////////////

}// namespace Color

/**
 *  cast operator to cv::Scalar
 *  converts color to OpenCV BGRA format
 *  by scaling range to 0..255 and setting alpha channel to 255
 */
inline ColorBase::operator cv::Scalar() const {
	Color::RGB c = toRGB();
	return cv::Scalar(c.b * 255.0f, c.g * 255.0f, c.r * 255.0f, 255.0f);
}

///////////////////////////////////////////////////////////////////////////////

}// namespace mira

#endif
