/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Polygon.h
 *    Simple Wrapper for Boost::geometry polygon.
 *
 * @author Jens Kessler
 * @date   2010/09/30
 */

#ifndef _MIRA_POLYGON_H_
#define _MIRA_POLYGON_H_

#include <serialization/adapters/std/vector>
#ifndef Q_MOC_RUN
#include <boost/geometry/geometries/register/ring.hpp>
#endif
#include <geometry/Point.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * The polygon is essentially a sequence of points with an edge also connecting
 * the first and last point. Since Eigen could not deal with polygons, we
 * switch to support boost geometry more.
 *
 * An example with 3D points
 *\code
 * |P1.x|   |P2.x|   |P3.x|   |P4.x|
 * |P1.y|...|P2.y|...|P3.y|...|P4.y|...
 * |P1.z|   |P2.z|   |P3.z|   |P4.z|
 *\endcode
 *
 * Boost::geometry polygons support so called "inner" and "outer" rings. The
 * outer ring defines the outer bound, where the polygon begins. The inner
 * rings define holes inside the polygon. There is only one outer ring but
 * multiple inner rings are possible.
 *
 * However in MIRA we only use polygons that have no inner rings. There we use
 * boost::geometry::model::ring instead of boost::geometry::polygon!
 *
 * NOTE: Intersections of polygons and rasterization can only be established
 * on 2D polygons. More dimensions for intersection are not supported in
 * boost::geometry and a 3D rasterization needs a triangulation before.
 */

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup GeometryModule
 *
 * Function to create a triangle from three given points.
 * The function returns a boost::geometry::ring.
 * @param[in] p1 : first point of the triangle
 * @param[in] p2 : second point of the triangle
 * @param[in] p3 : third point of the triangle
 */
template<typename PointType>
inline boost::geometry::model::ring<PointType> createTriangle(const PointType& p1,
                                                              const PointType& p2,
                                                              const PointType& p3)
{
	boost::geometry::model::ring<PointType> triangle;
	triangle.push_back(p1);
	triangle.push_back(p2);
	triangle.push_back(p3);
	triangle.push_back(p1);
	return triangle;
}

/**
 * Converts a line into a polygon.
 */
template<typename LineType>
inline boost::geometry::model::ring<typename LineType::PointType> polygonFromLine(const LineType& line)
{
	typedef typename LineType::PointType PointType;
	boost::geometry::model::ring<PointType> poly;
	poly.push_back(line.first);
	poly.push_back(line.second);
	return poly;
}

///////////////////////////////////////////////////////////////////////////////
// Some typedefs for 2D Polygons

/// A 2D polygon with integer precision.
typedef boost::geometry::model::ring<Point2i> Polygon2i;

/// A 2D polygon with 32 bit floating precision.
typedef boost::geometry::model::ring<Point2f> Polygon2f;

/// A 2D polygon with 64 bit floating precision.
typedef boost::geometry::model::ring<Point2d> Polygon2d;

///////////////////////////////////////////////////////////////////////////////
// Some typedefs for 3D Polygons

/// A 3D polygon with integer precision.
typedef boost::geometry::model::ring<Point3i> Polygon3i;

/// A 3D polygon with 32 bit floating precision.
typedef boost::geometry::model::ring<Point3f> Polygon3f;

/// A 3D polygon with 64 bit floating precision.
typedef boost::geometry::model::ring<Point3d> Polygon3d;

//////////////////////////////////////////////////////////////////////////////

/// non-intrusive reflect for boost::geometry::ring
template<typename Reflector, typename PointType>
void reflect(Reflector& r, boost::geometry::model::ring<PointType>& p)
{
	r.template reflectBase<std::vector<PointType>>(p);
}

template<typename PointType>
class IsCollection<boost::geometry::model::ring<PointType>> : public std::true_type {};

//////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
