/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LogFileSink.h
 *    Logging sink for file output with support for rolling and daily log files.
 *
 * @author Christian Martin
 * @date   2012/07/23
 */

#ifndef _MIRA_LOGFILESINK_H_
#define _MIRA_LOGFILESINK_H_

#include <fstream>

#include <error/LoggingCore.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A logging sink for file, which supports log file rotation and daily log
 * files.
 *
 * Based on the settings @c mFileRotationDepth, @c mDailyLogFiles and
 * @c mDailyLogFileStart, this log sink will create log files according the
 * following scheme:
 * - @c mFileRotationDepth = 0:
 *   - @c mDailyLogFiles = @c false:
 *      - Log file naming scheme:
 *        - @c logFilePrefix.log
 *      - Overwrites the log file at each start.
 *   - @c mDailyLogFiles = @c true:
 *      - Log file naming scheme:
 *        - @c logFilePrefix_YYYY-MM-DD.log
 *      - The log file will contain all messages starting at the specified time
 *        (@c mDailyLogFileStart) of the day. Log messages before this time will
 *        be written into the file of the day before.
 *      - If the log sink is started multiple times (e.g. multiple starts on one
 *        day), the log messages will be appended to the already existing log
 *        file.
 *
 * - @c mFileRotationDepth > 0:
 *   - @c mDailyLogFiles = @c false:
 *      - Log file naming scheme:
 *        - @c logFilePrefix.log (current start)
 *        - @c logFilePrefix.log.1 (log of one start ago)
 *        - @c logFilePrefix.log.2 (log of two starts ago)
 *      - Creates a new log file for each start, until @c mFileRotationDepth is
 *        reached. Older log files will be deleted.
 *   - @c mDailyLogFiles = @c true:
 *      - Log file naming scheme:
 *        - @c logFilePrefix_YYYY-MM-DD.log      (current day)
 *        - @c logFilePrefix_YYYY-MM-(DD-1).log  (one day ago)
 *        - @c logFilePrefix_YYYY-MM-(DD-2).log  (two days ago)
 *      - This mode creates a new log file starting at the specified time
 *        (@c mDailyLogFileStart) of the day. Log messages before this time
 *        will be written into the file of the day before.
 *      - The log file which is @c mFileRotationDepth+1 days old,
 *        will be deleted.
 *      - If the log sink is started multiple times (e.g. multiple starts on one
 *        day), the log messages will be appended to the already existing log
 *        file.
 *
 * - @c mFileRotationDepth < 0:
 *    - Same behaviour as @c mFileRotationDepth > 0, but unlimited number of
 *      files. Use carefully!
 *
 * By default, log file rotation and daily log files are disabled. The default
 * daily log file start is midnight.
 *
 * @note
 *   Using the same logFilePrefix with different settings (especially different
 *   values of @c mFileRotationDepth) might cause an unexpected behaviour,
 *   since for performance reasons only the file @c mFileRotationDepth+1 will
 *   be checked and deleted. Older files will not be touched.
 *
 * @ingroup LoggingModule
 */
class MIRA_BASE_EXPORT LogFileSink : public LogSink
{
public:
	/// Default constructor
	LogFileSink(const std::string& logFilePrefix,
	            int logFileRotationDepth = 0,
	            bool useDailyLogFiles = false,
	            Duration dailyLogFileStart = Duration(0, 0, 0));

	/// The copy constructor
	LogFileSink(const LogFileSink& other);

	/// Implementation of LogSink
	virtual void consume(const LogRecord& record);

private:
	void prepareFileStream();

	boost::gregorian::date getDateForDailyLogFile();

	std::string makeDateStr(boost::gregorian::date& date);

private:
	/// The current output stream.
	std::ofstream mStream;

	/// The log file name prefix.
	std::string mLogFilePrefix;

	/// The log file rotation depth (-1 = endless, 0 = no rotation).
	int mFileRotationDepth;

	/// Enable/disable daily log files.
	bool mDailyLogFiles;

	/// The start time of a daily log file.
	Duration mDailyLogFileStart;

private:
	bool mNeedRotation;
	boost::gregorian::date mCurrentDailyLogFileDate;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
