/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SQLiteDB.h
 *    A wrapper class for the SQLite SQL database.
 *
 * @author Tim Langner
 * @date   2010/10/25
 */

#ifndef _MIRA_SQLITEDB_H_
#define _MIRA_SQLITEDB_H_

#include <error/Exceptions.h>
#include <utils/Path.h>

#include <database/SQLiteQuery.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup DatabaseModule
 * An exception that occurs whenever accessing the SQLite db results in an error
 */
MIRA_DEFINE_SERIALIZABLE_EXCEPTION(XSQLite, XIO)

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup DatabaseModule
 * 
 * Wrapper class for the SQLite C API.
 * Used for constructing and accessing file based SQLite databases.
 */
class MIRA_BASE_EXPORT SQLiteDB : boost::noncopyable
{
public:
	/** @name Constructors and destructor */
	//@{

	/**
	 * Constructs a SQLite database by loading a db file.
	 * The database is created if not exist.
	 * @param[in] db The path to the database file
	 */
	SQLiteDB(const Path& db);

	/// Destructor
	~SQLiteDB();

	//@}

public:
	/** @name Database operations */
	//@{

	/**
	 * Asks if a given table exists in the database.
	 * @param[in] table The table name.
	 * @return true if table exists, false otherwise
	 */
	bool tableExists(const std::string& table);

	/**
	 * Executes a database query and returns the query object.
	 * @param[in] sql The query in SQL syntax.
	 * @return The query object.
	 */
	SQLiteQuery query(const std::string& sql);

	/**
	 * Executes a statement on the database
	 * @param[in] sql The statement
	 * @return Counter of how many rows have been affected.
	 */
	int exec(const std::string& sql);

	//@}

private:
	sqlite3_stmt* prepare(const std::string& sql);

private:
	sqlite3* mDB;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
