/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MulticastSender.h
 *    Class for sending multicast messages.
 *
 * @author Tim Langner
 * @date   2010/07/14
 */

#ifndef _MIRA_MULTICASTSENDER_H_
#define _MIRA_MULTICASTSENDER_H_

#ifndef Q_MOC_RUN
#include <boost/thread.hpp>
#include <boost/utility.hpp>
#endif

#include <communication/IOService.h>
#include <error/LoggingCore.h>
#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * This class acts as a sender for multicast messages. It can be bound to a
 * multicast address (224.0.0.0-239.255.255.255) and a port. It will then send
 * a specified message in a defined interval. This class can be used to publish
 * a service in a network or to send cyclic alive messages.
 *
 * @ingroup CommunicationModule
 */
class MIRA_BASE_EXPORT MulticastSender
{
public:
	/**
	 * Constructs a multicast sender at the given ip and port and sends the
	 * message each interval second.
	 * @param[in] address The multicast address
	 * @param[in] port The port
	 */
	MulticastSender(const std::string& address, uint16 port);

	/**
	 * Constructs a multicast sender at the given ip and port.
	 * @param[in] address The multicast address
	 * @param[in] port The port
	 * @param service A service that is shared with other objects
	 */
	MulticastSender(const std::string& address, uint16 port,
	                boost::asio::io_service& service);

	/**
	 * Starts sending a message in the given interval
	 * @param[in] interval Interval to send data
	 * @param[in] message The message to send
	 */
	void start(const Duration& interval, const std::string& message);

	/**
	 * Sends a single message
	 */
	void send(const std::string& message);

private:
	void handleTimeout(const boost::system::error_code& error);

private:
	IOService mService;
	boost::asio::ip::udp::endpoint mEndpoint;
	boost::asio::ip::udp::socket  mSocket;
	boost::asio::deadline_timer mTimer;
	boost::thread mThread;
	Duration mInterval;
	std::string mMessage;
};

/// A shared pointer wrapper for class MulticastSender.
typedef boost::shared_ptr<MulticastSender> MulticastSenderPtr;

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
