/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PromptProviderGui.C
 *    Description.
 *
 * @author Ronny Stricker
 * @date   2011/08/31
 */

#include "gui/PromptProviderGui.h"

#include <QVBoxLayout>
#include <QLineEdit>
#include <QMessageBox>
#include <QPushButton>
#include <QLabel>

#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

PasswordDialog::PasswordDialog( Credential const& credential, QWidget * parent , Qt::WindowFlags f ) :
	QDialog( parent, f ), mCredential( credential )
{
	QVBoxLayout* tLayout = new QVBoxLayout();
	setLayout( tLayout );
	QLabel* tLabel = new QLabel( QString("Login for ") + QString::fromStdString( credential.realm ) );
	tLayout->addWidget( tLabel );

	mUserName = new QLineEdit( "username", this );
	tLayout->addWidget( mUserName );

	mPassword = new QLineEdit( "password", this );
	mPassword->setEchoMode( QLineEdit::Password );
	tLayout->addWidget( mPassword );

	QHBoxLayout* buttonLayout = new QHBoxLayout();

	QPushButton* tButton = new QPushButton("login",this);
	buttonLayout->addWidget( tButton );
	connect( tButton, SIGNAL(clicked()), this, SLOT(accept()) );

	tButton = new QPushButton("cancel",this);
	buttonLayout->addWidget( tButton );
	connect( tButton, SIGNAL(clicked()), this, SLOT(reject()) );

	tLayout->addLayout( buttonLayout );

	mUserName->setFocus( Qt::ActiveWindowFocusReason );
}

void PasswordDialog::accept( )
{
	mCredential.user = mUserName->text().toStdString();
	mCredential.password = mPassword->text().toStdString();

	QDialog::accept();
}

///////////////////////////////////////////////////////////////////////////////

bool PromptProviderGui::getLogin( Credential& credential )
{
	PasswordDialog tDialog( credential, mParent );
	tDialog.mCredential =  credential;

	if ( tDialog.exec() ) {
		credential = tDialog.mCredential;
		return true;
	}
	return false;
}

PromptProviderGui::AcceptValue PromptProviderGui::acceptServerCertificate( ServerCertificate const& certificate )
{
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
	QMessageBox msgBox(QMessageBox::Warning, "Untrusted server certificate",
		"The server certificate of " + QString::fromLatin1(certificate.issuer_dname) + " is not secure",
		QMessageBox::NoButton, mParent );
	msgBox.setDetailedText( "Hostname: " + QString::fromLatin1(certificate.hostname) + "\nFingerprint: " +  QString::fromLatin1(certificate.fingerprint) );
#else
	QMessageBox msgBox(QMessageBox::Warning, "Untrusted server certificate",
		"The server certificate of " + QString::fromAscii(certificate.issuer_dname) + " is not secure",
		QMessageBox::NoButton, mParent );
	msgBox.setDetailedText( "Hostname: " + QString::fromAscii(certificate.hostname) + "\nFingerprint: " +  QString::fromAscii(certificate.fingerprint) );
#endif

	msgBox.addButton(QObject::tr("Reject"), QMessageBox::ActionRole);
	QPushButton *acceptOnceButton =
			msgBox.addButton(QObject::tr("Accept"), QMessageBox::ActionRole);
	QPushButton *acceptAlwaysButton =
			msgBox.addButton(QObject::tr("Always accept"), QMessageBox::ActionRole);

	msgBox.exec();

	if (msgBox.clickedButton() == acceptOnceButton) {
		return Temporary;
	} else if (msgBox.clickedButton() == acceptAlwaysButton) {
		return Always;
	}
	return Reject;
}

void PromptProviderGui::showErrorMessage( std::string const& message )
{
	QMessageBox::critical(mParent, "Error",
	                      QString::fromUtf8( message.c_str() ),
	                      QMessageBox::Ok);
}

bool PromptProviderGui::showYesNoErrorMessage( std::string const& message )
{
	int button = QMessageBox::critical(mParent, "Error",
	                                   QString::fromUtf8( message.c_str() ),
	                                   QMessageBox::Yes | QMessageBox::No);
	return(button == QMessageBox::Yes);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

