/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**se
 * @file Database.C
 *    Description.
 *
 * @author Erik Einhorn, Ronny Stricker
 * @date   2011/02/24
 */
#include "core/Database.h"
#include "core/PackageGroup.h"
#include "core/Package.h"
#include "core/Repository.h"
#include "core/Tools.h"

#include <serialization/Serialization.h>

#include <boost/algorithm/string/case_conv.hpp>

#include <fstream>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Database::Database() : mPromptProvider(NULL)
{
	auto tRepoTypes = Repository::CLASS().getDerivedClasses();
	foreach( auto& tRepoType, tRepoTypes ) {
		if ( !tRepoType.second.isAbstract() ) {
			RepositoryPtr repo = RepositoryPtr( tRepoType.second.newInstance<Repository>() );
			genericRepos[repo->name] = repo;
		}
	}
}

Database::~Database()
{
	clearRepositories();
}

void Database::load(const XMLDom& content)
{
	// read the database content from the XML node
	XMLDeserializer ds(content);
	ds.deserialize("database", *this);
}

void removeLocalPath( PackageGroup* group, Database const* database )
{
	Package* tPackage = dynamic_cast<Package*>( group );
	if ( tPackage ) {
		if ( !tPackage->mLocalPath.string().empty() &&
				!database->isInstalled( group, false ) ) {
			tPackage->mLocalPath = "";
		}
	}
	foreach( PackageGroup* tChild, group->mSubPackages ) {
		removeLocalPath( tChild, database );
	}
}

void Database::store(XMLDom& content) const
{
	// throw away the path of packages which are not installed
	removeLocalPath( const_cast<PackageGroup*>( &rootPackage ), this );

	// delete a possible existing old "database" node from the XML node.
	XMLDom::iterator iter = content.root().find("database");
	if (iter != content.root().end())
		iter.remove();

	// store the current database configuration into the XML node.
	XMLSerializer s(content);
	s.serialize("database", *this);
}

void recursiveAddExport( Database const& db, PackageGroup& newRoot,
		PackageGroup const* group,
		map<Path,string>& pathMap )
{
	if (!group)
		return;

	Package const* package = dynamic_cast<Package const*>( group );
	if ( package ) {
		if ( db.isInstalled( package, false ) &&
				db.getRepoFromUrl( package->mCurrentRepo ) != NULL ) {
			Package* newPackage = package->createCopy();
			// try to replace the mira path part of the path with the
			// associated path id
			Path miraPath = PathProvider::getAssociatedMiraPath( package->mLocalPath );
			if ( pathMap.find( miraPath ) != pathMap.end() )
				newPackage->mLocalPath = pathMap[ miraPath ] +
				package->mLocalPath.string().substr( miraPath.string().length() );
			else
				newPackage->mLocalPath = "";
			newPackage->mParentPackage = NULL;
			newPackage->mDependencies.clear();
			newPackage->mTags.clear();

			newRoot.mSubPackages.push_back( newPackage );
		}
	}
	else {
		foreach( PackageGroup const* subGroup, group->mSubPackages )
				recursiveAddExport( db, newRoot, subGroup, pathMap );
	}
}

void Database::exportInstalledPackages( Database& ioDB, map<Path,string>& oPathMap ) const
{
	// create a list of mira path entries to replace them with a unique path id
	// later
	int pathId = 0;
	foreach( Path const& tPath, PathProvider::miraPaths() ) {
		oPathMap[tPath] = toString( pathId )+"]";
		++pathId;
	}

	foreach( PackageGroup const* group, rootPackage.mSubPackages ) {
		recursiveAddExport( *this, ioDB.rootPackage, group, oPathMap );
	}
	ioDB.repos = repos;
}

void Database::importRepositories(const Path& path)
{
	Database tmpDB;

	XMLDom db;
	db.loadFromFile(path);
	tmpDB.load( db );

	// check if the repositories are already known
	foreach( auto const& tRepo, tmpDB.repos ) {
		if ( getRepoFromUrl( tRepo.second->url ) == NULL ||
				getRepoFromUrl( tRepo.second->url )->url != tRepo.second->url ) {
			// this is save due to usage of shared pointers (and tmpDB will not
			// be modified)
			addRepository( tRepo.second );
		}
	}
}

void Database::clearPackages()
{
	foreach( PackageGroup* group, rootPackage.mSubPackages ) {
		delete group;
	}
	rootPackage.mSubPackages.clear();
}

void Database::cleanUp( PackageGroup* package )
{
	if ( package==NULL )
		package = &rootPackage;
	// check if the package is not installed / will not be installed and is
	// not scheduled for deinstallation
	if ( !isInstalled(package,false) && !willChange(package) ) {
		Package* tPackage = dynamic_cast<Package*>( package );
		if ( tPackage ) {
			if ( tPackage->mLocalPath.string().length() )
				tPackage->mLocalPath = "";
		}
	}
	foreach( PackageGroup* child, package->mSubPackages ) {
		cleanUp( child );
	}
}

void Database::resetPackageFlags()
{
	mInstalledPackages.clear();
	mActionPlan.clear();
}

Package* getStdSource( PackageGroup const& e )
{
	vector<Package*> tPrimaryPackages = e.getPrimarySubPackages();

	if ( tPrimaryPackages.size() == 0 ) {
		MIRA_THROW( XLogical, "Missing primary package source for package "
				+ e.mName + "!" );
	}

	// assume that the packages are sorted according to version and type
	return tPrimaryPackages.back();
}

template<class InputIterator, class T>
InputIterator findName ( InputIterator first, InputIterator last, const T& value )
{
	for ( ;first!=last; ++first)
		if ( (*first)->mName==value ) break;
	return first;
}

Dependency* Database::getInstallDependencies(Package const& s, bool fullDependencies) const
{
	PackageGroup* tGroup = s.createDependency();
	Dependency* tReturn = dynamic_cast<Dependency*>( tGroup );
	assert( tReturn );
	gatherDependencies( tReturn, fullDependencies );
	return tReturn;
}

void Database::clearRepositories()
{
	repos.clear();
}

void Database::addRepository( RepositoryPtr repo )
{
	assert( repo );
	if ( repos.find( repo->name ) != repos.end() ) {
		MIRA_THROW(XLogical, "Repository "+repo->name+" does already exist!");
	}
	repos[repo->name] = repo;
	repo->prompt = mPromptProvider;
}

void Database::addPackage( PackageGroup* package )
{
	rootPackage.add( package );
}

void Database::removePackage( PackageGroup* package )
{
	rootPackage.removeChild( package );
}

bool Database::isInstalled( PackageGroup const* const package, bool regardCheckoutPlan ) const
{
	Package const* tPackage = dynamic_cast<Package const*>( package );
	if ( tPackage ) {
		// we are searching for a package.
		// check if this package is installed or (if requested) should be
		// installed
		if ( regardCheckoutPlan ) {
			if ( getFlag( tPackage ) & INSTALL )
				return true;
			if ( getFlag( tPackage ) & UNINSTALL )
				return false;
		}
		Package* tNCpackage = const_cast<Package*>( tPackage );
		return ( mInstalledPackages.find( tNCpackage ) != mInstalledPackages.end() );
	}
	else {
		// we are searching for a goup
		// check if one of the groups subpackages is installed
		foreach( PackageGroup* sub, package->mSubPackages ) {
			if ( isInstalled( sub, regardCheckoutPlan ) )
				return true;
		}
	}
	return false;
}

bool Database::isInstalled( std::string const& name, bool regardCheckoutPlan ) const
{
	vector<Package*> packages = rootPackage.findPackages( name, true );
	foreach( Package* package, packages ) {
		if ( package->isPrimary() && isInstalled( package, regardCheckoutPlan ) )
			return true;
	}
	return false;
}

bool Database::willChange( PackageGroup const* package ) const
{
	Package const* tPackage = dynamic_cast<Package const*>( package );
	if ( tPackage ) {
		if ( getFlag( tPackage ) & ( INSTALL | UNINSTALL ) )
			return true;
	}
	bool installed = false;
	foreach( PackageGroup* sub, package->mSubPackages ) {
		if ( willChange( sub ) )
			return true;
	}
	return false;
}
	
bool Database::willChange( std::string const& name ) const
{
	vector<Package*> packages = rootPackage.findPackages( name, true );
	foreach( Package* package, packages ) {
		if ( package->isPrimary() && willChange( package ) )
			return true;
	}
	return false;
}

Database::Action Database::getFlag( PackageGroup const* const package ) const
{
	Package const* tPackage = dynamic_cast<Package const*>( package );
	if ( tPackage ) {
		Package* tNCpackage = const_cast<Package*>( tPackage );
		Database::ActionPlan::const_iterator it = mActionPlan.find( tNCpackage );
		return ( it != mActionPlan.end() ? it->second : NONE );
	}
	return NONE;
}

Database::Action Database::getFlag( std::string const& name ) const
{
	vector<Package*> packages = rootPackage.findPackages( name, true );
	foreach( Package* package, packages ) {
		if ( package->isPrimary() && hasFlag( package ) )
			return getFlag( package );
	}
	return NONE;
}

bool Database::hasFlag( PackageGroup const* group ) const
{
	Package const* package = dynamic_cast<Package const*>( group );
	if ( !package )
		return false;
	Package* tNCpackage = const_cast<Package*>( package );
	return mActionPlan.find( tNCpackage ) != mActionPlan.end();
}

PackageGroup* Database::getIdentical( PackageGroup const* other, Package::Type ignoredFlags ) const
{
	return rootPackage.getIdentical( other, ignoredFlags );
}

void Database::getAllIdentical( PackageGroup const* other,
		std::vector<PackageGroup*>& identical, Package::Type ignoredFlags ) const
{
	rootPackage.getAllIdentical( other, identical, ignoredFlags );
}

void Database::setFlag( Package* package, Action flag )
{
	if ( flag == NONE ) {
		mActionPlan.erase( package );
	}
	else {
		mActionPlan[ package ] = flag;
		if ( flag == INSTALL ) {
			// select default source if none is selected up to now
			if ( package->mCurrentRepo.empty() )
				package->selectStdRepo( this );
		}
	}
	// emit flag changed signal(s) for item and all of its parents (except of root)
	PackageGroup* group = package;
	while ( group && group->mParentPackage ) {
		emit flagChanged( group );
		group = group->mParentPackage;
	}
}

void collectTags( PackageGroup* group, map<string, set<string> >& tags )
{
	// add all tags recursively
	// the second entry in the returned map collects the corresponding package name
	// doing so, tags of one package group are not counted twice (and that is nice)
	Package* package = dynamic_cast<Package*>( group );
	if ( package ) {
		Package* source = dynamic_cast<Package*>( package );
		foreach( std::string const& tag, package->mTags ) {
			tags[ tag ].insert( package->mName );
		}
	}
	foreach( PackageGroup* child, group->mSubPackages ) {
		collectTags( child, tags );
	}
}

void Database::buildTagIndex()
{
	tags.clear();
	map<string,set<string> > tmpTags;
	foreach( PackageGroup* child, rootPackage.mSubPackages ) {
		collectTags( child, tmpTags );
	}
	typedef pair<string, set<string> > mapType;
	foreach( mapType const& tag, tmpTags ) {
		tags[ tag.first ] = tag.second.size();
	}
}

map<string,Package*> Database::alternativeVersions( Package const* package ) const
{
	map<string,Package*> tReturn;
	if ( package ) {
		PackageGroup* group = rootPackage.findGroup( package->mName );
		// if we cannot find a group then we don't have to search for alternative versions...
		if ( group ) {
			foreach( PackageGroup* subGroup, group->mSubPackages ) {
				Package* subPackage = dynamic_cast<Package*>( subGroup );
				assert( subPackage );

				// add package if it has the same type but different version
				if ( subPackage->mType == package->mType
						&& subPackage->mVersion != package->mVersion ) {
					if ( subPackage->mRepos.size()>0 )
						tReturn[ subPackage->mVersion.str() ] = subPackage;
				}
			}
		}
	}

	return tReturn;
}

Package* Database::newVersionAvailable( Package const* oldPackage ) const
{
	Package* newVersion = NULL;
	if ( oldPackage ) {
		map<string,Package*> alternatives = alternativeVersions( oldPackage );
		Package::Version tVersion = oldPackage->mVersion;
		foreach( auto& alternative, alternatives ) {
			// check if we do have a higher version (independent from the devel state)
			// the devel state have to be equal or higher
			if (alternative.second->mVersion > tVersion)
			{
				newVersion = alternative.second;
				tVersion = newVersion->mVersion;
			}
		}
	}
	return newVersion;
}

void Database::recursiveSelectStdSource( PackageGroup* depTree )
{
	Dependency* dep = dynamic_cast<Dependency*>( depTree );
	if ( dep ) {
		// select primary packages (not optional ones) for install
		if ( dep->isPrimary()
			&& !( dep->mDepFlag & Dependency::FACULTATIVE ) ) {
			setFlag( dep, Database::INSTALL );
		}
		foreach( PackageGroup* group, dep->mSubPackages ) {
			recursiveSelectStdSource( group );
		}
	}
	else
	{
		// we do have a group of packages -> therefore we have to select one

		// check if one of the subpackages is already scheduled for install
		// otherwise we have to select one of the subpackages
		PackageGroup* source = NULL;
		foreach( PackageGroup* group, depTree->mSubPackages ) {
			if ( getFlag(group) & Database::INSTALL ) {
				source = group;
			}
		}

		if ( !source ) {
			source = stdSource( depTree->mSubPackages );
			Dependency* stdDep = dynamic_cast<Dependency*>( source );
			if ( stdDep
				&& !( stdDep->mDepFlag & Dependency::FACULTATIVE )
				&& !( getFlag( stdDep ) & Database::DONOTINSTALL )) {
				setFlag( stdDep, Database::INSTALL );
				recursiveSelectStdSource( stdDep );
			}
		}
		else {
			Dependency* dep = dynamic_cast<Dependency*>( source );
			recursiveSelectStdSource( source );
		}
	}
}

RepositoryPtr Database::getRepoFromUrl(Url const& url) const
{
	typedef std::pair<std::string, RepositoryPtr> mapType;
	foreach( mapType const& repo, repos ) {
		if ( repo.second->enabled && ((url+"/").find(repo.second->url+"/") == 0) ) {
			return repo.second;
		}
	}

	foreach( mapType const& repo, genericRepos ) {
		if ( repo.second->enabled && ((url+"/").find(repo.second->url+"/") == 0) )
			return repo.second;
	}
	return RepositoryPtr();
}

string Database::getRepoNameFromUrl(Url const& url) const
{
	RepositoryPtr tRepo = getRepoFromUrl( url );
	return tRepo ? tRepo->name : url;
}

RepositoryPtr Database::getRepoFromName(string const& name) const
{
	map<string, RepositoryPtr>::const_iterator it = repos.find( name );
	if ( it != repos.end() )
		return it->second;
	return RepositoryPtr();
}

Url Database::getRepoUrlFromName(string const& name) const
{
	RepositoryPtr tRepo = getRepoFromName( name );
	return tRepo ? tRepo->url : "";
}

RepositoryPtr Database::getRepositoryForLocalFile( Path const& localPath )
{
	typedef pair<string, RepositoryPtr> mapType;
	foreach( mapType const& repo, repos ) {
		if ( repo.second->enabled && repo.second->isResponsibleFor( localPath ) )
			return repo.second;
	}

	foreach( mapType const& repo, genericRepos ) {
		if ( repo.second->enabled && repo.second->isResponsibleFor( localPath ) ) {
			return repo.second;
		}
	}

	return RepositoryPtr();
}

void Database::setRepositoryPromptProvider( PromptProvider* prompt )
{
	mPromptProvider = prompt;
	typedef pair<string, RepositoryPtr> mapType;
	foreach( mapType const& repo, repos ) {
		repo.second->prompt = prompt;
	}
}

void Database::gatherDependencies( Dependency* package,
                                   bool fullDependencies,
                                   int recursionLevel /*= 15*/ ) const
{
	MIRA_LOG(NOTICE)<<"Gather dependencies for " << package->mName << " (Recursion lvl: " << recursionLevel << ")";
	if ( recursionLevel <= 0 ) {
		MIRA_THROW(XLogical,"Possible loop detected in gatherDependencies in package " << package->mName);
	}

	// add all entries we depend on
	foreach( Package* pkg, package->mDependencies )
	{
		Dependency* dep = dynamic_cast<Dependency*>( pkg );
		assert( dep );
		if ( !isInstalled( dep->mName, true ) ) {
			bool dependencyFound = false;
			// try to find a group with that name first of all
			// if we have found one, we can simply add that group and
			// are done
			// otherwise we have to search for packages with that name
			// (there should be only one, since packages with the same name should
			// be grouped)
			PackageGroup* group = rootPackage.findGroup( dep->mName, true );
			Dependency* selected = NULL;
			if ( group ) {
				PackageGroup* newGroup = group->createDependency();

				package->addChild( newGroup );

				markVersions( newGroup->mSubPackages, pkg->mVersion );

				// if the full dependencies are requested we have to
				// start the recursion for every child
				// otherwise, we have to select the standard source package
				// and proceed
				if ( fullDependencies ) {
					foreach( PackageGroup* group, newGroup->mSubPackages ) {
						Dependency* subPackage = dynamic_cast<Dependency*>( group );
						//subPackage->mType = dep->mType;
						assert( subPackage );

						// check the dependencies of the children
						gatherDependencies( subPackage, fullDependencies, recursionLevel-1 );
					}
				}

				// try to select stdSource (even if we have resolved the full
				// dependencies) in order to detect dependency errors
				if ( stdSource( newGroup->mSubPackages, pkg->mVersion ) != NULL )
					dependencyFound = true;
			}
			else {
				vector<Package*> packages = rootPackage.findPackages( dep->mName, true );
				// if we don't have found a group with that name while more than
				// one package with that name exists, something is seriously going
				// wrong.
				assert( packages.size() <= 1 );
				if ( packages.size() == 1 ) {
					Package* newPackage = packages[0]->createDependency();
					Dependency* newDep = dynamic_cast<Dependency*>( newPackage );
					newDep->mDepFlag = dep->mDepFlag;
					assert( newDep );
					//newDep->mType = dep->mType;
					package->addChild( newDep );
					// add package and mark it for recursion if it is
					// a primary one
					if ( newDep->isPrimary() && (
							dep->mVersion.isIndifferent() ||
							newDep->mVersion.isBinaryCompatible(dep->mVersion)) ) {
						selected = newDep;
						dependencyFound = true;
					}

					markVersion( newDep, pkg->mVersion );
				}
			}
			if ( selected ) {
				gatherDependencies( selected, fullDependencies, recursionLevel-1 );
			}
			else if ( !dependencyFound && !(dep->mDepFlag & Dependency::FACULTATIVE) ) {
				if ( fullDependencies ) {
					package->mDepFlag &= Dependency::DependencyFlags(255)^Dependency::VERSION_OK;
					package->mDepFlag |= Dependency::VERSION_INCOMP;
				}
				else {
					if ( !mPromptProvider->showYesNoErrorMessage("Cannot find dependency \""
							+ dep->mName + "\" (Version " + dep->mVersion.str()
							+ ") required by \"" + package->mName + "\" !\nDo you want to proceed?") ) {
						MIRA_THROW( XLogical,
							"Cannot schedule package \"" + package->mName + "\" for install, since"
							" required dependency \""  + dep->mName + "\" (Version "
							+ dep->mVersion.str() + ") cannot be found!");
					}
				}
			}
		}
		else {
			bool versionOK = false;
			Package::Version tVersion;
			vector<Package*> packages = rootPackage.findPackages( dep->mName, true );
			foreach( Package* installedPkg, packages ) {
				if ( installedPkg->isPrimary() && isInstalled( installedPkg, true ) ) {
					if ( pkg->mVersion.isIndifferent() || installedPkg->mVersion.isBinaryCompatible(pkg->mVersion) ) {
						versionOK = true;
					} else {
						tVersion = installedPkg->mVersion;
					}
				}
			}
			if ( !versionOK ) {
				if ( fullDependencies ) {
					dep->mDepFlag &= Dependency::DependencyFlags(255)^Dependency::VERSION_OK;
					dep->mDepFlag |= Dependency::VERSION_INCOMP;
				}
				else {
					MIRA_THROW( XLogical,
						"Required dependency \"" + dep->mName + "\" is installed, but the install version "
						+ tVersion.str() + " isn't compatible with the required version " + dep->mVersion.str()
						+ " (required by \"" + package->mName + "\" ) !");
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
