/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MIRACenterMdiArea.C
 *    Description.
 *
 * @author Erik Einhorn
 * @date   2011/01/28
 */

#include <app/MIRACenterMdiArea.h>

#include <iostream>

#include <QMenu>

#include <error/Exceptions.h>

#include <visualization/ChannelDragnDropUtils.h>
#include <visualization/Visualization.h>
#include <views/VisualizationView.h>
#include <views/VisualizationControlPage.h>

#include <rcp/Workbench.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

MIRACenterMdiArea::MIRACenterMdiArea(QWidget* parent) : EditorPartArea(parent)
{
	setAcceptDrops(true);
	mOfferAuxiliaryVisualizations = false;
}

void MIRACenterMdiArea::addSubWindow(EditorPartWindow* window)
{
	EditorPartArea::addSubWindow(window);
	VisualizationView* visView = dynamic_cast<VisualizationView*>(window);
	if (visView)
		visView->setOfferAuxiliaryVisualizations(mOfferAuxiliaryVisualizations);
}

void MIRACenterMdiArea::setOfferAuxiliaryVisualizations(bool on)
{
	mOfferAuxiliaryVisualizations = on;

	foreach(EditorPartWindow* w, getSubWindows()) {
		VisualizationView* visView = dynamic_cast<VisualizationView*>(w);
		if (visView)
			visView->setOfferAuxiliaryVisualizations(on);
	}
}

void MIRACenterMdiArea::dragEnterEvent (QDragEnterEvent* event)
{
	const std::string channelID = ChannelDragnDropUtils::getChannelIDFromMimeData(event->mimeData());
	if(channelID.empty()) // wrong MIME type
		return;

	// otherwise check if we have a visualization for the channel
	auto classes = ChannelDragnDropUtils::getVisualizationsFor(channelID, Visualization::CLASS());
	if(classes.empty())
		return;
	event->acceptProposedAction();
}

// mapping between a visualization view and its visualization
struct VisualizationViewMapping {
	ClassProxy viewClass;
	ClassProxy visualizationClass;
};

void MIRACenterMdiArea::dropEvent(QDropEvent* event)
{

	Workbench* workbench = dynamic_cast<Workbench*>(parent());
	if(workbench==NULL)
		return;

	const QMimeData* mimeData = event->mimeData();
	const std::string channelID = ChannelDragnDropUtils::getChannelIDFromMimeData(event->mimeData());
	if(channelID.empty()) // wrong MIME type
		return;

	std::list<VisualizationViewMapping> visualizationViewMappings;

	// collect all mappings
	// TODO: do this only, if number of visualization views changes, to safe
	// some performance? However, the performance penalty shouldn't be that
	// big here.
	visualizationViewMappings.clear();
	auto viewClasses = VisualizationView::CLASS().getDerivedClasses();
	foreach(auto p, viewClasses)
	{
		VisualizationViewMapping mapping;
		mapping.viewClass = p.second;
		try {
			// get visualization class associated with this visualization view class ...
			std::string classIdentifier = mapping.viewClass.getMetaInfo("Visualization");
			mapping.visualizationClass =  ClassFactory::instance().getClassByIdentifier(classIdentifier);
			visualizationViewMappings.push_back(mapping);
		} catch(...) {
			// querying of the class fails, just continue. Drag'n'drop to the empty workspace will then
			// be disabled for this view
		}
	}

	Visualization* vis = NULL;

	auto classes = ChannelDragnDropUtils::getVisualizationsFor(channelID, Visualization::CLASS());
	if(classes.empty()) {
		return;
	} else {
		assert(classes.size() > 0);
		MIRA_LOG(DEBUG) << "Found " << classes.size() << " visualization for dropped channel '" << channelID << "'";

		ClassProxy selectedClass;
		if(classes.size()==1)
			selectedClass = classes.front();
		else {

			if (!mOfferAuxiliaryVisualizations) {
				// filter out auxiliary visualizations
				std::list<ClassProxy> notAuxClasses;
				foreach(ClassProxy c, classes)
				{
					std::string aux = c.getMetaInfo("Auxiliary");
					boost::algorithm::to_lower(aux);
					if (aux != "true")
						notAuxClasses.push_back(c);
				}
				if (!notAuxClasses.empty()) // if ALL classes are auxiliary, just keep all of them
					classes = notAuxClasses;
			}

			if(classes.size()==1)
				selectedClass = classes.front();
			else {
				// ask user to select a visualization

				QMenu* menu = new QMenu(this);
	#if (QT_VERSION >= QT_VERSION_CHECK(5, 1, 0))
				menu->setToolTipsVisible(true);
	#endif
				int i=0;
				foreach(ClassProxy c, classes)
				{
					std::string name = c.getMetaInfo("Name");
					if(name.empty())
						name = c.getName(); // use class name, if no meta name was specified

					// find a visualization view for the visualization class c
					bool haveView = false;
					std::string viewName;
					foreach(VisualizationViewMapping& m, visualizationViewMappings)
					{
						if(c.isDerivedFrom(m.visualizationClass)) {
							viewName = m.viewClass.getMetaInfo("Name");
							if(viewName.empty())
								viewName = m.viewClass.getName();
							haveView = true;
							break;
						}
					}

					if(!haveView) {
						++i;
						continue; // skip, since visualization has no specified view
					}

					std::string text = viewName + ": " + name;
					QAction* a = menu->addAction(QString::fromStdString(text));
					a->setData(i);
					a->setToolTip(QString::fromStdString(c.getMetaInfo("Description")));
					++i;
				}

				QAction* chosenAction = menu->exec(mapToGlobal(event->pos()));
				if(chosenAction==NULL)
					return; // user has canceled

				// go to the i-th class that was chosen
				auto it = classes.begin();
				std::advance(it, chosenAction->data().toInt());
				selectedClass = (*it);
			}
		}

		// instantiate the visualization
		vis = selectedClass.newInstance<Visualization>();
	}

	assert(vis!=NULL);

	// create a visualization view for the chosen visualization
	VisualizationView* visView = NULL;
	try {
		QWidget* w = NULL;
		foreach(VisualizationViewMapping& m, visualizationViewMappings)
		{
			if(vis->getClass().isDerivedFrom(m.visualizationClass)) {
				w = workbench->createPart(m.viewClass);
				break;
			}
		}
		visView = dynamic_cast<VisualizationView*>(w);
	}
	catch (...)	{
		// ignore all errors
	}

	if(visView==NULL) {
		// clean up
		delete vis;
		return;
	}

	assert(visView!=NULL);


	vis->setupDataConnectionChannel(channelID);
	vis->setName(channelID);
	visView->addVisualization(vis);

	if(visView->getControl())
		visView->getControl()->addVisualizationToEditor(vis);

	event->acceptProposedAction();
}

///////////////////////////////////////////////////////////////////////////////

}

