/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UnicycleBasedRobotModel.h
 *    A robot model class for unicycle based motion models (differential drive, tricycle)
 *
 * @author Tim Langner
 * @date   2013/02/04
 */

#ifndef _MIRA_UNICYCLEBASEDROBOTMODEL_H_
#define _MIRA_UNICYCLEBASEDROBOTMODEL_H_

#include <serialization/PropertySerializer.h>

#include <robot/RobotModel.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

class MIRA_ROBOT_DATATYPES_EXPORT UnicycleBasedRobotModel : public RobotModel
{
	MIRA_ABSTRACT_OBJECT(UnicycleBasedRobotModel)

public:

		// keep using Angle for serialization compatibility
		template<typename Reflector>
		void reflectDegrees(Reflector& r,
		                    const char* name, float& member, const char* comment) {
			if (Reflector::isReadOnly::value) {
				Anglef a(member);
				r.property(name, a, comment, PropertyHint(), REFLECT_CTRLFLAG_TEMP_TRACKING);
			} else {
				Anglef a;
				r.property(name, a, comment, PropertyHint(), REFLECT_CTRLFLAG_TEMP_TRACKING);
				member = a.rad();
			}
		}

		void reflectDegrees(PropertySerializer& r,
		                    const char* name, float& member, const char* comment) {
			r.property(name,
			           rad2degGetter(member), deg2radSetter(member, Deg2RadNonNegative),
			           comment, PropertyHints::minimum(0.f));
		}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, RigidModel);
		r.property("MaxBackwardVelocity", maxBackwardVelocity,
		           "The maximum velocity for driving backward (both wheels) [m/s]");
		r.property("MaxForwardVelocity", maxForwardVelocity,
		           "The maximum velocity for driving forward (both wheels) [m/s]");
		reflectDegrees(r, "MaxRotVelocity", maxRotVelocity,
		               "The maximum velocity for rotating [deg/s]");
		r.property("TransEpsilonVelocity", transEpsilonVelocity,
		           "The min. translational speed that the robot platform can do [m/s]. "
		           "Lower velocities result in standing still.");
		reflectDegrees(r, "RotEpsilonVelocity", rotEpsilonVelocity,
		               "The min. rotational speed that the robot platform can do [deg/s]. "
		               "Lower velocities result in standing still.");

		r.property("MaxTransAcceleration", maxTransAcceleration,
		           "Max. acceleration [m/s^2]");
		r.property("MaxTransDeceleration", maxTransDeceleration,
		           "Max. deceleration [m/s^2]");
		reflectDegrees(r, "MaxRotAcceleration", maxRotAcceleration,
		               "Max. acceleration [deg/s^2]");
		reflectDegrees(r, "MaxRotDeceleration", maxRotDeceleration,
		               "Max. deceleration [deg/s^2]");
		r.property("MaxEmergencyDeceleration", maxEmergencyDeceleration,
		           "Max. deceleration in case of an emergency [m/s^2]. Must be greater than zero.");
	}

	float getMaxTransVelocity() const;

	/**
	 * Generates a trajectory by accelerating/decelerating starting at velocity v
	 * for lookAheadTime time up to the velocity targetV.
	 * The resulting trajectory will contain samples+1 trajectory samples starting
	 * at position 0,0 with velocity v.
	 * It uses the acceleration limits given in the robot model.
	 */
	virtual PoseVelocityTrajectory generateTrajectory(Velocity2 v, const Velocity2& targetV,
	                                                  float lookAheadTime, int samples);

	/// The maximum velocity for driving backward (both wheels) [m/s]. Use
	/// negative values, if the robot is allowed to drive backward.
	float maxBackwardVelocity;

	/// The maximum velocity for driving forward (both wheels) [m/s].
	float maxForwardVelocity;

	/// The maximum velocity for rotating [deg/s].
	float maxRotVelocity;

	/**
	 * the min. speed that the robot platform can do, lower velocities result
	 * in standing still.
	 */
	float transEpsilonVelocity;
	float rotEpsilonVelocity;

	/// Max. acceleration.
	float maxTransAcceleration;
	/// Max. deceleration. Must be greater than zero.
	float maxTransDeceleration;
	/// Max. deceleration in case of an emergency. Must be greater than zero.
	float maxEmergencyDeceleration;

	/// Max. acceleration.
	float maxRotAcceleration;
	/// Max. deceleration. Must be greater than zero.
	float maxRotDeceleration;
};

/// Typedef for UnicycleBasedRobotModel pointer
typedef boost::shared_ptr<UnicycleBasedRobotModel> UnicycleBasedRobotModelPtr;

///////////////////////////////////////////////////////////////////////////////

inline float UnicycleBasedRobotModel::getMaxTransVelocity() const
{
	return std::max(std::abs(maxBackwardVelocity), std::abs(maxForwardVelocity));
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
