/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Odometry.h
 *    Odometry data type.
 *
 * @author Tim Langner
 * @date   2010/08/03
 */

#ifndef _MIRA_ODOMETRY_H_
#define _MIRA_ODOMETRY_H_

#include <limits>

#include <math/Eigen.h>
#include <transform/Pose.h>
#include <transform/Velocity.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Odometry class.
 * Contains a Pose for the position and
 * a Velocity for translational and rotational speed.
 */
template <int D>
class Odometry
{
public:

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

	/** @name Constructors and reflect */
	//@{

	Odometry() {}

	/**
	 * Construct a new odometry object
	 * @param[in] p The current pose.
	 * @param[in] v The current velocity.
	 */
	Odometry(const RigidTransform<float, D>& p, const RigidTransform<float, D>& v) :
		pose(p),
		velocity(v)
	{}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Pose", pose, "The position");
		r.property("Velocity", velocity, "The velocity");
	}

	//@}

public:
	/*
	 * Returns true if *this is approximately equal to other, 
	 * within the precision determined by prec.
	 * @return true if approximately equal, false otherwise
	 */
	bool isApprox(const Odometry<D>& other,
	              float prec = std::numeric_limits<float>::epsilon()) const
	{
		return pose.isApprox(other.pose, prec) &&
		       velocity.isApprox(other.velocity, prec);
	}

public:
	/// The pose
	RigidTransform<float, D> pose;

	/// The velocity
	RigidTransform<float, D> velocity;
};

///////////////////////////////////////////////////////////////////////////////

/// 2D Odometry
typedef Odometry<2> Odometry2;

/// 3D Odometry
typedef Odometry<3> Odometry3;

///////////////////////////////////////////////////////////////////////////////

}}

#endif
