/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeshVectorVisualization.C
 *    Description.
 *
 * @author Christoph Weinrich
 */

#include <boost/scoped_ptr.hpp>

#include <utils/Path.h>
#include <serialization/Serialization.h>
#include <serialization/SetterNotify.h>

#include <visualization/3d/MeshObject.h>
#include <visualization/3d/MeshFactory.h>
#include <visualization/Visualization3D.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>

//#include <transform/RigidTransform.h>
#include <transform/Pose.h>

#include <OGRE/OgreSceneManager.h>

using namespace Eigen;
using namespace std;

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

class MeshProperties : public mira::Object{
	MIRA_OBJECT(MeshProperties)
public:

	MeshProperties(){
		//mParent = iParent;

		const std::vector<MeshFactory::MeshFormatInfo> formats =
				MeshFactory::instance().getSupportedFormats();

		mSupportedExtensions = "";
		std::string allFormats = "Supported Meshes (";
		foreach(const MeshFactory::MeshFormatInfo& format, formats)
		{
			mSupportedExtensions += ";;" + format.description + " (*" + format.extension +")";
			allFormats += "*" + format.extension + " ";
		}

		mSupportedExtensions = allFormats + ")" + mSupportedExtensions;

		mMeshPath = "Sphere.mesh";
		mScale = 0.5;
		mColor = Ogre::ColourValue::Blue;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("MeshPath",  mMeshPath, //setter(&PoseVectorMeshVisualizationBase::setPath, this),
		           "The path to the file containing the mesh.", "",
		           PropertyHints::file(mSupportedExtensions));
		r.property("Scale", mScale, "Description",0.5f);
		r.property("Color", mColor, "The color of the meshes for pose vector "
				"visualization", Ogre::ColourValue::Blue);
	};

public:
	std::string mSupportedExtensions;
	Path mMeshPath;
	Ogre::ColourValue mColor;
	float mScale;

	//PoseVectorMeshVisualizationBase* mParent;
};


template <typename Pose>
class PoseVectorMeshVisualizationBase :  public Visualization3D
{
public:
	typedef typename PoseVectorTrait<Pose>::VectorType PoseVector;

	EIGEN_MAKE_ALIGNED_OPERATOR_NEW

public:
	/** @name Constructor, destructor and reflect */
	//@{

	PoseVectorMeshVisualizationBase() :
		//mMesh(NULL),
		mMeshsCreatable(false)
		//mNumberOfVisiblePoseVectors(0)
	{
		MeshProperties meshProperties;
		mMeshes.resize(1, meshProperties);
		mMeshCount = mMeshes.size();

		mDataChannel.setDataChangedCallback(
			boost::bind(&PoseVectorMeshVisualizationBase::dataChangedCallback, this, _1));
		mMeshIDsChannel.setDataChangedCallback(
			boost::bind(&PoseVectorMeshVisualizationBase::meshIDsChangedCallback, this, _1));
	}

	virtual ~PoseVectorMeshVisualizationBase()
	{
		if(getSite()==NULL)
			return;

		foreach(MeshObject* mesh, mMeshVector){
			delete mesh;
		}
	}

	/// The reflect method.
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Visualization3D);

		channelProperty(r, "PoseVector", mDataChannel, "The channel to be visualized");

		r.property("MeshCount", mMeshCount, setter(&PoseVectorMeshVisualizationBase::setMeshCount, this),
			"Number of Meshes",1, PropertyHints::minimum(1));
		// TODO: the notifier does not work yet
		// if it would work the meshes could be updated, even when the
		// poses do not update (paused tape etc.)
		r.property("Meshes", mMeshes, //setterNotify(mMeshes, &PoseVectorMeshVisualizationBase::setMeshes, this),
			"The meshes");
		channelProperty(r, "MeshIDsChannel", mMeshIDsChannel,
		                "Optional channel that contains the id of the mesh, that"
		                " shall be visualized",
		                Visualization3D::NOT_REQUIRED);
	}


	//@}

public:
	/** @name Public implementation of Visualization3D and Visualization*/
	//@{

	virtual void setupScene(IVisualization3DSite* site)
	{
		mNode = site->getSceneManager()->getRootSceneNode()->createChildSceneNode();
		setPaths();
	}

	virtual Ogre::SceneNode* getNode()
	{
		return mNode;
	}

	virtual void setEnabled(bool enabled)
	{
		Visualization3D::setEnabled(enabled);
		for(uint32 i = 0; i < mMeshVector.size(); ++i){
			if(i < mCurrentPoseVector.size()){
				mMeshVector[i]->setVisible(enabled);
			}
			else{
				mMeshVector[i]->setVisible(false);
			}
		}
	}

	virtual DataConnection getDataConnection()
	{
		//return DataConnection(mPoseVector);
		return DataConnection(mDataChannel);
	}

	//@}

public:
	/** @name Setters */
	//@{

	void setMeshCount(uint32 iMeshCount){
		mMeshes.resize(iMeshCount, mMeshes.at(0));
		mMeshCount = mMeshes.size();
	}

	void setMeshes(){
		setPaths();
	}

	MeshObject* createNewMesh(const Path& iMeshPath){

		if(iMeshPath.empty() || getSite()==NULL)
			return NULL;

		MeshObject* newMesh = NULL;

		Ogre::SceneManager* mgr = getSite()->getSceneManager();
		std::string s = iMeshPath.string();

		try {
			newMesh = new MeshObject(s, mgr, mNode);
			newMesh->setVisible(isEnabled());
		} catch(Exception& ex) {
			error("Mesh", ex.message());
			newMesh = NULL;
		}
		return newMesh;
	}

	MeshObject* createNewMesh(uint32 iPoseID, Path& oMeshPath){
		MeshObject* meshObject;
		if(iPoseID < mMeshIDs.size() && mMeshIDs.at(iPoseID) < mMeshes.size()){
			oMeshPath = mMeshes.at(mMeshIDs.at(iPoseID)).mMeshPath;
			meshObject = createNewMesh(oMeshPath);
		}
		else{
			oMeshPath = mMeshes.at(0).mMeshPath;
			meshObject = createNewMesh(oMeshPath);
		}
		if(meshObject == NULL){
			MIRA_THROW(XLogical, "This should not happen1");
		}
		return meshObject;
	}

	void checkPaths(){
		foreach(const MeshProperties& meshProperties, mMeshes){
			MeshObject* mesh = createNewMesh(meshProperties.mMeshPath);
			if(mesh != NULL){
				mMeshsCreatable = true;
				delete mesh;
				ok("Mesh");
			}
			else{
				mMeshsCreatable = false;
				//MIRA_THROW(XLogical, "Can not load mesh from path "<<mMeshPath);
			}
		}
	}

	void setPaths()
	{
		checkPaths();
		//TODO: change all meshes
		if(mMeshsCreatable){
			uint32 id = 0;
			foreach(auto mesh, mMeshVector){
				delete mesh;
				if(mMeshIDs.size() > id && mMeshes.size() > mMeshIDs.at(id)){
					mesh = createNewMesh(mMeshes.at(mMeshIDs.at(id)).mMeshPath);
				}
				else{
					mesh = createNewMesh(mMeshes.at(0).mMeshPath);
				}
				++id;
			}
		}


	}

	//@}

protected:
	/** @name Protected implementation Visualization*/
	//@{




	virtual void dataChanged()
	{

		mDataFrameID = mNewestDataRead->frameID;
		mDataTimestamp = mNewestDataRead->timestamp;
		mCurrentPoseVector = mNewestDataRead->value();
		if(mMeshIDsChannel.isValid()){
			mMeshIDs = mNewestMeshIdsRead->value();
		}
		else{
			mMeshIDs.clear();
		}
		mOldMeshPaths = mNewMeshPaths;
		mNewMeshPaths.clear();

		if(!mMeshsCreatable){
			return;
		}

		for(uint32 poseID = 0; poseID < mCurrentPoseVector.size(); ++poseID){
			// for all poseIDs where the old mesh vector is big enough
			if(poseID < mMeshVector.size()){
				//
				if(poseID < mOldMeshPaths.size() && poseID < mMeshIDs.size() && mMeshes.size() > mMeshIDs.at(poseID) && mMeshes.at(mMeshIDs.at(poseID)).mMeshPath == mOldMeshPaths.at(poseID)){
					mNewMeshPaths.push_back(mOldMeshPaths.at(poseID));
				}
				else{
					delete mMeshVector.at(poseID);
					Path meshPath;
					mMeshVector.at(poseID) = createNewMesh(poseID, meshPath);
					mNewMeshPaths.push_back(meshPath);
				}
			}
			else{
				Path meshPath;
				mMeshVector.push_back(createNewMesh(poseID, meshPath));
				mNewMeshPaths.push_back(meshPath);
			}
		}

		updatePoses();
		//mNumberOfVisiblePoseVectors = data->size();

		// hide unused poses
		for(std::size_t i=mCurrentPoseVector.size(); i<mMeshVector.size(); ++i)
		{
			//mPoses[i]->axes.setVisible(false);
			//mPoses[i]->cov.setVisible(false);
			mMeshVector[i]->setVisible(false);
		}
	}

	virtual void update(Duration dt)
	{
		//if(mMesh==NULL)
		//	MIRA_THROW(XInvalidRead, "Mesh not set or not loaded");
		//
		//// get the transform of the range scan
		//if(!mFrameID.empty()) {
		//	Time t = Time::now();
		//	mPose = getAuthority().getTransform<RigidTransform3f>(mFrameID, getSite()->getCameraFrame(), t);
		//}
		//
		//if(mMesh!=NULL)
		//	mMesh->setTransform(mPose);
		if (!mDataChannel.isValid())
			return;

		updatePoses();
	}

	//@}

private:
	void dataChangedCallback(ChannelRead<PoseVector> data)
	{
		mNewestDataRead = data;
		if(!mMeshIDsChannel.isValid()){
			dataChanged();
		}
		else if(mNewestDataRead->timestamp == mNewestMeshIdsRead->timestamp){
			//mDataFrameID = mNewestDataRead->frameID;
			//mDataTimestamp = mNewestDataRead->timestamp;
			//mCurrentPoseVector = mNewestDataRead->value();
			//mMeshIDs = mNewestMeshIdsRead->value();
			dataChanged();
		};
	}

	void meshIDsChangedCallback(ChannelRead<std::vector<uint32> > meshIDs)
	{
		mNewestMeshIdsRead = meshIDs;
		if(mNewestDataRead.isValid() && mNewestDataRead->timestamp == mNewestMeshIdsRead->timestamp){
			dataChanged();
		};
	}

	virtual void setPoseAndScale(MeshObject*& iMeshObject, const Pose& iPose, float iScale){
		iMeshObject->setTransform(iPose);
		iMeshObject->setScale(Ogre::Vector3(iScale, iScale, iScale));
	}

	void updatePoses()
	{
		//TODO: mDataTimestamp instead of Time::now()???
		//-> would be the same for PoseVectorVisualization

		Pose parentTransform = getAuthority().template getTransform<Pose>(
		    mDataFrameID, getSite()->getCameraFrame(), Time::now());

		if(mMeshVector.size() < mCurrentPoseVector.size()){
			return;
		}

		// set the poses
		for(std::size_t i=0; i<mCurrentPoseVector.size(); ++i)
		{
			Pose pose = parentTransform * mCurrentPoseVector[i];
			if(mMeshIDs.size() > i && mMeshes.size() > mMeshIDs[i]){
				mMeshVector[i]->setColor(mMeshes.at(mMeshIDs[i]).mColor);
				setPoseAndScale(mMeshVector[i], pose, mMeshes.at(mMeshIDs[i]).mScale);
			}
			else{
				mMeshVector[i]->setColor(mMeshes.at(0).mColor);
				setPoseAndScale(mMeshVector[i], pose, mMeshes.at(0).mScale);
			}
			mMeshVector[i]->setVisible(isEnabled());
		}
	}

private:
	// TODO: maybe shared pointer
	std::vector<uint32> mMeshIDs;
	//std::vector<uint32> mOldMeshIDs;
	std::vector<Path> mOldMeshPaths;
	std::vector<Path> mNewMeshPaths;
	Ogre::SceneNode* mNode;
	std::vector<MeshObject*> mMeshVector;
	uint32 mMeshCount;
	std::vector<MeshProperties> mMeshes;

	ChannelProperty<PoseVector> mDataChannel;
	ChannelProperty<std::vector<uint32> > mMeshIDsChannel;

	ChannelRead<PoseVector> mNewestDataRead;
	ChannelRead<std::vector<uint32> > mNewestMeshIdsRead;

	bool mMeshsCreatable;


	//TODO: replace this three by ChannelRead
	std::string mDataFrameID;
	Time mDataTimestamp;
	PoseVector mCurrentPoseVector;
};

///////////////////////////////////////////////////////////////////////////////

class Pose3VectorMeshVisualization3D : public PoseVectorMeshVisualizationBase<Pose3>
{
	MIRA_META_OBJECT(Pose3VectorMeshVisualization3D,
		("Name", "PoseVectorMesh3D")
		("Description", "Visualization of a vector of 3D points")
		("Category", "Pose"))

	virtual void setPoseAndScale(MeshObject*& iMeshObject, const Pose3& iPose, float iScale){
			iMeshObject->setTransform(Pose3(iPose.x(), iPose.y(), 0.0, iPose.yaw(), iPose.pitch(), iPose.roll()));
			float scale = iPose.z() * iScale;
			iMeshObject->setScale(Ogre::Vector3(scale , scale, scale));
		}
};

class Pose2VectorMeshVisualization3D : public PoseVectorMeshVisualizationBase<Pose2>
{
	MIRA_META_OBJECT(Pose2VectorMeshVisualization3D,
		("Name", "PoseVectorMesh2D")
		("Description", "Visualization of a vector of 2D points")
		("Category", "Pose"))
};

class Pose3CovVectorMeshVisualization3D : public PoseVectorMeshVisualizationBase<PoseCov3>
{
	MIRA_META_OBJECT(Pose3CovVectorMeshVisualization3D,
		("Name", "PoseCovVectorMesh3D")
		("Description", "Visualization of a vector of 3D covariance points")
		("Category", "Pose"))


	virtual void setPoseAndScale(MeshObject*& iMeshObject, const PoseCov3& iPose, float iScale){
			iMeshObject->setTransform(Pose3(iPose.x(), iPose.y(), 0.0, iPose.yaw(), iPose.pitch(), iPose.roll()));
			float scale = iPose.z() * iScale;
			iMeshObject->setScale(Ogre::Vector3(scale , scale, scale));
		}
};

class Pose2CovVectorMeshVisualization3D : public PoseVectorMeshVisualizationBase<PoseCov2>
{
	MIRA_META_OBJECT(Pose2CovVectorMeshVisualization3D,
		("Name", "PoseCovVectorMesh2D")
		("Description", "Visualization of a vector of 2D covaraince points")
		("Category", "Pose"))
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Pose3VectorMeshVisualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose2VectorMeshVisualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose3CovVectorMeshVisualization3D,
                         mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Pose2CovVectorMeshVisualization3D,
                         mira::Visualization3D);

MIRA_CLASS_SERIALIZATION(mira::gui::MeshProperties,
                         mira::Object);
