/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 *  and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PanoramaCameraVisualization.C
 *    Description.
 *
 * @author Steffen Mueller, Tim Langner, Erik Einhorn
 * @date   2018/09/17
 */

#include <OGRE/OgreFrustum.h>
#include <OGRE/OgreManualObject.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreHardwarePixelBuffer.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>
#include <image/Img.h>

#include <serialization/SetterNotify.h>

#include <visualization/3d/ImageObject.h>
#include <visualization/Visualization3DBasic.h>
#include <visualization/ColormapProperty.h>
#include <cameraparameters/PanoramaCameraIntrinsic.h>

namespace mira { namespace camera {

///////////////////////////////////////////////////////////////////////////////

class PanoramaCameraVisualization :  public Visualization3DBasic<Img<>>
{
	MIRA_META_OBJECT(PanoramaCameraVisualization,
			("Name", "PanoramaCamera")
			("Category", "Images")
			("Description", "Visualizes a panorama camera with cylinder image."))

	typedef Visualization3DBasic<Img<>> Base;
public:

	PanoramaCameraVisualization() :
		Base("Image"), mColormap("mira::GrayscaleColormap")
	{
		mColormap.setFilter(ColormapProperty::filterContinuous);

		mAlpha = 1.0f;
		mNumSegments=64;
		mFar = 1.0f;
		mMin =  0.0;
		mMax = -1.0;
		mImageSize = Size2i(0,0);

		mFrustum = NULL;
		mFrustumColor = Ogre::ColourValue(0.0f,0.0f,0.0f,0.5f);
		mLineColor = Ogre::ColourValue::Black;

		mImageObject = NULL;
		mIntrinsicParams=PanoramaCameraIntrinsicNormalized(-M_PI/2.,M_PI/2,-1,1,RigidTransform3f(0,0,0,0,0,0));
		mIntrinsicParamsChannel.setDataChangedCallback(boost::bind(&PanoramaCameraVisualization::onIntrinsicChanged, this, _1));
	}

	virtual ~PanoramaCameraVisualization() {

		if(mFrustum!=NULL)
			getSite()->getSceneManager()->destroyManualObject(mFrustum);

		delete mImageObject;

		if (!mFrustumMaterial.isNull())
			Ogre::MaterialManager::getSingletonPtr()->remove(mFrustumMaterial->getName());
	}

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		//MIRA_REFLECT_BASE(r, Base); // bypass Visualization3DBasic to add the channel by ourself
		MIRA_REFLECT_BASE(r, Visualization3D); // but call reflect of our grandparent class

		channelProperty(r, "Image", mDataChannel, "The camera image to be visualized", NOT_REQUIRED);
		channelProperty(r, "Intrinsic", mIntrinsicParamsChannel, "Optional channel that"
		                " contains the normalized panorama camera intrinsic parameters, if not specified", NOT_REQUIRED);

		r.roproperty("Size", mImageSize, "The size of the image");

		r.property("Radius", mFar, setterNotify(mFar, &PanoramaCameraVisualization::updateFrustum, this),
		           "The distance of image cylinder to the camera", 1.0f);
		r.property("NumSegments", mNumSegments, setterNotify(mNumSegments, &PanoramaCameraVisualization::updateFrustum, this),
		           "number of subdivisions for the cylinder", 64);

		r.property("Alpha", mAlpha, setter(&PanoramaCameraVisualization::setAlpha,this), "The alpha value of the image",1.0f, PropertyHints::limits<float>(0.0f,1.0f));

		r.property("Colormap", mColormap, "The color palette",
		           ColormapProperty("mira::GrayscaleColormap"));
		r.property("Min", mMin,
		           "Min. value in the image (only valid for float and uint16 images)",
		           0.0f);
		r.property("Max", mMax,
		           "Max. value in the image (only valid for float and uint16 images)",
		           -1.0f);

		r.property("Line Color", mLineColor, setterNotify(mLineColor, &PanoramaCameraVisualization::updateFrustum, this),
		           "The color of the frustum lines. Set alpha to 0 to hide the lines.", Ogre::ColourValue::Black);
		r.property("Frustum Color", mFrustumColor,  setterNotify(mFrustumColor, &PanoramaCameraVisualization::updateFrustum, this),
		           "The color of the frustum planes. Set alpha to 0 to hide the planes.", Ogre::ColourValue(0.0f,0.0f,0.0f,0.5f));
	}


public:

	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{

		mFrustumMaterial = Ogre::MaterialManager::getSingleton().create("Frustum"+toString(this)+"/Material", "MIRAMaterials");
		mFrustumMaterial->setReceiveShadows(false);
		mFrustumMaterial->setLightingEnabled(false);
		mFrustumMaterial->setCullingMode(Ogre::CULL_CLOCKWISE);
		mFrustumMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
		mFrustumMaterial->setDepthWriteEnabled(false);

		// create manual object for frustum
		mFrustum = mgr->createManualObject("Frustum" + toString(this));
		node->attachObject(mFrustum);

		mImageObject = new ImageObject(false, mgr, node);
		mImageObject->setAlpha(mAlpha);

		updateFrustum();
	}

	void onDataChanged(ChannelRead<Img<>> read)
	{
		const ContinuousColormap& colormap = dynamic_cast<const ContinuousColormap&>(mColormap.getColormap());
		mImageSize = read->value().size();
		mImageObject->setImage(read->value(),mMin, mMax, colormap);
	}

	void onIntrinsicChanged(ChannelRead<PanoramaCameraIntrinsicNormalized> read)
	{
		mIntrinsicParams = read->value();
		updateFrustum();
	}

	virtual void setEnabled(bool enabled)
	{
		Base::setEnabled(enabled);
		if (mImageObject != NULL)
			mImageObject->setVisible(enabled);
	}


	void updateFrustum()
	{

		const PanoramaCameraIntrinsicNormalized& p = mIntrinsicParams; // for abbrev.

		// compute coords of frustum
		float leftx = mFar* sin(p.minPhi) ;
		float leftz = mFar* cos(p.minPhi) ;
		float rightx = mFar* sin(p.maxPhi) ;
		float rightz = mFar* cos(p.maxPhi) ;

		float bottom = mFar * p.minHeight;
		float top = mFar* p.maxHeight;

		if(mFrustum) {
			//mFrustum->setRenderQueueGroup(Ogre::RENDER_QUEUE_MAX);
			mFrustum->clear();

			if(mFrustumColor.a>0.0f) {
				mFrustum->begin(mFrustumMaterial->getName(), Ogre::RenderOperation::OT_TRIANGLE_LIST);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mFrustumColor);
				mFrustum->position(rightx, top,    rightz); mFrustum->colour(mFrustumColor);
				mFrustum->position(rightx, bottom, rightz); mFrustum->colour(mFrustumColor);


				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mFrustumColor);
				mFrustum->position(leftx, bottom,  leftz); mFrustum->colour(mFrustumColor);
				mFrustum->position(leftx, top,  leftz); mFrustum->colour(mFrustumColor);

				mFrustum->end();
			}

			if(mLineColor.a>0.0f) {
				mFrustum->begin(mFrustumMaterial->getName(), Ogre::RenderOperation::OT_LINE_LIST);

				mFrustum->position(rightx, top,   rightz); mFrustum->colour(mLineColor);
				mFrustum->position(rightx, bottom,rightz); mFrustum->colour(mLineColor);

				mFrustum->position(leftx,  bottom,leftz); mFrustum->colour(mLineColor);
				mFrustum->position(leftx,  top,   leftz); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(rightx, top,   rightz); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(rightx, bottom, rightz); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(leftx, bottom,  leftz); mFrustum->colour(mLineColor);

				mFrustum->position(0.0f,0.0f,0.0f); mFrustum->colour(mLineColor);
				mFrustum->position(leftx, top,  leftz); mFrustum->colour(mLineColor);

				mFrustum->end();
			}
		}

		if(mImageObject) {
			// prepare image object
			Ogre::ManualObject* imageObj = mImageObject->getPlane();
			imageObj->clear();
			imageObj->begin(mImageObject->getMaterial()->getName(), Ogre::RenderOperation::OT_TRIANGLE_LIST);

			float anglediff=p.maxPhi-p.minPhi;
			for(size_t n=0;n<size_t(mNumSegments);n++)
			{
				float phi=p.minPhi+n * anglediff / mNumSegments;
				float phi1=p.minPhi+(n+1) * anglediff / mNumSegments;

				imageObj->position(sin(phi)*mFar, bottom, cos(phi)*mFar);
				imageObj->textureCoord(1.0f/mNumSegments*n, 0.0f);
				imageObj->position(sin(phi)*mFar, top, cos(phi)*mFar);
				imageObj->textureCoord(1.0f/mNumSegments*n, 1.0f);
				imageObj->position(sin(phi1)*mFar, bottom, cos(phi1)*mFar);
				imageObj->textureCoord(1.0f/mNumSegments*(n+1), 0.0f);

				imageObj->position(sin(phi)*mFar, top, cos(phi)*mFar);
				imageObj->textureCoord(1.0f/mNumSegments*n, 1.0f);
				imageObj->position(sin(phi1)*mFar, top, cos(phi1)*mFar);
				imageObj->textureCoord(1.0f/mNumSegments*(n+1), 1.0f);
				imageObj->position(sin(phi1)*mFar, bottom, cos(phi1)*mFar);
				imageObj->textureCoord(1.0f/mNumSegments*(n+1), 0.0f);
			}

			imageObj->end();
		}

	}



	void setAlpha(float alpha) {
		mAlpha = alpha;
		if(mImageObject)
			mImageObject->setAlpha(mAlpha);
	}





protected:

	float mFar;
	float mAlpha;
	int mNumSegments;

	ColormapProperty mColormap;
	float mMin, mMax;

	Ogre::MaterialPtr mFrustumMaterial;
	Ogre::ManualObject* mFrustum;
	ImageObject* mImageObject;
	Size2i mImageSize;

	Ogre::ColourValue mLineColor;
	Ogre::ColourValue mFrustumColor;

	ChannelProperty<PanoramaCameraIntrinsicNormalized> mIntrinsicParamsChannel;
	PanoramaCameraIntrinsicNormalized mIntrinsicParams;
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION( mira::camera::PanoramaCameraVisualization, mira::Visualization3D);
