###############################################################################
# Copyright (C) 2012 by
#   MetraLabs GmbH (MLAB), GERMANY
# and
#   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
# All rights reserved.
#
# Contact: info@mira-project.org
#
# Commercial Usage:
#   Licensees holding valid commercial licenses may use this file in
#   accordance with the commercial license agreement provided with the
#   software or, alternatively, in accordance with the terms contained in
#   a written agreement between you and MLAB or NICR.
#
# GNU General Public License Usage:
#   Alternatively, this file may be used under the terms of the GNU
#   General Public License version 3.0 as published by the Free Software
#   Foundation and appearing in the file LICENSE.GPL3 included in the
#   packaging of this file. Please review the following information to
#   ensure the GNU General Public License version 3.0 requirements will be
#   met: http://www.gnu.org/copyleft/gpl.html.
#   Alternatively you may (at your option) use any later version of the GNU
#   General Public License if such license has been publicly approved by
#   MLAB and NICR (or its successors, if any).
#
# IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
# THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
# "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
# FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
#
###############################################################################
#
# Build a binary ZIP file for a for usage in mirapackage.
#
# This cmake script file needs the following variables to be defined:
#
#   MIRA_INSTALL_PACKAGE:          Name of the package
#   MIRA_INSTALL_VERSION:          Full version (Major.Minor.Patch)
#   MIRA_INSTALL_DATECODE:         Build ID (e.g. date code)
#   MIRA_INSTALL_COMPONENTS:       Components of this package
#   MIRA_INSTALL_REQUIREMENTS:     List of requirements
#                                  (Format: "Pkg1=Major.Minor;Pkg2=Major.Minor;...")
#   MIRA_INSTALL_DESTPATH:         The destination path for the ZIP files.
#   MIRA_INSTALL_STATUS_FILE:      The path to ${InstallerRootDir}/status.txt
#                                  (see PackageInstaller.cmake)
#
#   MIRA_ROOT_DIR:                 Path to MIRA to locate some files.
#   BUILD_BINARY_DIR:              Current build environment
#
# Author: Christian Martin
#
###############################################################################

###############################################################################
# A macro for reading the current installation build status from the file
# ${InstallerRootDir}/status.txt (see PackageInstaller.cmake)
#
macro(__MIRA_GET_BINARY_VERSION pkgName outVersion)

	unset(${outVersion})

	file(READ "${MIRA_INSTALL_STATUS_FILE}" status)
	foreach(package_BuildStatus ${status})
		string(STRIP "${package_BuildStatus}" package_BuildStatus)
		separate_arguments(package_BuildStatus)
		list(LENGTH package_BuildStatus listLen)

		if(listLen EQUAL 8)
			list(GET package_BuildStatus 0 packageName)
			list(GET package_BuildStatus 5 newPackageVersion)

			if("${packageName}" STREQUAL "${pkgName}")
				set(${outVersion} ${newPackageVersion})
			endif()
		endif()

	endforeach()

endmacro(__MIRA_GET_BINARY_VERSION)

###############################################################################
# A macro for patching the package files. This macro will made a copy of the
# given package file and will add the following things:
#  - Add the macro MIRA_PACKAGE_TYPE(BINARY)
#  - Add the macro MIRA_BUILD_ID(${buildID})
#  - Extend/replace all version information in MIRA_REQUIRE_PACKAGE with the
#    information provided by MIRA_INSTALL_REQUIREMENTS
#
macro(MIRA_PATCH_PACKAGE_FILE pkgFile buildID)

	# Read the content of the package file.
	file(READ ${pkgFile} lines)

	# Replace all ";" by "§_§" and all "\n" with ";" for list separation.
	string(REPLACE ";" "§_§" lines "${lines}")
	string(REPLACE "\n" ";" lines "${lines}")

	# We need a special handling for empty lines. To avoid the disappearing of
	# this lines in the patched file, we add a single " " here.
	string(REPLACE ";;" "; ;" lines "${lines}")

	# Iterate over all lines
	file(WRITE ${pkgFile}.new "")
	foreach(line ${lines})
		# Revert the replacement of ";" by "§_§"
		string(REPLACE "§_§" ";" line "${line}")

		# Store lines with a single " " as empty lines (see above)
		if("${line}" STREQUAL " ")
			SET(line "")
		endif()

		# Look for MIRA_VERSION and insert MIRA_PACKAGE_TYPE add MIRA_BUILD_ID.
		string(REGEX MATCH "^MIRA_VERSION.*" matchResult "${line}")
		if(matchResult)
			file(APPEND ${pkgFile}.new "MIRA_PACKAGE_TYPE(BINARY)\n")
			file(APPEND ${pkgFile}.new "MIRA_BUILD_ID(${buildID})\n")
			file(APPEND ${pkgFile}.new "\n")
		endif()

		# Look for MIRA_REQUIRE_PACKAGE
		string(REGEX MATCH "^MIRA_REQUIRE_PACKAGE\\((.*)\\)" matchResult "${line}")

		# Don't write lines starting with MIRA_REQUIRE_PACKAGE
		if (NOT matchResult)
			# Store all other lines into the package file
			file(APPEND ${pkgFile}.new "${line}\n")
		else()
			# Get the version of the required packaged and added this
			# to the MIRA_REQUIRE_PACKAGE macro.
			set(requiredPkgName ${CMAKE_MATCH_1})
			foreach(requirement ${MIRA_INSTALL_REQUIREMENTS})
				string(REGEX REPLACE "\(.*\)=.*" "\\1" pkgName "${requirement}")
				string(REGEX REPLACE ".*=\(.*\)" "\\1" pkgVersion "${requirement}")
				if ("${requiredPkgName}" STREQUAL "${pkgName}")
					file(APPEND ${pkgFile}.new "MIRA_REQUIRE_PACKAGE(${pkgName} ${pkgVersion})\n")
				endif()
			endforeach()
		endif()
	endforeach()

	file(RENAME ${pkgFile}.new ${pkgFile})

endmacro(MIRA_PATCH_PACKAGE_FILE)

###############################################################################
# cleanup and re-create all directories

# The temporary install directory.
set(INSTALL_ROOT "${BUILD_BINARY_DIR}/install_root")

file(REMOVE_RECURSE ${INSTALL_ROOT})
file(MAKE_DIRECTORY ${INSTALL_ROOT})

###############################################################################
# Install all cmake components for the current package

foreach(component ${MIRA_INSTALL_COMPONENTS})
	exec_program(${CMAKE_COMMAND} ARGS
		-D "BUILD_TYPE=Release"
		-D "CMAKE_INSTALL_DO_STRIP=1"
		-D "CMAKE_INSTALL_COMPONENT=${component}"
		-D "CMAKE_INSTALL_PREFIX=${INSTALL_ROOT}"
		-P "${BUILD_BINARY_DIR}/cmake_install.cmake"
	)
endforeach()

###############################################################################
# Check if there is at least one package file installed.

file(GLOB_RECURSE pkgFileList "${INSTALL_ROOT}/*.package")
file(GLOB_RECURSE chlogFileList "${INSTALL_ROOT}/*.changelog")
list(LENGTH pkgFileList pkgFileListLen)
if(${pkgFileListLen} EQUAL 0)
	message(FATAL_ERROR "*** Error building ZIP for package ${MIRA_INSTALL_PACKAGE}: No installed package file.")
	return()
endif()

###############################################################################
# Modify the directory structure of the install files.
# (But not for MIRAenviroment and external, since these two packages use a
#  slightly different structure.)

if((NOT "${MIRA_INSTALL_PACKAGE}" STREQUAL "MIRAenvironment") AND
   (NOT "${MIRA_INSTALL_PACKAGE}" STREQUAL "external"))

	# Up to here, we might have the following structure installed
	#   ${INSTALL_ROOT}/foo/Bar/Bar.package
	#   ${INSTALL_ROOT}/bin/BarExample
	#   ${INSTALL_ROOT}/lib/libBar.so

	# Get the first package file from the list and determine parent path.
	# (The most packages contain only one package file.)
	list(GET pkgFileList 0 packageFileName)

	# If a packages contains multiple package files (e.g. sub kind of sub-
	# packages), we have to ensure, that we pick the major package file, which
	# should have the same name as the package itself.
	foreach(pkgFile ${pkgFileList})
		get_filename_component(pkgName ${pkgFile} NAME_WE)
		if (${pkgName} STREQUAL ${MIRA_INSTALL_PACKAGE})
			set(packageFileName ${pkgFile})
		endif()
	endforeach()

	get_filename_component(packageDir ${packageFileName} PATH)
	get_filename_component(packageDir ${packageDir} PATH)

	file(RELATIVE_PATH relPkgPath ${INSTALL_ROOT} ${packageFileName})
	get_filename_component(relPkgPath ${relPkgPath} PATH)
	get_filename_component(parentRelPkgPath ${relPkgPath} PATH)

	# Now we have extracted the following path components:
	#   packageDir:        ${INSTALL_ROOT}/foo
	#   relPkgPath:        foo/Bar
	#   parentRelPkgPath:  foo
	#
	# The zip file will contain the content of ${packageDir}. All other
	# (global) directories (like bin, lib, ...) will be moved and renamed
	# to ${packageDir}/MIRA_${name}.

	###########################################################################
	# Go over all 'root' entries, search all "global" MIRA directory (like bin,
	# lib, ...) and move them into ${packageDir}

	file(GLOB FileList RELATIVE ${INSTALL_ROOT} "${INSTALL_ROOT}/*")
	foreach(fileName ${FileList})
		# If ${relPkgPath} does not start with ${filename}, we have a global dir
		string(REGEX MATCH "^${fileName}" matchResult "${relPkgPath}")
		if(NOT matchResult)
			file(RENAME ${INSTALL_ROOT}/${fileName} ${packageDir}/MIRA_${fileName})
		endif()
	endforeach()

endif()

###############################################################################
# Create zip destination directory

set(destDir "${MIRA_INSTALL_DESTPATH}/${MIRA_INSTALL_PACKAGE}-${MIRA_INSTALL_VERSION}-${MIRA_INSTALL_DATECODE}")
file(REMOVE_RECURSE ${destDir})
file(MAKE_DIRECTORY ${destDir})

# Copy all installed *.package files to the 'root' directory
foreach(pkgFile ${pkgFileList})
	MIRA_PATCH_PACKAGE_FILE(${pkgFile} ${MIRA_INSTALL_DATECODE})
	file(COPY ${pkgFile} DESTINATION ${destDir})
endforeach()

foreach(chlogFile ${chlogFileList})
	MIRA_PATCH_PACKAGE_FILE(${chlogFile} ${MIRA_INSTALL_DATECODE})
	file(COPY ${chlogFile} DESTINATION ${destDir})
endforeach()

###############################################################################
# get some more system information

if(WIN32)
	set(SYSTEM_ARCH "win32")
else()
	exec_program(uname ARGS -m OUTPUT_VARIABLE SYSTEM_ARCH)
endif()

###############################################################################
# Create the zip archive for all files

SET(zipFileName MIRA-${MIRA_INSTALL_PACKAGE}-${MIRA_INSTALL_VERSION}-${MIRA_INSTALL_DATECODE}-${SYSTEM_ARCH}.zip)

message(STATUS "*** Building ZIP package ${zipFileName} in ${destDir}")

if(WIN32)
	set(moreZipArgs "")
else()
	# store symbolic links as the link instead of the referenced file
	set(moreZipArgs "-y")
endif()

# create the zip archive using the "zip" command
exec_program(zip ${INSTALL_ROOT}/${parentRelPkgPath} ARGS
	-q -r ${moreZipArgs} ${zipFileName} *
	RETURN_VALUE zipResult
)
if(NOT ${zipResult} EQUAL 0)
	message(FATAL_ERROR "*** Failed to create ZIP file")
endif()

# copy the zip file to the destination directory
file(COPY ${INSTALL_ROOT}/${parentRelPkgPath}/${zipFileName} DESTINATION ${destDir})

###############################################################################
# Clean up

file(REMOVE_RECURSE ${INSTALL_ROOT})
