/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditorTreeView.h
 *    For internal use by PropertyEditor only!
 *
 * @author Erik Einhorn
 * @date   2011/05/09
 */

#ifndef _MIRA_PROPERTYEDITOR_H_
#  error "This file must be included by PropertyEditor.C"
#endif

#include <QAbstractItemModel>
#include <QTreeView>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PropertyEditor::Model : public QAbstractItemModel
{
	friend class PropertyEditor; // to call some protected QAbstractItemModel methods

public:
	Model(PropertyEditor* editor, QTreeView* view) :
		QAbstractItemModel(editor),
		mEditor(editor), mTreeView(view) {}

	virtual ~Model() {}

	//Returns the index of the item in the model specified by the given row, column and parent index.
	QModelIndex index(int row, int column, const QModelIndex& parent = QModelIndex()) const;

	// Returns the parent of the model item with the given index.
	// If the item has no parent, an invalid QModelIndex is returned.
	QModelIndex parent(const QModelIndex& index) const;

	int rowCount(const QModelIndex& parent = QModelIndex()) const;

	int columnCount(const QModelIndex & parent = QModelIndex()) const;

	// Returns the data for the given role for the item referred to by the index.
	QVariant data(const QModelIndex &index, int role) const;

	QVariant headerData(int section, Qt::Orientation orientation, int role = Qt::DisplayRole) const;

	//bool setData(const QModelIndex &index, const QVariant &value, int role = Qt::EditRole);

	Qt::ItemFlags flags(const QModelIndex &index) const;

	const PropertyNode* index2node(const QModelIndex& index) const;
	PropertyNode* index2node(const QModelIndex& index);
	const QModelIndex& node2index(const PropertyNode* node) const;

	QColor getBackgroundColor(const QModelIndex& index) const;

public:

	std::vector<std::pair<PropertyNode*, QVariant>> mProperties;

private:

	PropertyEditor* mEditor;
	QTreeView* mTreeView;
	mutable std::map<const PropertyNode*,QModelIndex> mNodesToIndex;
};

///////////////////////////////////////////////////////////////////////////////

QModelIndex PropertyEditor::Model::index(int row, int column,
                                         const QModelIndex& parent) const
{
	PropertyNode* node = NULL;
	if (parent.isValid()) {
		const PropertyNode* parentNode = index2node(parent);
		if (row >= (int)parentNode->children().size())
			return QModelIndex();
		node = parentNode->children()[row];
	} else {
		// top level node
		if(mEditor->getHideSingleRootNode() && mProperties.size()==1) {
			const PropertyNode* parentNode = mProperties[0].first;
			if (row >= (int)parentNode->children().size())
				return QModelIndex();
			node = parentNode->children()[row];
		} else { // show top level nodes
			if (row >= (int)mProperties.size())
				return QModelIndex();
			node = mProperties[row].first;
		}
	}

	assert(node!=NULL);

	QModelIndex idx = createIndex(row, column, (void*)node);
	if(column==0)
		mNodesToIndex[node] = idx;
	return idx;
}

// Returns the parent of the model item with the given index.
// If the item has no parent, an invalid QModelIndex is returned.
QModelIndex PropertyEditor::Model::parent(const QModelIndex& index) const
{
	//std::cout << "PARENT: " << index.column() << std::endl;
	if (index.isValid()) {
		const PropertyNode* parent = index2node(index)->parent();
		return node2index(parent);
	}
	return QModelIndex();
}

int PropertyEditor::Model::rowCount(const QModelIndex& parent) const
{
	if(parent.isValid())
		return (int)index2node(parent)->children().size();
	else {
		// top level nodes
		if(mEditor->getHideSingleRootNode() && mProperties.size()==1) {
			const PropertyNode* parentNode = mProperties[0].first;
			return (int)parentNode->children().size();
		} else // show top level nodes
			return (int)mProperties.size();
	}
}

int PropertyEditor::Model::columnCount(const QModelIndex & parent) const
{
	return 2;
}

// Returns the data for the given role for the item referred to by the index.
QVariant PropertyEditor::Model::data(const QModelIndex &index, int role) const
{
	if(!index.isValid())
		return QVariant();

	const PropertyNode* node = index2node(index);
	switch(role)
	{
		case Qt::DisplayRole:
			if(index.column()==0)
				return QVariant(mEditor->getName(node).c_str());
			else if(index.column()==1)
				return QVariant(mEditor->getText(node).c_str());
			else
				return QVariant();
		case Qt::ForegroundRole:
			if(index.column()==0 || index.column()==1)
			{
				if (node->isReadOnly())
					return QVariant(QColor(Qt::darkGray));
				return QVariant(QColor(Qt::black));

			}
			else
				return QVariant();
		case Qt::DecorationRole:

			// do not show an icon, if this property is edited at the moment
			if(mEditor->getCurrentEditedProperty()==node)
				return QVariant();

			if(index.column()==1)
				return QVariant(mEditor->getIcon(node));
			else
				return QVariant();
		case Qt::ToolTipRole:
		{
			if(index.column()==1) {
				// show text (that is shown in DisplayRole) as tooltip, if
				// that text is elided. Otherwise show the comment as tooltip
				QString text = mEditor->getText(node).c_str();
				QRect visualRect = mTreeView->visualRect(index);
				QFontMetrics metrics = mTreeView->fontMetrics();
				const int border = 10; // the left and right border of the text are ca. 5 pixels
				if(metrics.boundingRect(text).width() > (visualRect.width()-border))
					return QVariant(text);
			}

			return QVariant(node->comment().c_str());
		}
		case Qt::SizeHintRole:
			return QSize(0,QFontMetrics(QFont()).lineSpacing()+6);
/*
		case Qt::BackgroundRole:
			return QVariant(getBackgroundColor(index));
*/
			default:
			return QVariant();
	}
}

QVariant PropertyEditor::Model::headerData(int section,
                                           Qt::Orientation orientation,
                                           int role) const
{
	if (orientation != Qt::Horizontal || role != Qt::DisplayRole)
		return QVariant();

	switch (section)
	{
		case 0:
			return tr("Property");
		case 1:
			return tr("Value");
		default:
			return QVariant();
	}
}

Qt::ItemFlags PropertyEditor::Model::flags(const QModelIndex &index) const
{
	if (index.isValid() && index.column() == 1)
		return QAbstractItemModel::flags(index) | Qt::ItemIsEditable;
	else
		return QAbstractItemModel::flags(index);
}

const PropertyNode* PropertyEditor::Model::index2node(const QModelIndex& index) const
{
	if(!index.isValid())
		return NULL;

	const PropertyNode* node = static_cast<const PropertyNode*>(index.internalPointer());
	assert(node!=NULL);
	return node;
}

PropertyNode* PropertyEditor::Model::index2node(const QModelIndex& index)
{
	const PropertyEditor::Model* This = this;
	return const_cast<PropertyNode*>(This->index2node(index));
}

const QModelIndex& PropertyEditor::Model::node2index(const PropertyNode* node) const
{
	static QModelIndex sInvalid;
	auto it = mNodesToIndex.find(node);
	if(it==mNodesToIndex.end())
		return sInvalid;
	return it->second;
}

QColor PropertyEditor::Model::getBackgroundColor(const QModelIndex& index) const
{
	const PropertyNode* p = index2node(index);

	// find a color
	assert(p!=NULL);
	QColor c = mEditor->getBackgroundColor(p); // some delegates provide their own color

	QModelIndex idx = index;
	while(!c.isValid() && idx.isValid())
	{
		p = index2node(idx);
		assert(p!=NULL);
		c = mEditor->getColor(p); // get color that might be set for the property

		idx = idx.parent(); // move up to parent

		// if no parent, but only one top level node that is hidden
		if(!idx.isValid() && mEditor->getHideSingleRootNode() && mProperties.size()==1)
			c = mEditor->getColor(mProperties[0].first); // get color only from that root node
	}
	
	return c;
}

///////////////////////////////////////////////////////////////////////////////

}
