/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditorTreeView.h
 *    For internal use by PropertyEditor.C only!
 *
 * @author Erik Einhorn
 * @date   2011/05/09
 */

#ifndef _MIRA_PROPERTYEDITOR_H_
#  error "This file must be included by PropertyEditor.C"
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PropertyEditor::DelegateRegistry :
	public LazySingleton<PropertyEditor::DelegateRegistry>
{
public:

	enum DelegateKind {
		TEXT=0,
		ICON,
		COLOR,
		EDITOR,
		DIALOG,
		NAME,

		COUNT
	};

public:

	DelegatePtr getDelegate(const PropertyNode* property, DelegateKind kind);

private:

	DelegatePtr getDelegate(const Typename& type, DelegateKind kind);

	// ask class factory for all Delegate classes and add new ones (if any)
	void lookForDelegates();

private:

	std::map<ClassProxy, DelegatePtr> mKnownDelegates;

	typedef std::map<Typename, DelegatePtr> DelegatesMap;
	DelegatesMap mDelegates[COUNT];
};

///////////////////////////////////////////////////////////////////////////////

PropertyEditor::DelegatePtr PropertyEditor::DelegateRegistry::getDelegate(const PropertyNode* property,
                                                                          DelegateKind kind)
{
	assert(property);
	DelegatePtr p = getDelegate(property->type(), kind);

	// if we did not find a proper delegate, but if we have an enumeration hint
	// then try to handle it as "normal" int
	if(p==NULL && property->hasHint("enumeration"))
		p = getDelegate(typeName<int>(), kind);

	return p;
}

PropertyEditor::DelegatePtr PropertyEditor::DelegateRegistry::getDelegate(const Typename& type,
                                                                          DelegateKind kind)
{
	// look for an appropriate text delegate
	auto it = mDelegates[kind].find(type);

	if(it==mDelegates[kind].end()) {
		// if not found, update our known delegates (there might be some
		// new since last time, since a library could have been loaded)
		lookForDelegates();

		// look for the delegate again
		it = mDelegates[kind].find(type);

		// still nothing found, so abort
		if(it==mDelegates[kind].end())
			return DelegatePtr();
	}
	return it->second;
}

void PropertyEditor::DelegateRegistry::lookForDelegates()
{
	typedef std::map<std::string, ClassProxy > ClassMap;
	ClassMap delegateClasses = Delegate::CLASS().getDerivedClasses();

	if(delegateClasses.size()<=mKnownDelegates.size())
		return; // already abort here, if there are no new classes since last time

	// loop through all delegate classes
	foreach(ClassMap::value_type& c, delegateClasses)
	{
		if(mKnownDelegates.count(c.second)==0) { // if this class is new
			// instantiate the delegate
			DelegatePtr newDelegate(c.second.newInstance<Delegate>());

			// the delegate is now known
			mKnownDelegates.insert(std::make_pair(c.second, newDelegate));

			// ask the delegate for it's capabilities and supported types
			const auto& supportedTypes = newDelegate->supportedTypes();

			foreach(const Delegate::SupportedType& t, supportedTypes)
			{
				if(t.flags & Delegate::SupportedType::TEXT)
					mDelegates[TEXT].insert(std::make_pair(t.type, newDelegate));
				if(t.flags & Delegate::SupportedType::ICON)
					mDelegates[ICON].insert(std::make_pair(t.type, newDelegate));
				if(t.flags & Delegate::SupportedType::COLOR)
					mDelegates[COLOR].insert(std::make_pair(t.type, newDelegate));
				if(t.flags & Delegate::SupportedType::EDITOR)
					mDelegates[EDITOR].insert(std::make_pair(t.type, newDelegate));
				if(t.flags & Delegate::SupportedType::DIALOG)
					mDelegates[DIALOG].insert(std::make_pair(t.type, newDelegate));
				if(t.flags & Delegate::SupportedType::NAME)
					mDelegates[NAME].insert(std::make_pair(t.type, newDelegate));
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////

}
