/***************************************************************************
qconsole.cpp  -  description
-------------------
begin                : mar mar 15 2005
copyright            : (C) 2005 by Houssem BDIOUI
email                : houssem.bdioui@gmail.com
 ***************************************************************************/

// Source: http://sourceforge.net/projects/qconsole/
// migrated to Qt4 by YoungTaek Oh. date: Nov 29, 2010
// adapted to MIRA and removed unnecessary bloat (e.g. saved script, etc)

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <widgets/QConsole.h>

//#include <QDebug>
#include <QVBoxLayout>
#include <QApplication>
#include <QScrollBar>
#include <QDesktopWidget>
#include <QTextBlock>
#include <QTextDocumentFragment>
#include <QMimeData>

#include <iostream>

#include <utils/Foreach.h>

namespace mira {

MIRA_EXTENSIBLE_ENUM_DEFINE(QConsole::SearchMode, SEARCH_NONE, SEARCH_HISTORY)

/**
 * Subclassing QListWidget
 *
 * @author YoungTaek Oh
 */
class QConsolePopupListWidget : public QListWidget
{

public:
	QConsolePopupListWidget(QWidget *parent = 0): QListWidget(parent) {
		setUniformItemSizes(true);
		setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	}
	virtual ~QConsolePopupListWidget() { }

protected:
	virtual QSize sizeHint() const;
	virtual void keyPressEvent(QKeyEvent *e) {
		if (e->key() == Qt::Key_Tab ||
				e->key() == Qt::Key_Return)
			Q_EMIT itemActivated(currentItem());
		else
			QListWidget::keyPressEvent(e);
	}
};

///////////////////////////////////////////////////////////////////////////////

QSize QConsolePopupListWidget::sizeHint() const
{
	QAbstractItemModel *model = this->model();
	QAbstractItemDelegate *delegate = this->itemDelegate();
	const QStyleOptionViewItem sovi;
	int left, top, right, bottom = 0;
#if QT_VERSION >= 0x040600
	QMargins margin = this->contentsMargins();

	top = margin.top();
	bottom = margin.bottom();
	left = margin.left();
	right = margin.right();
#else
	getContentsMargins(&left, &top, &right, &bottom);
#endif
	const int vOffset = top + bottom;
	const int hOffset = left + right;



	bool vScrollOn = false;
	int height = 0;
	int width = 0;
	for (int i=0; i<this->count(); ++i) {
		QModelIndex index = model->index(i, 0);
		QSize itemSizeHint = delegate->sizeHint(sovi, index);
		if (itemSizeHint.width() > width)
			width = itemSizeHint.width();

		// height
		const int nextHeight = height + itemSizeHint.height();
		if (nextHeight + vOffset < this->maximumHeight())
			height = nextHeight;
		else {
			// early termination
			vScrollOn = true;
			break;
		}
	}

	QSize sizeHint(width + hOffset, 0);
	sizeHint.setHeight(height + vOffset);
	if (vScrollOn) {
		int scrollWidth = this->verticalScrollBar()->sizeHint().width();
		sizeHint.setWidth(sizeHint.width() + scrollWidth);
	}
	return sizeHint;
}

QConsolePopupCompleter::QConsolePopupCompleter(const QStringList& sl, QWidget *parent)
: QDialog(parent, Qt::Popup)
{
	setModal(true);

	listWidget_ = new QConsolePopupListWidget();
	listWidget_->setMaximumHeight(200);
	//qDebug() << "sizeHint(): " << listWidget_->sizeHint();
	Q_FOREACH(QString str, sl) {
		QListWidgetItem *item = new QListWidgetItem;
		item->setText(str);
		listWidget_->addItem(item);
	}
	//qDebug() << "sizeHint(): " << listWidget_->sizeHint();
	listWidget_->setFixedSize(listWidget_->sizeHint());
	listWidget_->setCurrentRow(0);


	QLayout *layout = new QVBoxLayout();
	layout->setSizeConstraint(QLayout::SetFixedSize);
	layout->setContentsMargins(0, 0, 0, 0);
	layout->addWidget(listWidget_);

	setLayout(layout);

	// connect signal
	connect(listWidget_, SIGNAL(itemActivated(QListWidgetItem *)),
	        SLOT(onItemActivated(QListWidgetItem*)));
}

QConsolePopupCompleter::~QConsolePopupCompleter()
{
}

void QConsolePopupCompleter::showEvent(QShowEvent* /*event*/)
{
	listWidget_->setFocus();
}

void QConsolePopupCompleter::onItemActivated(QListWidgetItem *event)
{
	selected_ = event->text();
	done(QDialog::Accepted);
}

/**
 * @brief execute PopupCompleter at appropriate position.
 *
 * @param parent Parent of this popup completer. usually QConsole.
 * @return see QDialog::exec
 * @see QDialog::exec
 */
int QConsolePopupCompleter::exec(QTextEdit *parent)
{
	QSize popupSizeHint = this->sizeHint();
	QRect cursorRect = parent->cursorRect();
	QPoint globalPt = parent->mapToGlobal(cursorRect.bottomRight());
	QDesktopWidget *dsk = QApplication::desktop();
	QRect screenGeom = dsk->screenGeometry(this);
	if (globalPt.y() + popupSizeHint.height() > screenGeom.height()) {
		globalPt = parent->mapToGlobal(cursorRect.topRight());
		globalPt.setY(globalPt.y() - popupSizeHint.height());
	}
	this->move(globalPt);
	this->setFocus();
	return QDialog::exec();
}

/**
 * returns a common word of the given list
 *
 * @param list String list
 *
 * @return common word in the given string.
 */
static
QString getCommonWord(QStringList& list)
{
	QChar ch;
	QVector<QString> strarray = list.toVector();
	QString common;
	int col = 0,  min_len;
	bool cont = true;

	// get minimum length
	min_len = strarray.at(0).size();
	for (int i=1; i<strarray.size(); ++i) {
		const int len = strarray.at(i).size();
		if (len < min_len)
			min_len = len;
	}

	while(col < min_len) {
		ch = strarray.at(0)[col];
		for (int i=1; i<strarray.size(); ++i) {
			const QString& current_string = strarray.at(i);
			if (ch != current_string[col])
			{
				cont = false;
				break;
			}
		}
		if (!cont)
			break;

		common.push_back(ch);
		++col;
	}
	return common;
}

////////////////////////////////////////////////////////////////////////////////

//Clear the console
void QConsole::clear()
{
	QTextEdit::clear();
}

//Reset the console
void QConsole::reset(const QString &welcomeText)
{
	clear();
	resetSearch();

	//set the style of the QTextEdit
/*#ifdef __APPLE__
	setCurrentFont(QFont("Monaco"));
#else
	QFont f;
	f.setFamily("Courier");
	setCurrentFont(f);
#endif*/

	append(welcomeText);
//	append(""); // instead, subclasses can enter a new paragraph (configured to their requirements)

	//init attributes
	historyIndex = 0;
	history.clear();
}

//QConsole constructor (init the QTextEdit & the attributes)
QConsole::QConsole(QWidget *parent, const QString &welcomeText)
: QTextEdit(parent), errColor_(Qt::red),
  outColor_(Qt::blue), completionColor(Qt::darkGreen),
  promptLength(0), promptPosition(0), currentSearchMode(SearchMode::SEARCH_NONE)
{
	QPalette palette = QApplication::palette();
	setCmdColor(palette.text().color());

	//Disable undo/redo
	setUndoRedoEnabled(false);

	resetSearch();

	//Disable context menu
	//This menu is useful for undo/redo, cut/copy/paste, del, select all,
	// see function QConsole::contextMenuEvent(...)
	//setContextMenuPolicy(Qt::NoContextMenu);

	connect(this, SIGNAL(cursorPositionChanged()), this, SLOT(onCursorPositionChanged()));


	//resets the console
	reset(welcomeText);
#if QT_VERSION > QT_VERSION_CHECK(5, 12, 0)
	const int tabwidth = QFontMetrics(currentFont()).horizontalAdvance('a') * 4;
	setTabStopDistance(tabwidth);
#else
	const int tabwidth = QFontMetrics(currentFont()).width('a') * 4;
	setTabStopWidth(tabwidth);
#endif
}

//Sets the prompt and cache the prompt length to optimize the processing speed
void QConsole::setPrompt(const QString &newPrompt, bool display)
{
	prompt = newPrompt;
	promptLength = prompt.length();
	//display the new prompt
	if (display)
		displayPrompt();
}


//Displays the prompt and move the cursor to the end of the line.
void QConsole::displayPrompt()
{
	//Prevent previous text displayed to be undone
	setUndoRedoEnabled(false);
	//displays the prompt
	setTextColor(cmdColor_);
	QTextCursor cur = textCursor();
	cur.insertText(prompt);
	cur.movePosition(QTextCursor::EndOfBlock);
	setTextCursor(cur);
	//Saves the paragraph number of the prompt
	promptPosition = cur.position();

	//Enable undo/redo for the actual command
	setUndoRedoEnabled(true);
}

void QConsole::onCursorPositionChanged()
{
	//if(textCursor().position() < promptPosition)
	//	setHome();
}

void QConsole::setFont(const QFont& f) {
	QTextCharFormat format;
	QTextCursor oldCursor = textCursor();
	format.setFont(f);
	selectAll();
	textCursor().setBlockCharFormat(format);
	setCurrentFont(f);
	setTextCursor(oldCursor);
}



//Give suggestions to autocomplete a command (should be reimplemented)
//the return value of the function is the string list of all suggestions
QStringList QConsole::suggestCommand(const QString&, QString& prefix)
{
	prefix = "";
	return QStringList();
}

//Treat the tab key & autocomplete the current command
void QConsole::handleTabKeyPress()
{
	QString command = getCurrentCommand();
	QString commandPrefix;
	QStringList sl = suggestCommand(command, commandPrefix);
	if (sl.count() == 0)
		textCursor().insertText("\t");
	else {
		if (sl.count() == 1)
			replaceCurrentCommand(commandPrefix + sl[0]);
		else
		{
			// common word completion
			QString commonWord = getCommonWord(sl);
			command = commonWord;
			QConsolePopupCompleter *popup = new QConsolePopupCompleter(sl);
			if (popup->exec(this) == QDialog::Accepted)
				replaceCurrentCommand(commandPrefix + popup->selected());
			delete popup;
		}
	}
}

// If return pressed, do the evaluation and append the result
void QConsole::handleReturnKeyPress()
{
	//Get the command to validate
	QString command = getCurrentCommand();
	//execute the command and get back its text result and its return value
	if (isCommandComplete(command))
		execCommand(command, false);
	else
	{
		append("");
		moveCursor(QTextCursor::EndOfLine);
	}
}

void QConsole::handleUpKeyPress()
{
	if (history.size()>0)
	{
		QString command = getCurrentCommand();
		do
		{
			if (historyIndex)
				historyIndex--;
			else
			{
				break;
			}
		} while(history[historyIndex] == command);
		replaceCurrentCommand(history[historyIndex]);
	}
}

void QConsole::handleDownKeyPress()
{
	if (history.size()>0 && historyIndex!=history.size())
	{
		QString command = getCurrentCommand();
		do
		{
			if (++historyIndex >= history.size())
				break;
		} while(history[historyIndex] == command);

		if (historyIndex >= history.size()) {
			historyIndex = history.size();
			replaceCurrentCommand("");
		} else
			replaceCurrentCommand(history[historyIndex]);
	}
}


void QConsole::setHome()
{
	QTextCursor cur = textCursor();
	cur.setPosition(promptPosition);
	setTextCursor(cur);
}

//Reimplemented key press event
void QConsole::keyPressEvent( QKeyEvent *e )
{
	//If the user wants to copy or cut outside
	//the editing area we perform a copy
	if(textCursor().hasSelection())
	{
		if(e->modifiers() == Qt::CTRL) {
			if( e->matches(QKeySequence::Cut) ) {
				e->accept();
				if(!isSelectionInEditionZone())
					copy();
				else
					cut();
				return;
			}
			else if(e->matches(QKeySequence::Copy) ){
				e->accept();
				copy();
			}
			else {
				QTextEdit::keyPressEvent( e );
				return;
			}
		}
	}

	// Check whether this key event is relevant for search modes. If so,
	// we don't let any other key handling occur.
	const bool eventHandled = handleSearchKeyEvent(e);
	if(eventHandled)
		return;

	// control is pressed
	if((e->modifiers() & Qt::ControlModifier))
	{
		if(e->key() == Qt::Key_C)
		{
			if(isSelectionInEditionZone())
			{
				copy();
				return;
			}
		}

		else if(e->key() == Qt::Key_U)
		{
			if(isSelectionInEditionZone())
			{
				// Clear the current selection if any
				clearCurrentLine();
				setPrompt(prompt, true);
				return;
			}
		}
	}
	else {
		switch (e->key()) {
		case Qt::Key_Tab:
			if(isSelectionInEditionZone())
				handleTabKeyPress();
			return;
		case Qt::Key_Escape:
			// Clear the current selection if any
			clearCurrentLine();
			setPrompt(prompt, true);
			break;
		case Qt::Key_Enter:
		case Qt::Key_Return:
			if (isSelectionInEditionZone())
				handleReturnKeyPress();
			// ignore return key
			return;
		case Qt::Key_Backspace:
			if (textCursor().position()<=promptPosition || !isSelectionInEditionZone())
				return; // ignore backspace
			break;
		case Qt::Key_Home:
			setHome();
			return;
		case Qt::Key_Down:
			if (textCursor().hasSelection())
				break; // if we have selected text -> default behaviour

			// otherwise get cursor back into edit zone and handle history
			if(!isInEditionZone())
				moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
			handleDownKeyPress();
			return;
		case Qt::Key_Up:
			if (textCursor().hasSelection())
				break; // if we have selected text -> default behaviour
			// otherwise get cursor back into edit zone and handle history
			if ( !isInEditionZone() )
				moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
			handleUpKeyPress();
			return;
		//Default behaviour
		case Qt::Key_Left:
		case Qt::Key_End: // END and RIGHT will act the same as the ENTER key in search mode
		case Qt::Key_Right:
		case Qt::Key_Shift:
			break;

		default:
			historyIndex = history.size(); // reset history pointer
			setCursorToEditZone();
		} //end of switch
	} //end of else : no control pressed

	QTextEdit::keyPressEvent( e );
}

//Get the current command
QString QConsole::getCurrentCommand()
{
	QTextCursor cursor = textCursor();    //Get the current command: we just remove the prompt
	QString cmd = cursor.block().text();
	cmd.remove(prompt);
	return cmd;
}

//Replace current command with a new one
void QConsole::replaceCurrentCommand(const QString &newCommand)
{
	QTextCursor cursor = textCursor();
	cursor.movePosition(QTextCursor::StartOfBlock);
	cursor.movePosition(QTextCursor::Right, QTextCursor::MoveAnchor, promptLength);
	cursor.movePosition(QTextCursor::EndOfBlock, QTextCursor::KeepAnchor);
	cursor.insertText(newCommand);
	ensureCursorVisible();
}

//default implementation: command always complete
bool QConsole::isCommandComplete(const QString &)
{
	return true;
}

//Tests whether the cursor is in th edition zone or not (after the prompt
//or in the next lines (in case of multi-line mode)
bool QConsole::isInEditionZone()
{
	return textCursor().position()>=promptPosition;
}


//Tests whether position (in parameter) is in the edition zone or not (after the prompt
//or in the next lines (in case of multi-line mode)
bool QConsole::isInEditionZone(int pos)
{
	return pos>=promptPosition;
}


//Tests whether the current selection is in th edition zone or not
bool QConsole::isSelectionInEditionZone()
{
	if(textCursor().selectionStart() < promptPosition ||
	   textCursor().selectionEnd() < promptPosition)
		return false;
	return true;
}


//Basically, puts the command into the history list
//And emits a signal (should be called by reimplementations)
QString QConsole::interpretCommand(const QString &command, int *res)
{
	//update the history and its index
	QString modifiedCommand = command;
	modifiedCommand.replace("\n", "\\n");

	// append command to history, if it is not already the recent one in the history
	if(history.size()==0 || history.last()!=modifiedCommand)
		history.append(modifiedCommand);

	historyIndex = history.size();
	//emit the commandExecuted signal
	Q_EMIT commandExecuted(modifiedCommand);
	return "";
}

//execCommand(QString) executes the command and displays back its result
bool QConsole::execCommand(const QString &command, bool writeCommand,
                           bool showPrompt, QString *result)
{
	//Display the prompt with the command first
	if (writeCommand)
	{
		if (getCurrentCommand() != "")
		{
			append("");
			displayPrompt();
		}
		textCursor().insertText(command);
	}

	int res = 0;
	if(!command.isEmpty()) {
		//execute the command and get back its text result and its return value
		QString strRes = interpretCommand(command, &res);
		//According to the return value, display the result either in red or in blue
		setFormat(false,false);
		if (res == 0)
			setTextColor(outColor_);
		else
			setTextColor(errColor_);

		if(result){
			*result = strRes;
		}
		if (!(strRes.isEmpty() || strRes.endsWith("\n")))
			strRes.append("\n");
		append(strRes);
	} else {
		append("");
	}
	moveCursor(QTextCursor::End);
	//Display the prompt again
	if (showPrompt)
		displayPrompt();
	return !res;
}

void QConsole::setFormat(QColor color, bool bold, bool italic)
{
	setTextColor(color);
	setFormat(bold,italic);
}

void QConsole::setFormat(bool bold, bool italic)
{
	setFontWeight(bold ? QFont::Bold : QFont::Normal);
	setFontItalic(italic);
}

void QConsole::println(const QString& s)
{
	append(s);
}

void QConsole::println(const std::string& s)
{
	println(QString::fromStdString(s));
}

void QConsole::println(const char* s)
{
	println(QString(s));
}

//Change paste behaviour
void QConsole::insertFromMimeData(const QMimeData *source)
{
	if (isSelectionInEditionZone()) {
		if (currentSearchMode != SearchMode::SEARCH_NONE) {
			if ((source != NULL) && source->hasText()) {
				// fake key input from the pasted text
				QApplication::postEvent(this, new QKeyEvent(QEvent::KeyPress, 0, Qt::NoModifier, source->text()));
			}
		} else {
			textCursor().insertFragment(QTextDocumentFragment::fromPlainText(source->text()));
		}
	}
}

void QConsole::mousePressEvent(QMouseEvent* event)
{
	// Workaround to make sure that TextEdit base implementation does not
	// cut the text that is dragged into another text window or application.
	setTextInteractionFlags(Qt::TextSelectableByMouse);
	QTextEdit::mousePressEvent(event);
}

void QConsole::mouseReleaseEvent(QMouseEvent* event)
{
	setTextInteractionFlags(Qt::TextEditorInteraction);
	QTextEdit::mouseReleaseEvent(event);
}


void QConsole::dragEnterEvent(QDragEnterEvent *e)
{
}

void QConsole::dragLeaveEvent(QDragLeaveEvent *e)
{
}

void QConsole::dragMoveEvent(QDragMoveEvent *e)
{
}

void QConsole::dropEvent(QDropEvent *e)
{
}

#if 0

//Redefinition of the dropEvent to have a copy paste
//instead of a cut paste when copying out of the
//editable zone
void QConsole::dropEvent ( QDropEvent * event)
{
	if(!isInEditionZone())
	{
		//Execute un drop a drop at the old position
		//if the drag started out of the editable zone
		QTextCursor cur = textCursor();
		cur.setPosition(oldPosition);
		setTextCursor(cur);
	}
	//Execute a normal drop
	QTextEdit::dropEvent(event);
}
void QConsole::dragMoveEvent( QDragMoveEvent * event)
{
	//Get a cursor for the actual mouse position
	QTextCursor cur = textCursor();
	cur.setPosition(cursorForPosition(event->pos()).position());

	if(!isInEditionZone(cursorForPosition(event->pos()).position()))
	{
		//Ignore the event if out of the editable zone
		event->ignore(cursorRect(cur));
	}
	else
	{
		//Accept the event if out of the editable zone
		event->accept(cursorRect(cur));
	}
}
#endif

void QConsole::contextMenuEvent ( QContextMenuEvent * event)
{
	QMenu *menu = new QMenu(this);

	QAction *undo = new QAction(tr("Undo"), this);
	undo->setShortcut(tr("Ctrl+Z"));
	QAction *redo = new QAction(tr("Redo"), this);
	redo->setShortcut(tr("Ctrl+Y"));
	QAction *cut = new QAction(tr("Cut"), this);
	cut->setShortcut(tr("Ctrl+X"));
	QAction *copy = new QAction(tr("Copy"), this);
	copy->setShortcut(tr("Ctrl+Ins"));
	QAction *paste = new QAction(tr("Paste"), this);
	paste->setShortcut(tr("Ctrl+V"));
	QAction *selectAll = new QAction(tr("Select All"), this);
	selectAll->setShortcut(tr("Ctrl+A"));

	menu->addAction(undo);
	menu->addAction(redo);
	menu->addSeparator();
	menu->addAction(cut);
	menu->addAction(copy);
	menu->addAction(paste);
	menu->addSeparator();
	menu->addAction(selectAll);

	connect(undo, SIGNAL(triggered()), this, SLOT(undo()));
	connect(redo, SIGNAL(triggered()), this, SLOT(redo()));
	connect(cut, SIGNAL(triggered()), this, SLOT(cut()));
	connect(copy, SIGNAL(triggered()), this, SLOT(copy()));
	connect(paste, SIGNAL(triggered()), this, SLOT(paste()));
	connect(selectAll, SIGNAL(triggered()), this, SLOT(selectAll()));


	menu->exec(event->globalPos());

	delete menu;
}

void QConsole::cut()
{
	//Cut only in the editing zone,
	//perfom a copy otherwise
	if(isSelectionInEditionZone())
	{
		QTextEdit::cut();
		return;
	}

	QTextEdit::copy();
}

///////////////////////////////////////////////////////////////////////////////

const QConsole::SearchMode& QConsole::searchModeCommand(QKeyEvent* e)
{
	if((e->modifiers() & Qt::ControlModifier) && (e->key() == Qt::Key_R))
		return SearchMode::SEARCH_HISTORY;

	return SearchMode::SEARCH_NONE;
}

void QConsole::enterSearchMode(const SearchMode& mode)
{
	currentSearchMode = mode;
}

QString QConsole::getSearchModeDescriptor()
{
	return "history-search";
}

QStringList QConsole::getSearchList()
{
	return history;
}

std::multimap<QString, QString> QConsole::getExtendedSearchList()
{
	// empty list, we only use the "normal" search list from above in this class
	return std::multimap<QString, QString>();
}

bool QConsole::handleSearchKeyEvent(QKeyEvent* e)
{
	// Check whether we should trigger search mode
	const SearchMode& mode = searchModeCommand(e);
	if ((mode != SearchMode::SEARCH_NONE) &&
	    ((currentSearchMode == SearchMode::SEARCH_NONE) || (currentSearchMode == mode)))
	{
		setCursorToEditZone();
		handleSearch(mode);
		return true;
	}

	// We only catch certain keys in search mode
	else if(currentSearchMode != SearchMode::SEARCH_NONE)
	{
		// Certain key combinations only work with CTRL
		if((e->modifiers() & Qt::ControlModifier))
		{
			// CTRL+U will discard the selection
			if(e->key() == Qt::Key_U)
			{
				setCursorToEditZone();
				handleSearchEscapeKey();
				return true;
			}
		}

		// Non-CTRL modified keys
		else
		{
			switch(e->key())
			{
				// Catch the "accept search selection" case,
				// which can also be triggered by return, enter, tab and right arrow
				case Qt::Key_Tab:
				case Qt::Key_Enter:
				case Qt::Key_Return:
				case Qt::Key_Right:
				{
					setCursorToEditZone();
					handleSearchReturnKey();
					return true;
				}

				// Catch the "discard search mode"
				case Qt::Key_Escape:
				{
					setCursorToEditZone();
					handleSearchEscapeKey();
					return true;
				}

				// Otherwise pass it through to our default handler, which will
				// discard all events that are not suitable.
				default:
				{
					setCursorToEditZone();
					handleSearchInput(e);
					return true; // don't let QTextEdit handle the key press event
				}
			}
		}
	}

	// Event wasn't handled
	return false;
}

void QConsole::overwriteCommand(QString command)
{
	clearCurrentLine();
	setTextColor(cmdColor_);
	setFormat(false,false);

	// Set the new text
	QTextCursor cur = textCursor();
	cur.insertText(command);
	cur.movePosition(QTextCursor::EndOfBlock);
	setTextCursor(cur);
}

void QConsole::clearCurrentLine()
{
	// Remove the current line
	QTextCursor cur = textCursor();
	cur.movePosition(QTextCursor::StartOfBlock);
	cur.movePosition(QTextCursor::EndOfBlock, QTextCursor::KeepAnchor);
	cur.removeSelectedText();
}

void QConsole::setCursorToEditZone()
{
	if (textCursor().hasSelection() ) {
		if (!isSelectionInEditionZone())
			moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
	}
	else
	{ //no selection
		//when typing normal characters,
		//make sure the cursor is positionned in the
		//edition zone
		if ( !isInEditionZone() )
			moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
	}
}

void QConsole::resetSearch()
{
	currentSearchMode = SearchMode::SEARCH_NONE;
	searchMatchIndex = 0;
	searchMatches.clear();
	previousPrompt = "";
}

void QConsole::updateSearchMatches()
{
	searchMatches.clear();

	// Get current input and start searching. Everything that matches will be stored in the list
	if(searchInput.isEmpty())
		return;

	foreach(const auto& l, getSearchList())
	{
		if(l.contains(searchInput, Qt::CaseInsensitive))
			searchMatches.append(l);
	}
	foreach(const auto& l, getExtendedSearchList())
	{
		if(l.first.contains(searchInput, Qt::CaseInsensitive))
			searchMatches.append(l.second);
	}

}

void QConsole::updateSearchCursor(QString query, QString result, int index, int numResults)
{
	const QString queryAug = query.isEmpty() ? QString("") :
			QString(" \"%1\" [%2/%3]").arg(query).arg(numResults > 0 ? index + 1 : 0).arg(numResults);
	overwriteCommand(QString("(%1)%2: %3").arg(getSearchModeDescriptor()).arg(queryAug).arg(result));
}

void QConsole::handleSearch(const SearchMode& mode)
{
	// We have to save the previous prompt and enter our own
	if(currentSearchMode == SearchMode::SEARCH_NONE)
	{
		enterSearchMode(mode);

		previousPrompt = prompt;
		searchInput = getCurrentCommand();
		setUndoRedoEnabled(false);
		updateSearchCursor(searchInput, "", 0, 0);
	}

	// Do we have a search result list? If not, we need to create it before
	// we can start iterating through it.
	if(searchMatches.empty())
	{
		// Get current input and start searching. Everything that matches will be stored in the list
		updateSearchMatches();

		// Select the first one for now. If there are none, nothing will be displayed
		searchMatchIndex = 0;
	}

	// If we do have a history, we simply select the next one
	else
	{
		if((searchMatchIndex + 1) >= searchMatches.size())
			searchMatchIndex = 0;
		else
			++searchMatchIndex;
	}

	// Update our command with the current match result
	if(searchMatchIndex < searchMatches.size())
	{
		updateSearchCursor(searchInput, searchMatches[searchMatchIndex],
		                   searchMatchIndex, searchMatches.size());
	}
}

void QConsole::handleSearchInput(QKeyEvent* event)
{
	// Very simple, we only allow pure text input
	QString text = event->text();

	// shift and keypad modifiers won't harm the text input
	if(event->modifiers() & ~Qt::ShiftModifier & ~Qt::KeypadModifier)
		return;

	const int size = searchInput.size();
	if(event->key() == Qt::Key_Backspace)
	{
		if(size == 0)
			return;
		searchInput.remove(size - 1, 1);
	}
	else if(!text.isEmpty())
	{
		// Add the character to our input text
		searchInput.append(text);
	}
	else
	{
		return;
	}

	// Search with current input
	updateSearchMatches();
	searchMatchIndex = 0;

	if(searchMatchIndex >= searchMatches.size())
		updateSearchCursor(searchInput, "", 0, 0);
	else
		updateSearchCursor(searchInput, searchMatches[searchMatchIndex],
		                   searchMatchIndex, searchMatches.size());
}

void QConsole::handleSearchReturnKey()
{
	setUndoRedoEnabled(true);

	// Insert selected command
	overwriteCommand("");
	setPrompt(previousPrompt, true);

	if(searchMatchIndex < searchMatches.size())
		replaceCurrentCommand(searchMatches[searchMatchIndex]);

	resetSearch();
}

void QConsole::handleSearchEscapeKey()
{
	overwriteCommand("");
	setPrompt(previousPrompt, true);

	resetSearch();
}

///////////////////////////////////////////////////////////////////////////////

}
