/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LineListObject.C
 *    Implementation of LineListObject.h.
 *
 * @author Erik Einhorn
 * @date   2012/11/06
 */

#include <visualization/3d/LineListObject.h>

#include <OGRE/OgreBillboardChain.h>
#include <OGRE/OgreMaterialManager.h>
#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreTechnique.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

LineListObject::LineListObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent, int lineCountHint) :
		VisualizationObject(sceneManager,parent), mLineCountHint(lineCountHint)
{
	std::string materialName = "LineListObject"+toString(this)+"/Material";
	mMaterial = Ogre::MaterialManager::getSingleton().create(materialName, "MIRAMaterials");
	mMaterial->setReceiveShadows(false);
	mMaterial->getTechnique(0)->setLightingEnabled(false);

	mColor = Ogre::ColourValue::Black;
	mWidth = 0.1f;

	newChain();
	mCurChain = 0;
}

LineListObject::~LineListObject()
{
	foreach(Ogre::BillboardChain* c, mChains)
		mSceneManager->destroyBillboardChain(c);
	if (!mMaterial.isNull())
		Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());
}

///////////////////////////////////////////////////////////////////////////////

void LineListObject::setColor(const Ogre::ColourValue& color)
{
	mColor = color;
	int idx=0;
	foreach(Ogre::BillboardChain* c, mChains)
	{
		for(int j=0; j<mLinesInChain[idx]; ++j)
		{
			for(size_t i=0; i<2; ++i)
			{
				Ogre::BillboardChain::Element e = c->getChainElement(j, i);
				e.colour = mColor;
				c->updateChainElement(0,i,e);
			}
		}
		++idx;
	}
}


void LineListObject::enableAlphaBlending(bool useAlpha)
{
	if(useAlpha)
		mMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
	else
		mMaterial->setSceneBlending(Ogre::SBT_REPLACE);
}

///////////////////////////////////////////////////////////////////////////////

void LineListObject::setLineWidth(float width)
{
	mWidth = width;

	int idx=0;
	foreach(Ogre::BillboardChain* c, mChains)
	{
		for(int j=0; j<mLinesInChain[idx]; ++j)
		{
			for(size_t i=0; i<2; ++i)
			{
				Ogre::BillboardChain::Element e = c->getChainElement(j, i);
				e.width = mWidth;
				c->updateChainElement(j,i,e);
			}
		}

		++idx;
	}
}

void LineListObject::clear()
{
	foreach(Ogre::BillboardChain* c, mChains)
		c->clearAllChains();

	foreach(int& c, mLinesInChain)
		c=0;

	mCurChain = 0;
}


void LineListObject::add(const Ogre::Vector3& p1,const Ogre::Vector3& p2)
{
	add(p1, p2, mColor);
}

void LineListObject::add(const Ogre::Vector3& p1,const Ogre::Vector3& p2, const Ogre::ColourValue& color)
{
	if(mLinesInChain[mCurChain]+1>=(int)mChains[mCurChain]->getNumberOfChains())
		nextChain();

	int line = mLinesInChain[mCurChain];

	// add both points
	Ogre::BillboardChain::Element e;
	e.position = p1;
	e.width = mWidth;
	e.colour = color;
	mChains[mCurChain]->addChainElement(line, e);

	e.position = p2;
	mChains[mCurChain]->addChainElement(line, e);

	++mLinesInChain[mCurChain];
}

void LineListObject::nextChain()
{
	++mCurChain;
	if(mCurChain>=(int)mChains.size())
		newChain();
}

void LineListObject::newChain()
{
	Ogre::BillboardChain* chain = mSceneManager->createBillboardChain();
	chain->setMaterialName(mMaterial->getName());
	chain->setMaxChainElements(2);
	chain->setNumberOfChains(mLineCountHint);
	mNode->attachObject(chain);

	mChains.push_back(chain);
	mLinesInChain.push_back(0);

	mCurChain = (int)mChains.size()-1;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

