/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AxesObject.C
 *    Implementation of AxesObject.
 *
 * @author Tim Langner
 * @date   2011/01/04
 */

#include <visualization/3d/AxesObject.h>
#include <visualization/3d/ArrowObject.h>
#include <widgets/OgreUtils.h>

#include <OGRE/OgreSceneManager.h>
#include <OGRE/OgreSceneNode.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

AxesObject::AxesObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent) :
	VisualizationObject(sceneManager, parent),
	mVisible(true),
	mShowXAxis(true),
	mShowYAxis(true),
	mShowZAxis(true),
	mShowArrows(true),
	mAxisLength(1.0f),
	mAxisRadius(0.025f),
	mXAxis(NULL),
	mYAxis(NULL),
	mZAxis(NULL)
{
	mXAxisColor = Ogre::ColourValue(1.0f, 0.0f, 0.0f, 1.0f);
	mYAxisColor = Ogre::ColourValue(0.0f, 1.0f, 0.0f, 1.0f);
	mZAxisColor = Ogre::ColourValue(0.0f, 0.0f, 1.0f, 1.0f);

	mXAxis = new ArrowObject(mSceneManager, mNode);
	mXAxis->setOrientation(Ogre::Quaternion(Ogre::Degree(-90), Ogre::Vector3::UNIT_Z));

	mYAxis = new ArrowObject(mSceneManager, mNode);

	mZAxis = new ArrowObject(mSceneManager, mNode);
	mZAxis->setOrientation(Ogre::Quaternion(Ogre::Degree(90), Ogre::Vector3::UNIT_X));

	setAlpha(1.0f);

	updateVisibility();
	updateScales();
}

AxesObject::~AxesObject()
{
	delete mXAxis;
	delete mYAxis;
	delete mZAxis;
}

void AxesObject::setLength(float length)
{
	mAxisLength = length;
	updateScales();
}

void AxesObject::setRadius(float radius)
{
	mAxisRadius = radius;
	updateScales();
}

void AxesObject::setColor(const Ogre::ColourValue& uniform)
{
	mXAxisColor = mYAxisColor = mZAxisColor = uniform;
	updateColor();
}

void AxesObject::setColor(const Ogre::ColourValue& xaxis, const Ogre::ColourValue& yaxis, const Ogre::ColourValue& zaxis)
{
	mXAxisColor = xaxis;
	mYAxisColor = yaxis;
	mZAxisColor = zaxis;
	updateColor();
}

void AxesObject::updateScales()
{
	mXAxis->setRadius(mAxisRadius);
	mXAxis->setLength(mAxisLength);
	mYAxis->setRadius(mAxisRadius);
	mYAxis->setLength(mAxisLength);
	mZAxis->setRadius(mAxisRadius);
	mZAxis->setLength(mAxisLength);
}

void AxesObject::updateColor()
{
	mXAxis->setColor(mXAxisColor);
	mYAxis->setColor(mYAxisColor);
	mZAxis->setColor(mZAxisColor);
}

void AxesObject::showXAxis(bool show)
{
	mShowXAxis = show;
	updateVisibility();
}

void AxesObject::showYAxis(bool show)
{
	mShowYAxis = show;
	updateVisibility();
}

void AxesObject::showZAxis(bool show)
{
	mShowZAxis = show;
	updateVisibility();
}

void AxesObject::showArrows(bool show)
{
	mShowArrows = show;
	updateVisibility();
}

void AxesObject::setAlpha(float alpha)
{
	mXAxisColor.a = mYAxisColor.a = mZAxisColor.a = alpha;
	updateColor();
}

void AxesObject::setVisible(bool visible, bool cascade)
{
	VisualizationObject::setVisible(visible, cascade);
	mVisible = visible;
	updateVisibility();
}

void AxesObject::updateVisibility()
{
	mXAxis->setVisible(mVisible && mShowXAxis);
	mXAxis->showHead(mVisible && mShowXAxis && mShowArrows);

	mYAxis->setVisible(mVisible && mShowYAxis);
	mYAxis->showHead(mVisible && mShowYAxis && mShowArrows);

	mZAxis->setVisible(mVisible && mShowZAxis);
	mZAxis->showHead(mVisible && mShowZAxis && mShowArrows);
}

///////////////////////////////////////////////////////////////////////////////

}
