/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationTextBasic.h
 *    A base class for simple text visualizations.
 *
 * @author Tim Langner
 * @date   2011/07/04
 */

#ifndef _MIRA_VISUALIZATIONTEXTBASIC_H_
#define _MIRA_VISUALIZATIONTEXTBASIC_H_

#include <serialization/SetterNotify.h>
#include <visualization/VisualizationText.h>

#include <widgets/MultiInputDialog.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class VisualizationTextBasic : public VisualizationText
{
public:

	VisualizationTextBasic(const std::string& channelName) :
		mName(channelName),
		mFormatted(true),
		mItem(NULL)
	{
		mChannel.setDataChangedCallback(boost::bind(&VisualizationTextBasic::dataChanged, this, _1));
		mChannel.setChannelChangedCallback(boost::bind(&VisualizationTextBasic::channelChanged, this));
	}

	virtual ~VisualizationTextBasic()
	{
		if (getSite() && mItem)
			getSite()->destroyItem(mItem);
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, VisualizationText);
		channelProperty(r, mName, mChannel, "");
		r.property("Formatted", mFormatted,
		           setterNotify(mFormatted, &VisualizationTextBasic::redraw, this),
		           "Formatted json output", true);
	}

	virtual void setupScene(IVisualizationTextSite* site)
	{
		mItem = site->addItem();
		mItem->setName(mChannel.getID());
		mItem->setSize(mSize);
		setupScene(mItem);
	}

	virtual void setSize(const Size2i& size)
	{
		VisualizationText::setSize(size);
		if (mItem)
			mItem->setSize(size);
	}

	virtual Size2i getSize() const
	{
		if (mItem)
			return mItem->getSize();
		return mSize;
	}

	virtual void setupScene(IVisualizationTextItem* item) {}

	virtual DataConnection getDataConnection() { return DataConnection(mChannel); }

protected:

	virtual void channelChanged()
	{
		try {
			this->setName(mChannel.getID());
		}
		catch (XRuntime&) {}
		if (!mItem)
			return;
		mItem->setName(mChannel.getID());
		mItem->setText("");
	}

	virtual void dataChanged(ChannelRead<T> data) {}

	virtual std::string defaultText() { return ""; }

	virtual void onEdit()
	{
		if (mChannel.isValid())
		{
			MultiInputDialog dialog("Add new channel data", NULL);
			dialog.addInputField("time", "Timestamp", toString(Time::now()).c_str());
			dialog.addInputField("frame", "FrameID", "");
			dialog.addInputField("sequence", "SequenceID", "0");
			if (mLastValue)
				dialog.addInputField("value", "Value", json::write(*mLastValue, true, mPrecision).c_str(), true);
			else
				dialog.addInputField("value", "Value", defaultText().c_str(), true);
			if (dialog.exec() == 0)
				return;

			json::Value value;
			json::read(dialog.getInput("value").toStdString(), value);
			writeJSON(boost::posix_time::time_from_string(dialog.getInput("time").toStdString()),
			          dialog.getInput("frame").toStdString(),
			          fromString<int>(dialog.getInput("sequence").toStdString()),
			          value);
		}
	}

	virtual void redraw()
	{
		if (mLastValue)
			mItem->setText(json::write(*mLastValue, mFormatted, mPrecision));
	}
	
	virtual void writeJSON(const Time& timestamp, const std::string& frameID,
	                       int sequenceID, const json::Value& value)
	{
		ConcreteChannel<void>* channel = MIRA_FW.getChannelManager().getConcreteChannel<void>(this->mChannel.getChannel().getID());
		ChannelWrite<void> cw = channel->write();
		
		cw->timestamp = timestamp;
		cw->frameID = frameID;
		cw->sequenceID = sequenceID;

		try {
			JSONDeserializer jds(value, JSONDeserializer::ACCEPT_STRING_MAP_AS_OBJECT);
			cw.writeJSON(jds);
		}
		catch (XNotImplemented&) {
			cw.writeJSON(value);
		}
	}

	virtual IVisualizationTextItem* getItem() { return mItem; }

protected:
	std::string mName;
	ChannelProperty<T> mChannel;
	bool mFormatted;
	IVisualizationTextItem* mItem;

	boost::optional<json::Value> mLastValue;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
