/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MonitorConnectionTestUnits.C
 *    Utility units for monitor connection test
 *
 * @author Christof Schröter
 * @date   2022/03/09
 */

#include <fw/Unit.h>

namespace mira {
namespace fw {
namespace test {
namespace monitor {

///////////////////////////////////////////////////////////////////////////////

// <unit id="Observed" class="mira::fw::test::monitor::ObservedUnit" />

class ObservedUnit : public MicroUnit
{
MIRA_OBJECT(ObservedUnit)

public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.method("gotService", &ObservedUnit::gotService, this,
		         "Found the IMonitorTest service?");
		r.method("foundAuthority", &ObservedUnit::foundAuthority, this,
		         "Found the Monitor authority?");
	}

protected:

	virtual void initialize()
	{
		mReceivedChannelUpdate = publish<int>("ReceivedUpdate");
		mReceivedChannelUpdate.post(0);
		subscribe<int>("Subscribed", &ObservedUnit::onUpdate, this);

		std::string service = waitForServiceInterface("IMonitorTest", Duration::seconds(5));
		mGotService = !service.empty();
		mFoundAuthority = MIRA_FW.getAuthorityManager().hasAuthority("/Publisher");

		publishTransform<Pose2>("ChildFrame", Pose2(0.f, 1.f, 2.f), Time::now());

		auto channel = publish<std::string>("Link");
		FrameworkTransformerNode* node = NULL;
		while (!node)
			node = MIRA_FW.getTransformer()->getNode("/ChildFrame");
		channel.post(node->getParent()->getID());
		publishService(*this);
	}

	void onUpdate(ChannelRead<int> read)
	{
		mReceivedChannelUpdate.post(read->value());
	}

	bool foundAuthority() { return mFoundAuthority; }
	bool gotService() { return mGotService; }

private:

	Channel<int> mReceivedChannelUpdate;

	bool mGotService;
	bool mFoundAuthority;
};

///////////////////////////////////////////////////////////////////////////////

// <unit id="Publisher" class="mira::fw::test::monitor::PublisherUnit" />

class PublisherUnit : public MicroUnit
{
MIRA_OBJECT(PublisherUnit)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IMonitorTest");
	}

protected:

	virtual void initialize()
	{
		auto channel = publish<int>("Published");
		channel.post(1);
		publishService(*this);
	}
};

///////////////////////////////////////////////////////////////////////////////

}}}} /// namespaces

MIRA_CLASS_SERIALIZATION(mira::fw::test::monitor::ObservedUnit, mira::MicroUnit);
MIRA_CLASS_SERIALIZATION(mira::fw::test::monitor::PublisherUnit, mira::MicroUnit);
