/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MonitorConnectionTest.C
 *    Test Monitoring Framework Connection
 *
 * @author Christof Schröter
 * @date   2022/03/09
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;

// Testing Connection: standard, monitoring

struct TestAuthority : public Authority
{
	TestAuthority() : Authority("/TestAuthority") {}
};

BOOST_AUTO_TEST_CASE(MonitorConnectionTest)
{
	const char* argv[] = {"MonitorConnectionTest",
	                      "${find MonitorConnectionTest.xml}",
	                      "-k1357", "-k1358@monitor"};
	Framework fw(4,(char**)argv);
	fw.load();
	fw.start();

	TestAuthority tester;
	BOOST_REQUIRE(tester.waitForService("/1357/Observed", Duration::seconds(10)));
	BOOST_REQUIRE(tester.waitForService("/1358/Observed", Duration::seconds(10)));

	auto channel1357 = tester.subscribe<int>("/1357/ReceivedUpdate");
	auto channel1358 = tester.subscribe<int>("/1358/ReceivedUpdate");
	auto link1357 = tester.subscribe<std::string>("/1357/Link");
	auto link1358 = tester.subscribe<std::string>("/1358/Link");

	Pose2 pose;

	// with standard connection:
	// - remote units see local services and authorities, and vice versa
	BOOST_CHECK_EQUAL(tester.callService<bool>("/1357/Observed", "gotService").get(), true);
	BOOST_CHECK_EQUAL(tester.callService<bool>("/1357/Observed", "foundAuthority").get(), true);
	// - remote units get updates on subscribed channels from local publishers, and vice versa
	BOOST_CHECK_EQUAL(channel1357.read()->value(), 1);
	// - remote units can publish links+transformations to local, and vice versa
	pose = tester.getTransform<Pose2>("/1357/ChildFrame", "/GlobalFrame");
	BOOST_CHECK_EQUAL(pose.x(), 0.f);
	BOOST_CHECK_EQUAL(pose.y(), 1.f);
	BOOST_CHECK_EQUAL(pose.phi(), 2.f);
	BOOST_CHECK_EQUAL(link1357.read()->value(), "/GlobalFrame");

	// with monitor connection:
	// - local units see remote  services and authorities
	// - remote units do NOT see local services and authorities
	BOOST_CHECK_EQUAL(tester.callService<bool>("/1358/Observed", "gotService").get(), false);
	BOOST_CHECK_EQUAL(tester.callService<bool>("/1358/Observed", "foundAuthority").get(), false);
	// - local units get updates on subscribed channels from remote publishers
	// - remote units do NOT get updates on subscribed channels from local publishers
	BOOST_CHECK_EQUAL(channel1358.read()->value(), 0);
	// - remote units can publish links+transformations to local
	// - local units can publish links to remote
	// (- local units can NOT publish transformations to remote, because remote does not receive channel updates)
	pose = tester.getTransform<Pose2>("/1358/ChildFrame", "/GlobalFrame");
	BOOST_CHECK_EQUAL(pose.x(), 0.f);
	BOOST_CHECK_EQUAL(pose.y(), 1.f);
	BOOST_CHECK_EQUAL(pose.phi(), 2.f);
	BOOST_CHECK_EQUAL(link1358.read()->value(), "/GlobalFrame");
}
