/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelPromoteByTypenameTest.C
 *    Test for channel promotion by runtime typename.
 *
 * @author Christof Schröter
 * @date   2018/08/29
 */

#include <boost/test/unit_test.hpp>

#include <fw/Framework.h>

using namespace mira;

BOOST_AUTO_TEST_CASE(ChannelPromoteByTypenameTestLocal)
{
	// create the framework with publisher from config file
	const char* argv[] = {"ChannelPromoteByTypenameTest",
	                      "${find ChannelPromoteByTypenameTest.xml}",
	                      "-v", "binary=mira"};
	Framework fw(4,(char**)argv);
	fw.load();

	// now start processing
	fw.start();

	// give it a few seconds to start and connect all channels
	MIRA_SLEEP(5000);

	ChannelManager& manager = fw.getChannelManager();

	BOOST_CHECK_EQUAL(manager.getChannels().size(), 53);

	// 26 locally published, 27 locally subscribed
	BOOST_CHECK_EQUAL(manager.getChannelList(true, false).size(), 26);
	BOOST_CHECK_EQUAL(manager.getChannelList(false, true).size(), 27);

	// remotely published channels have an 'internal' publisher locally
	std::map<std::string, Typename> channels = manager.getPublishedChannels(true);
	BOOST_CHECK_EQUAL(channels.size(), 52);

	BOOST_CHECK_EQUAL(channels["/1/DummyVector2"],
	                           "std::vector<boost::shared_ptr<mira::fw::channel_promote_test::ns1::DummyBase>,"
	                           "std::allocator<boost::shared_ptr<mira::fw::channel_promote_test::ns1::DummyBase>>>");
	BOOST_CHECK_EQUAL(channels["/1/DummyMap2"],
	                           "std::map<int,boost::shared_ptr<mira::fw::channel_promote_test::ns1::DummyBase>,"
	                           "std::less<int>,std::allocator<std::pair<int const,"
	                           "boost::shared_ptr<mira::fw::channel_promote_test::ns1::DummyBase>>>>");
	BOOST_CHECK_EQUAL(channels["/1/Int2"], "int");
	BOOST_CHECK_EQUAL(channels["/1/Float2"], "float");
	BOOST_CHECK_EQUAL(channels["/1/Char2"], "char");
	BOOST_CHECK_EQUAL(channels["/1/Double2"], "double");
	BOOST_CHECK_EQUAL(channels["/1/String2"], "mira::fw::channel_promote_test::String");

	BOOST_CHECK_EQUAL(channels["/2/DummyVector2"],
	                           "std::vector<boost::shared_ptr<mira::fw::channel_promote_test::ns2::DummyBase>,"
	                           "std::allocator<boost::shared_ptr<mira::fw::channel_promote_test::ns2::DummyBase>>>");
	BOOST_CHECK_EQUAL(channels["/2/DummyMap2"],
	                           "std::map<int,boost::shared_ptr<mira::fw::channel_promote_test::ns2::DummyBase>,"
	                           "std::less<int>,std::allocator<std::pair<int const,"
	                           "boost::shared_ptr<mira::fw::channel_promote_test::ns2::DummyBase>>>>");
	BOOST_CHECK_EQUAL(channels["/2/Int2"], "int");
	BOOST_CHECK_EQUAL(channels["/2/Float2"], "float");
	BOOST_CHECK_EQUAL(channels["/2/Char2"], "char");
	BOOST_CHECK_EQUAL(channels["/2/Bool2"], "bool");
	BOOST_CHECK_EQUAL(channels["/2/String2"], "mira::fw::channel_promote_test::String");

	BOOST_CHECK_EQUAL(channels["/3/Time2"], "mira::Time");
	BOOST_CHECK_EQUAL(channels["/3/Int2"], "int");
	BOOST_CHECK_EQUAL(channels["/3/Float2"], "float");
	BOOST_CHECK_EQUAL(channels["/3/Char2"], "char");
	BOOST_CHECK_EQUAL(channels["/3/Double2"], "double");
	BOOST_CHECK_EQUAL(channels["/3/String2"], "mira::fw::channel_promote_test::String");

	BOOST_CHECK_EQUAL(channels["/4/Time2"], "mira::Time");
	BOOST_CHECK_EQUAL(channels["/4/Int2"], "int");
	BOOST_CHECK_EQUAL(channels["/4/Float2"], "float");
	BOOST_CHECK_EQUAL(channels["/4/Char2"], "char");
	BOOST_CHECK_EQUAL(channels["/4/Double2"], "double");
	BOOST_CHECK_EQUAL(channels["/4/String2"], "mira::fw::channel_promote_test::String");

	// all channels are untyped as long as nothing is posted
	foreach (auto p, channels)
		BOOST_CHECK_EQUAL(manager.getTypeId(p.first), -1);

	manager.publish<bool>("/Signal2", "", false, "bool")->write().finish();  // just an arbitrary write access to signal publishers
	// local publishers post data now, promoting their published channels
	MIRA_SLEEP(2000);

	BOOST_CHECK(manager.getTypeId("/1/DummyVector2") >= 0);
	BOOST_CHECK(manager.getTypeId("/1/DummyMap2") >= 0);
	BOOST_CHECK(manager.getTypeId("/1/Int2") >= 0);
	BOOST_CHECK(manager.getTypeId("/1/Float2") >= 0);
	BOOST_CHECK(manager.getTypeId("/1/Char2") >= 0);
	BOOST_CHECK(manager.getTypeId("/1/Double2") >= 0);

	BOOST_CHECK_EQUAL(manager.getTypeId("/1/String"), -1); // string channel type is not registered by anyone

	BOOST_CHECK(manager.getTypeId("/2/DummyVector2") >= 0);
	BOOST_CHECK(manager.getTypeId("/2/DummyMap2") >= 0);
	BOOST_CHECK(manager.getTypeId("/2/Int2") >= 0);
	BOOST_CHECK(manager.getTypeId("/2/Float2") >= 0);
	BOOST_CHECK(manager.getTypeId("/2/Char2") >= 0);
	BOOST_CHECK(manager.getTypeId("/2/Bool2") >= 0);

	BOOST_CHECK_EQUAL(manager.getTypeId("/2/String"), -1); // string channel type is not registered by anyone

	BOOST_CHECK(manager.getTypeId("/3/Time2") >= 0);
	BOOST_CHECK(manager.getTypeId("/3/Int2") >= 0);
	BOOST_CHECK(manager.getTypeId("/3/Float2") >= 0);
	BOOST_CHECK(manager.getTypeId("/3/Char2") >= 0);
	BOOST_CHECK(manager.getTypeId("/3/Double2") >= 0);

	BOOST_CHECK_EQUAL(manager.getTypeId("/3/String"), -1); // string channel type is not registered by anyone

	BOOST_CHECK(manager.getTypeId("/4/Time2") >= 0);
	BOOST_CHECK(manager.getTypeId("/4/Int2") >= 0);
	BOOST_CHECK(manager.getTypeId("/4/Float2") >= 0);
	BOOST_CHECK(manager.getTypeId("/4/Char2") >= 0);
	BOOST_CHECK(manager.getTypeId("/4/Double2") >= 0);

	BOOST_CHECK_EQUAL(manager.getTypeId("/4/String"), -1); // string channel type is not registered by anyone

	ClassFactory& factory = ClassFactory::instance();
	auto registeredTypes = TypedChannelPromoterBase::CLASS().getDerivedClasses();
	int count = 0;
	foreach (auto c, registeredTypes) {
		if (strstr(c.first.c_str(), "framework_tests_fw") &&
		    strstr(c.first.c_str(), "_ChannelPromoteByTypenameTestUnits"))
			++count;
	}
	// called MIRA_REGISTER_CHANNELTYPE 14 times --> created 14 promoter classes for 8 distinct channel types
	BOOST_CHECK_EQUAL(count, 14);
}
