/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

#include <serialization/adapters/std/vector>

#include <fw/Unit.h>

#include "../PerfStats.h"

using namespace std;
using namespace mira;


#define D 100

class Vector
{
public:
	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("rdtsc", rdtsc, "");
		r.member("Values",     values,     "The actual vector");
	}

public:
	uint64 rdtsc;
	vector<uint8> values;
};


class BenchmarkPublisher : public Unit
{
	MIRA_OBJECT(BenchmarkPublisher)
public:


	BenchmarkPublisher() :
		Unit(Duration::milliseconds(D)),
		count(0),total(0)
	{
		// skip first 10 measurements
		first=10;
	}

	~BenchmarkPublisher()
	{
		if (count > 0)
			std::cout << "Publishing Average: " << total/count << std::endl;
	}

	virtual void initialize()
	{
		MIRA_SLEEP(2000); // make sure all libs are loaded and mem stats are stable

		memStart = getMemoryUsage();
		std::cout << "Publisher Start1: " << memStart << std::endl;

		mChannel = publish<Vector>("Data");
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		MIRA_REFLECT_BASE(r, Unit);
		r.member("N", N, "DataSize");
	}

	virtual void process(const Timer& timer)
	{
		uint64 memDiff1 = getMemoryUsage() - memStart;
		std::cout << "Publisher Memory1: " << memDiff1 << std::endl;

		uint64 start = getCycleCount();
		{
			ChannelWrite<Vector> data = mChannel.write();
			data->rdtsc = start;
			data->values.resize(N*1024);

		}

		// skip first measurements
		if(first>0) {
			--first;
			return;
		}

		uint64 d = getCycleCount() - start;
		std::cout << "Publishing took: " << d << std::endl;
		total += d;
		++count;

		uint64 memDiff2 = getMemoryUsage() - memStart;
		std::cout << "Publisher Memory2: " << memDiff2 << std::endl;
	}

protected:
	Channel<Vector> mChannel;
	int N;
	int count;
	uint64 total;
	int first;

	uint64 memStart;
};

MIRA_CLASS_SERIALIZATION( BenchmarkPublisher, Unit );



class BenchmarkSubscriber : public MicroUnit
{
	MIRA_OBJECT(BenchmarkSubscriber)
public:

	BenchmarkSubscriber() : count(0), total(0)
	{
		// skip first 10 measurements
		first=10;
	}

	~BenchmarkSubscriber()
	{
		if (count > 0)
			std::cout << "Latency Average: " << total/count << std::endl;
	}

	void callback(ChannelRead<Vector> data)
	{
		// skip first measurements
		if(first>0) {
			--first;
			return;
		}

		uint64 d = getCycleCount() - data->value().rdtsc;
		std::cout << "Transport took: " << d << std::endl;
		total += d;
		++count;

		uint64 memDiff = getMemoryUsage() - memStart;
		std::cout << "Subscriber Memory: " << memDiff << std::endl;
	}

	virtual void initialize()
	{
		MIRA_SLEEP(2000); // make sure all libs are loaded and mem stats are stable

		memStart = getMemoryUsage();
		std::cout << "Subscriber Start1: " << memStart << std::endl;

		subscribe<Vector>("Data", boost::bind(&BenchmarkSubscriber::callback, this, _1));
	}

	int count;
	uint64 total;
	int first;

	uint64 memStart;
};

MIRA_CLASS_SERIALIZATION( BenchmarkSubscriber, MicroUnit );

