/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RemoteIncomingConnection.C
 *    Implementation of RemoteIncomingConnection.
 *
 * @author Tim Langner
 * @date   2010/11/18
 */

#include <fw/RemoteConnection.h>

#include <boost/array.hpp>

#include <utils/StringAlgorithms.h>
#include <security/RSASignature.h>

#include <fw/Framework.h>
#include <fw/FrameworkDefines.h>
#include <fw/RemoteModule.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

RemoteIncomingConnection::RemoteIncomingConnection()
{
	mPTPOutgoing = false;
}

void RemoteIncomingConnection::start()
{
	RemoteConnection::start();
	mService.runThreads(1, false);

	// obtain the address from the peer endpoint we are connected to
	address.address = mPeerEndpoint.address().to_string() + ":" +
			toString(mPeerEndpoint.port());
	address.keep = false;

	boost::asio::async_read(mSocket, mHeader.asBuffer(), boost::bind(
				&RemoteIncomingConnection::handleReadHeader, this,
				boost::asio::placeholders::error));
}

void RemoteIncomingConnection::onDisconnect()
{
	MIRA_FW.getRemoteModule()->onIncomingDisconnected(this);
}

void RemoteIncomingConnection::handleReadHeader(
		const boost::system::error_code& error)
{
	if (!error && checkMessageHeader())
	{
		mHeaderReceived = Time::now();

		mMessage.resize(mHeader.messageSize);
		boost::asio::async_read(mSocket, boost::asio::buffer(mMessage.data(), mMessage.size()),
		                        boost::bind(&RemoteIncomingConnection::handleReadMessage, this,
		                                    boost::asio::placeholders::error));
	}
	else
	{
		MIRA_LOG(ERROR) << "Closing incoming connection in handleReadHeader: " << error.message();
		stop();
	}
}

void RemoteIncomingConnection::handleReadMessage(const boost::system::error_code& error)
{
	try {

	if (!error)
	{
		try {

		if(mAuthState==AUTHSTATE_ACCEPTED) {
			parseMessage();
		} else {
			if(mHeader.messageType == CONNECT_MSG && mAuthState==AUTHSTATE_NONE) {

				const uint32 majorVersion = MIRA_MAJOR_VERSION(RemoteModule::getCurrentVersion());
				std::string group = MIRA_FW.getRemoteModule()->getAuthSettings().group;
				uint32 authMode = MIRA_FW.getRemoteModule()->getAuthSettings().getMode();

				BinaryBufferIstream os(&mMessage);
				std::string remoteGroup;
				uint32 remoteAuthMode;

				os >> remoteVersion >> remoteGroup >> remoteID >> frameworkID >> remoteAuthMode;
				// disable PTP sync for incoming connections, unless remote runs older protocol
				uint32 remoteMajorVersion = MIRA_MAJOR_VERSION(remoteVersion);

				if(remoteMajorVersion != majorVersion) {
					mAuthState = AUTHSTATE_DENIED;
					// exchange remote and local versions since message is delivered
					// to the remote side
					MIRA_THROW(XConnection, "Framework protocol major versions do not match. "
					           "This version: "+ toString(majorVersion) +
					           " remote version: " + toString(remoteMajorVersion));
				}

				if(remoteGroup!=group) {
					mAuthState = AUTHSTATE_DENIED;
					MIRA_THROW(XConnection, "Connecting frameworks do not belong to the same workgroup.");
				}

				boost::tuple<int,float> t;
				if(remoteAuthMode!=authMode) {
					mAuthState = AUTHSTATE_DENIED;
					MIRA_THROW(XConnection, "Connecting frameworks do not use the same authentication method.");
				}

				if(authMode==RemoteModule::AUTH_NONE) {
					if (!MIRA_FW.getRemoteModule()->onIncomingConnected(this)) {
						MIRA_THROW(XConnection, "Already connected.");
					} else {
						// success
						mAuthState = AUTHSTATE_ACCEPTED;
						writeMessage(CONNECT_ACCEPT_MSG,
						             MIRA_FW.getRemoteModule()->getID(), MIRA_FW.getID(),
						             RemoteModule::getCurrentVersion());
					}

				} else if(authMode==RemoteModule::AUTH_PASSWORD) {
					std::string remotePasswd;
					os >> remotePasswd;

					std::string passwd = MIRA_FW.getRemoteModule()->getAuthSettings().getPassword();

					if(remotePasswd!=passwd) {
						mAuthState = AUTHSTATE_DENIED;
						MIRA_THROW(XConnection, "Invalid password.");
					}

					if (!MIRA_FW.getRemoteModule()->onIncomingConnected(this)) {
						MIRA_THROW(XConnection, "Already connected.");
					} else {
						// success
						mAuthState = AUTHSTATE_ACCEPTED;
						writeMessage(CONNECT_ACCEPT_MSG,
						             MIRA_FW.getRemoteModule()->getID(), MIRA_FW.getID(),
						             RemoteModule::getCurrentVersion());
					}

				} else if(authMode==RemoteModule::AUTH_KEY) {
					std::string clientMsg;
					os >> clientMsg;

					const RSAKey& privateKey = MIRA_FW.getRemoteModule()->getAuthSettings().getKey();

					// sign the clients message to prove our integrity
					std::string signatureStr;
					try {
						RSASignature signature =
							RSASignature::signMessage(privateKey,
							                          RSASignature::DIGEST_MD5,
							                          clientMsg.c_str(), clientMsg.size());
						signatureStr = toString(signature);
					} catch(...) {}

					// prepare our own random message to check the client
					boost::uuids::random_generator gen;
					mAuthSignMsg = toString(gen());

					// send signature and own msg
					writeMessage(SERVER_AUTH_MSG, signatureStr, mAuthSignMsg);
					mAuthState=AUTHSTATE_AUTHENTICATING;
				}
			}
			else if(mHeader.messageType == CLIENT_AUTH_MSG && mAuthState==AUTHSTATE_AUTHENTICATING) {

				std::string clientSignatureMsg;
				BinaryBufferIstream os(&mMessage);
				os >> clientSignatureMsg;

				const RSAKey& privateKey = MIRA_FW.getRemoteModule()->getAuthSettings().getKey();
				RSAKey publicKey(privateKey.getNStr(),privateKey.getEStr(),"");

				// check signed message from client
				bool verified = false;
				try {
					RSASignature clientSignature = fromString<RSASignature>(clientSignatureMsg);
					verified = RSASignature::verifyMessage(publicKey,
					                                       RSASignature::DIGEST_MD5,
					                                       mAuthSignMsg.c_str(), mAuthSignMsg.size(),
					                                       clientSignature);
				} catch(...) {}

				if(!verified) {
					mAuthState = AUTHSTATE_DENIED;
					MIRA_THROW(XConnection, "Invalid signature.");
				}

				// success
				if (!MIRA_FW.getRemoteModule()->onIncomingConnected(this)) {
					MIRA_THROW(XConnection, "Already connected.");
				} else {
					mAuthState=AUTHSTATE_ACCEPTED;
					writeMessage(CONNECT_ACCEPT_MSG,
					             MIRA_FW.getRemoteModule()->getID(), MIRA_FW.getID(),
					             RemoteModule::getCurrentVersion());
				}
			}
			else {
				MIRA_THROW(XConnection, "Invalid protocol.");
			}
		}

		} catch (XConnection& ex) {
			sendConnectDenied(ex.message());
		}

		// In all cases we have to schedule the next read operation. If the
		// connection was denied, the remote end will close the connection,
		// then async_read operation will end in an error and the connection
		// will be closed.
		boost::asio::async_read(mSocket, mHeader.asBuffer(),
		                        boost::bind(&RemoteIncomingConnection::handleReadHeader,
		                                    this, boost::asio::placeholders::error));
	}
	else
	{
		MIRA_LOG(ERROR) << "Closing incoming connection in handleReadMessage: " << error.message();
		// calling stop here could cause problems as stop() may be called twice (here and in
		// handleReadHeader which will in turn call the onIncomingDisconnected handler twice
		// which finally results in deleting this connection and accessing the invalid pointer
		// afterwards
		stop();
	}

	} catch(std::exception& ex) {
		MIRA_LOG_EXCEPTION(ERROR, ex) << "Closing incoming connection after exception: ";
		// calling stop here could cause problems as stop() may be called twice (here and in
		// handleReadHeader which will in turn call the onIncomingDisconnected handler twice
		// which finally results in deleting this connection and accessing the invalid pointer
		// afterwards
		stop();
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::RemoteIncomingConnection, mira::RemoteConnection);
