/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Loader.h
 *    Configuration loader for loading XML application configuration files.
 *
 * @author Tim Langner
 * @date   2010/09/15
 */

#ifndef _MIRA_LOADER_H_
#define _MIRA_LOADER_H_

#ifndef Q_MOC_RUN
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/assign.hpp>
#endif

#include <factory/Factory.h>
#include <xml/XMLDom.h>

#include <fw/FrameworkExports.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/// forward declaration
class ConfigurationLoader;

/**
 * Base class for prepare plugins.
 * Prepare plugins are used to modify a configuration file before it gets
 * interpreted by loader plugins.
 */
class MIRA_FRAMEWORK_EXPORT ConfigurationPreparePlugin : public Object
{
	MIRA_ABSTRACT_OBJECT(ConfigurationPreparePlugin)
public:
	virtual ~ConfigurationPreparePlugin() {}

	/**
	 * Called by the loader of the configuration file.
	 * @param[in,out] xml The configuration document that can be modified
	 */
	virtual void prepareDocument(XMLDom& xml) = 0;

	/**
	 * Derived classes must return an order for their prepare plugin.
	 * This value is required, since the order of the prepare plugins is
	 * important. Some prepare plugins must be evaluated before others to
	 * produce the correct results. The plugins which return the smallest
	 * order value are evaluated first.
	 */
	virtual int getOrder() const = 0;
};

/// typedef for ConfigurationPreparePlugin pointer
typedef boost::shared_ptr<ConfigurationPreparePlugin> ConfigurationPreparePluginPtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Base class for loader plugins.
 * Loader plugins are used to interpret single XML tags in a configuration file.
 *
 * The Framework queries all available plugin types from the ClassFactory at startup
 * and registers them at the ConfigurationLoader.
 *
 * On registration, the ConfigurationLoader inspects each plugin for the configuration
 * tags it handles and whether it needs to be informed of document start/end.
 * In order to avoid the unconditional need for instantiation, these properties are
 * specified through the plugin's object meta data registered with the ClassFactory
 * using MIRA_META_OBJECT. The loader reads the following properties:
 * - 'StartEndDocument' : if 'true', the plugin is instantiated on registration and
 *                        its startDocument()/endDocument() methods are called at
 *                        start/end of document.
 * - single 'Tag' or
 *   sequence 'Tag0', 'Tag1', ...
 *   (consecutive, starting at 0) : when one of these tags is read, the plugin is
 *                                  instantiated and its parseNode() method called.
 * - 'AlwaysInstantiate' : if 'true', the plugin is instantiated on registration,
 *                         regardless of the other properties.
 *
 * Examples:
 * \code
 * MIRA_META_OBJECT(NamespaceLoader,
 *                  ("Tag", "namespace")
 *                  ("StartEndDocument", "true"))
 *
 * MIRA_META_OBJECT(InitialWorkspaceLoader,
 *                 ("Tag0", "initial-workspace")
 *                 ("Tag1", "initial_workspace")
 *	               ("AlwaysInstantiate", "true"))
 * \endcode
 *
 *
 * Each plugin is instantiated (by each loader) once at most, and then stored and
 * used from that point onward. If neither StartEndDocument nor AlwaysInstantiate
 * is true, the plugin is only instantiated when a respective tag is actually
 * read in the configuration.
 */
class MIRA_FRAMEWORK_EXPORT ConfigurationLoaderPlugin : public Object
{
	MIRA_ABSTRACT_OBJECT(ConfigurationLoaderPlugin)
public:
	virtual ~ConfigurationLoaderPlugin() {}

	/**
	 * Is called when a new document is parsed.
	 * Can be used for reset and initialization, or for setting default values
	 * to the loaders context.
	 * @param[in,out] ioLoader The loader that called us.
	 *                Can be used to modify the context of the loader.
	 */
	virtual void startDocument(ConfigurationLoader* ioLoader) {};

	/**
	 * Is called after the whole document was parsed.
	 * Can be used to invoke the parsed content or for clean up.
	 * @param[in,out] ioLoader The loader that called us.
	 */
	virtual void endDocument(ConfigurationLoader* ioLoader) {};

	/**
	 * Is called whenever a XML tag is found that this loader supports.
	 * @param[in] node The node containing the XML tag.
	 * @param[in,out] ioLoader The loader that called us.
	 *                Can be used to modify the context of the loader.
	 */
	virtual void parseNode(const XMLDom::const_iterator& node,
	                       ConfigurationLoader* ioLoader) = 0;
};

/// typedef for ConfigurationLoaderPlugin pointer
typedef boost::shared_ptr<ConfigurationLoaderPlugin> ConfigurationLoaderPluginPtr;

//////////////////////////////////////////////////////////////////////////////

/**
 * Class for loading, parsing, modifying and interpreting application
 * configuration files.
 */
class MIRA_FRAMEWORK_EXPORT ConfigurationLoader
{
public:
	typedef std::map<std::string, std::string> Context;
public:

	/**
	 * The constructor.
	 * @param[in] unknownTagWarning Flag controlling if warning messages are shown when
	 *                              encountering unknown tags.
	 */
	ConfigurationLoader(bool unknownTagWarning = true);

	/**
	 * Register a prepare plugin. They get inserted in a list of
	 * prepare plugins that is sorted ascending by the order returned by the
	 * call to ConfigurationPreparePlugin::getOrder() (smallest order first).
	 * @param[in] plugin The plugin
	 */
	void registerPreparePlugin(ConfigurationPreparePluginPtr plugin);
	/**
	 * Register a loader plugin.
	 * These plugins are called whenever their supported tag is found.
	 * @param[in] plugin The plugin class description
	 */
	void registerLoaderPlugin(const ClassProxy& plugin);

	/**
	 * Prepare the document.
	 * Prepare plugins are called in the order they appear in the sorted
	 * list that is sorted ascending by the order number of each preparer
	 * (smallest order first).
	 * @param[in,out] ioXML The xml document
	 */
	void prepare(XMLDom& ioXML);
	/**
	 * Load/interpret the document.
	 * All loader plugins are called whenever their supported tag is found.
	 * @param[in] xml The xml document
	 */
	void load(const XMLDom& xml);

	/**
	 * Get the context of the loader.
	 * The context can be used to store the current state of a plugin.
	 * This state can be used by other plugins.
	 * e.g. the NamespaceLoader stores the current namespace context,
	 * that can be used by the UnitLoader.
	 */
	Context& getContext()
	{
		return mContext;
	}

	///@cond INTERNAL

	/**
	 * Internal use only but public interface.
	 */
	void parseNode(const XMLDom::const_iterator& node);
	/**
	 * Internal use only but public interface.
	 */
	void parse(const XMLDom::const_iterator& iNode);

	///@endcond

private:

	struct CompareConfigurationPreparePlugin {
		bool operator()(const ConfigurationPreparePluginPtr& a,
		                const ConfigurationPreparePluginPtr& b) const {
			return a->getOrder() < b->getOrder();
		}
	};

	/// Current context of all loaders. E.g. stores the current namespace of
	/// the node we are parsing.
	Context mContext;
	typedef std::set<ConfigurationPreparePluginPtr,
	                 CompareConfigurationPreparePlugin> PreparePluginMap;

	typedef std::vector<std::pair<ClassProxy, ConfigurationLoaderPluginPtr>> LoaderPluginSet;
	typedef std::map<std::string, LoaderPluginSet> LoaderPluginMap;

	bool mEnableUnknownTagWarnings;

	/// List of registered prepare plugins
	PreparePluginMap mPreparePlugins;

	/// List of registered loader plugins
	LoaderPluginMap mLoaderPluginsMap;
	LoaderPluginSet mLoaderPlugins;

	bool mAddedInfoToException;
};

//////////////////////////////////////////////////////////////////////////////

}

#endif
