/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDomTest.C
 *
 * @author Tim Langner
 * @date   2010/07/01
 */

#include <boost/test/unit_test.hpp>

#include <iostream>

#include <xml/XMLDom.h>

using namespace mira;
using namespace std;

XMLDom createDocument()
{
	XMLDom xml;
	XMLDom::iterator node1 = xml.root().
			add_child("Node").
			add_attribute("StringAttribute", "text");
	XMLDom::iterator node2 = node1.
				add_child("SubNode");
	node2.add_attribute("IntAttribute", 1);
	node2.add_child("A").add_content("1");
	node2.add_child("A").add_content("2");
	node2.add_child("A").add_content("3");
	xml.root().add_child("MultiNode").add_content("1");
	xml.root().add_child("MultiNode").add_content("2");
	xml.root().add_child("MultiNode").add_content("3");
	xml.root().add_child("CDataNode").add_cdata_content("<root><node attr=\"test\">Test string</node></root>");
	return xml;
}

BOOST_AUTO_TEST_CASE( testXMLDomSaveLoad )
{
	XMLDom xml = createDocument();
	xml.saveToFile("test.xml");

	XMLDom xmlLoad;
	xmlLoad.loadFromFile("test.xml");
	XMLDom::iterator root = xmlLoad.root();
	BOOST_CHECK_EQUAL(*root.begin(), "Node");
	BOOST_CHECK_EQUAL((*root.begin().find_attribute("StringAttribute")).second, "text");
	XMLDom::iterator findNode = std::find(root.begin().begin(), root.begin().end(), "SubNode");
	BOOST_CHECK(findNode != root.begin().end());
	BOOST_CHECK_EQUAL((*findNode.find_attribute("IntAttribute")).second, "1");
}

BOOST_AUTO_TEST_CASE( testXMLDomConst )
{
	std::stringstream documentStream;
	documentStream << "<ROOT>";
	documentStream << "<Node attribute=\"1\"/>";
	documentStream << "<Node>";
	documentStream << "<SubNode/>";
	documentStream << "</Node>";
	documentStream << "</ROOT>";
	XMLDom xml;
	xml.loadFromString(documentStream.str());

	BOOST_CHECK_EQUAL(*xml.root(), "ROOT");
	XMLDom::const_iterator i = xml.croot().cbegin();
	BOOST_CHECK_EQUAL((*i.find_attribute("attribute")).second, "1");
	BOOST_CHECK_EQUAL(i.get_attribute<int>("attribute"), 1);
	BOOST_CHECK(i.find_attribute("notvalid") == i.attribute_end());
	BOOST_CHECK_EQUAL(i.get_attribute<int>("notvalid", 5), 5);
	BOOST_CHECK(i.parent() == xml.croot());
}

BOOST_AUTO_TEST_CASE( testXMLDom )
{
	std::stringstream documentStream;
	documentStream << "<ROOT>";
	documentStream << "<Node attribute=\"1\"/>";
	documentStream << "<Node>";
	documentStream << "<SubNode/>";
	documentStream << "</Node>";
	documentStream << "</ROOT>";
	XMLDom xml;
	xml.loadFromString(documentStream.str());

	BOOST_CHECK_EQUAL(*xml.root(), "ROOT");
	XMLDom::iterator i = xml.root().begin();
	BOOST_CHECK_EQUAL((*i.find_attribute("attribute")).second, "1");
	BOOST_CHECK_EQUAL(i.get_attribute<int>("attribute"), 1);
	BOOST_CHECK(i.find_attribute("notvalid") == i.attribute_end());
	BOOST_CHECK_EQUAL(i.get_attribute<int>("notvalid", 5), 5);
	BOOST_CHECK(i.parent() == xml.root());
}

BOOST_AUTO_TEST_CASE( testConversion )
{
	XMLDom xml = createDocument();
	XMLDom::iterator i = xml.root();
	XMLDom::const_iterator ci = i;
	BOOST_CHECK_EQUAL(*ci.begin(), "Node");
}

BOOST_AUTO_TEST_CASE( testReplace )
{
	XMLDom xml = createDocument();
	xml.root().replace(xml.root().begin());
	BOOST_CHECK_EQUAL(*xml.root(), "Node");
}

BOOST_AUTO_TEST_CASE( testFind )
{
	XMLDom xml = createDocument();
	BOOST_CHECK_EQUAL(*xml.root().find("Node/SubNode/A",2).content_begin(), "3");
	XMLDom::iterator i = xml.root().find("Node");
	BOOST_CHECK_EQUAL(*i, "Node");
	i = xml.root().find("NoNode");
	BOOST_CHECK(i == xml.root().end());
}

BOOST_AUTO_TEST_CASE( testDelete )
{
	XMLDom xml = createDocument();
	XMLDom::iterator i = xml.root().find("Node/SubNode");
	BOOST_CHECK(i != xml.root().end());
	i.remove();
	i = xml.root().find("Node/SubNode");
	BOOST_CHECK(i == xml.root().end());
}

BOOST_AUTO_TEST_CASE( iteratorTest )
{
	XMLDom xml = createDocument();
	XMLDom::iterator i1 = xml.root().find("MultiNode");
	i1 += 2;
	BOOST_CHECK_EQUAL(*i1, "MultiNode");
	XMLDom::iterator i2 = xml.root().find("MultiNode") + 2;
	BOOST_CHECK_EQUAL(*i2, "MultiNode");
}

BOOST_AUTO_TEST_CASE( cdataTest )
{
	XMLDom xml = createDocument();
	XMLDom::iterator i = xml.root().find("CDataNode");
	BOOST_CHECK_EQUAL(*i.content_begin(), "<root><node attr=\"test\">Test string</node></root>");
	BOOST_CHECK_EQUAL(*i.content_cbegin(), "<root><node attr=\"test\">Test string</node></root>");
}

BOOST_AUTO_TEST_CASE( insertBeforeAfter )
{
	XMLDom xml;
	XMLDom::iterator i = xml.root().add_child("Child");
	i.insert_before("Before");
	i.insert_after("After");

	i = xml.root().begin();
	BOOST_CHECK_EQUAL(*i, "Before");
	++i;
	BOOST_CHECK_EQUAL(*i, "Child");
	++i;
	BOOST_CHECK_EQUAL(*i, "After");
}

BOOST_AUTO_TEST_CASE( uri )
{
	XMLDom xml1;
	xml1.setUri("xml1");

	XMLDom xml2;
	xml2.setUri("xml2");
 
 	XMLDom xml3;
	xml3.setUri("xml3");

	XMLDom xml4;
	xml4.setUri("xml4");

	xml1.root().add_child("Child1");
	XMLDom::iterator i1 = xml1.root().add_child("Child3"); // to be replaced
	       
	XMLDom::iterator i2 = xml2.root().add_child("Child2");
	XMLDom::iterator i3 = xml3.root().add_child("Child3");
	XMLDom::iterator i4 = xml4.root().add_child("Child4");
	
	XMLDom::iterator i3r = i1.replace(i3);
	i3r.insert_before(i2);
	i3r.insert_after(i4);
	
	BOOST_CHECK_EQUAL(xml1.root().uri(), "xml1");

	XMLDom::iterator i = xml1.root().begin();
	BOOST_CHECK_EQUAL(*i, "Child1");
	BOOST_CHECK_EQUAL(i.uri(), "xml1");
	++i;
	BOOST_CHECK_EQUAL(*i, "Child2");
	BOOST_CHECK_EQUAL(i.uri(), "xml2");
	++i;
	BOOST_CHECK_EQUAL(*i, "Child3");
	BOOST_CHECK_EQUAL(i.uri(), "xml3");
	++i;
	BOOST_CHECK_EQUAL(*i, "Child4");
 	BOOST_CHECK_EQUAL(i.uri(), "xml4");
 }
