/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ColormapTest.C
 *    test cases for the color map class.
 *
 * @author Markus Eisenbach
 */

#include <image/Img.h>
#include <image/Colormap.h>
#include <iostream>

#include <boost/test/unit_test.hpp>

using namespace mira;

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( ColormapGrayscaleTest )
{
	GrayscaleColormap gsp;
	float weightedSumR = 0.0f;
	float weightedSumG = 0.0f;
	float weightedSumB = 0.0f;
	for (std::size_t i = 0; i < gsp.size(); ++i) {
		Color::RGB col = gsp(i);
		weightedSumR += col.r * (float)i;
		weightedSumG += col.g * (float)(i*2);
		weightedSumB += col.b * (float)(i/2);
	}
	BOOST_CHECK_CLOSE(weightedSumR, 21717.5, 0.01);
	BOOST_CHECK_CLOSE(weightedSumG, 43435, 0.01);
	BOOST_CHECK_CLOSE(weightedSumB, 10826.8, 0.01);

	BOOST_CHECK(Img8U3::Pixel(gsp.getf(-1.0f)) == Img8U3::Pixel(Color::RGB(0.0f, 0.0f, 0.0f)));
	BOOST_CHECK(Img8U3::Pixel(gsp.getf(0.5f)) == Img8U3::Pixel(Color::RGB(0.5f, 0.5f, 0.5f)));
	BOOST_CHECK(Img8U3::Pixel(gsp.getf(2.0f)) == Img8U3::Pixel(Color::RGB(1.0f, 1.0f, 1.0f)));
}

BOOST_AUTO_TEST_CASE( ColormapHSVTest )
{
	HSVColormap hsvp;
	float weightedSumR = 0.0f;
	float weightedSumG = 0.0f;
	float weightedSumB = 0.0f;
	for (std::size_t i = 0; i < hsvp.size(); ++i) {
		Color::RGB col = hsvp(i);
		weightedSumR += col.r * (float)i;
		weightedSumG += col.g * (float)(i*2);
		weightedSumB += col.b * (float)(i/2);
	}
	BOOST_CHECK_CLOSE(weightedSumR, 16254.9, 0.01);
	BOOST_CHECK_CLOSE(weightedSumG, 21846, 0.01);
	BOOST_CHECK_CLOSE(weightedSumB, 10890, 0.01);

	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(-1.0f)) == Img8U3::Pixel(Color::RGB(1,       0,       0)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(0.15f)) == Img8U3::Pixel(Color::RGB(1,       0.9f,    0)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(0.3f))  == Img8U3::Pixel(Color::RGB(0.2002f, 1,       0)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(0.45f)) == Img8U3::Pixel(Color::RGB(0,       1,       0.7002f)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(0.6f))  == Img8U3::Pixel(Color::RGB(0,       0.3999f, 1)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(0.75f)) == Img8U3::Pixel(Color::RGB(0.4998f, 0,       1)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(0.9f))  == Img8U3::Pixel(Color::RGB(1,       0,       0.5998f)));
	BOOST_CHECK(Img8U3::Pixel(hsvp.getf(1.2f))  == Img8U3::Pixel(Color::RGB(1,       0,       0)));
}

BOOST_AUTO_TEST_CASE( ColormapRedBlueTest )
{
	RedBlueColormap rbp;
	float weightedSumR = 0.0f;
	float weightedSumG = 0.0f;
	float weightedSumB = 0.0f;
	for (std::size_t i = 0; i < rbp.size(); ++i) {
		Color::RGB col = rbp(i);
		weightedSumR += col.r * (float)i;
		weightedSumG += col.g * (float)(i*2);
		weightedSumB += col.b * (float)(i/2);
	}
	BOOST_CHECK_CLOSE(weightedSumR, 21717.5, 0.01);
	BOOST_CHECK_CLOSE(weightedSumG, 0, 0.01);
	BOOST_CHECK_CLOSE(weightedSumB, 5429.25, 0.01);

	BOOST_CHECK(Img8U3::Pixel(rbp.getf(-1.0f)) == Img8U3::Pixel(Color::RGB(0.0f, 0.0f, 1.0f)));
	BOOST_CHECK(Img8U3::Pixel(rbp.getf(0.5f)) == Img8U3::Pixel(Color::RGB(0.5f, 0.0f, 0.5f)));
	BOOST_CHECK(Img8U3::Pixel(rbp.getf(2.0f)) == Img8U3::Pixel(Color::RGB(1.0f, 0.0f, 0.0f)));
}

BOOST_AUTO_TEST_CASE( ColormapComplementary10Test )
{
	Complementary10Colormap c10p;
	float weightedSumR = 0.0f;
	float weightedSumG = 0.0f;
	float weightedSumB = 0.0f;
	for (std::size_t i = 0; i < c10p.size(); ++i) {
		Color::RGB col = c10p(i);
		weightedSumR += col.r * (float)i;
		weightedSumG += col.g * (float)(i*2);
		weightedSumB += col.b * (float)(i/2);
	}
	BOOST_CHECK_CLOSE(weightedSumR, 27.5176, 0.01);
	BOOST_CHECK_CLOSE(weightedSumG, 74.0157, 0.01);
	BOOST_CHECK_CLOSE(weightedSumB, 9, 0.01);
}

BOOST_AUTO_TEST_CASE( ColormapComplementary6Test )
{
	Complementary6Colormap c6p;
	float weightedSumR = 0.0f;
	float weightedSumG = 0.0f;
	float weightedSumB = 0.0f;
	for (std::size_t i = 0; i < c6p.size(); ++i) {
		Color::RGB col = c6p(i);
		weightedSumR += col.r * (float)i;
		weightedSumG += col.g * (float)(i*2);
		weightedSumB += col.b * (float)(i/2);
	}
	BOOST_CHECK_CLOSE(weightedSumR, 12, 0.01);
	BOOST_CHECK_CLOSE(weightedSumG, 20, 0.01);
	BOOST_CHECK_CLOSE(weightedSumB, 2, 0.01);
}

BOOST_AUTO_TEST_CASE( ColormapHSVOperatorTest )
{
	HSVColormap hsvp;
	for (std::size_t i = 0; i < hsvp.size(); ++i) {
		Color::RGB col = hsvp(i);
		Color::RGB col2 = hsvp[i];
		BOOST_CHECK(col.r == col2.r);
		BOOST_CHECK(col.g == col2.g);
		BOOST_CHECK(col.b == col2.b);
	}
}

