/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file BinaryJSONConverter.C
 *    Implementation of BinaryJSONConverter.h.
 *
 * @author Erik Einhorn
 * @date   2012/10/29
 */

#include <serialization/BinaryJSONConverter.h>
#include <serialization/BinarySerializer.h>

namespace mira {
namespace BinaryJSONConverter {

///////////////////////////////////////////////////////////////////////////////

int32 sItemCount = -1;
bool sBlockDump = false;

template <typename T, typename U>
static void atomicToJSONHelper(BinaryBufferIstream& stream, JSONValue& oObject)
{
	T v;
	stream >> v;
	oObject = JSONValue((U)v);
}

///////////////////////////////////////////////////////////////////////////////

namespace v0 {

static void checkVersion(const TypeMeta& type)
{
	if (type.version() != 0)
		MIRA_THROW(XIO, "binaryToJSON: Version mismatch for meta type '" << type.getIdentifier()
		             << "'. Binary version 0/1, type meta version " << (int)type.version() << ".");
}

static void atomicToJSON(BinaryBufferIstream& stream, const TypeMeta& type, JSONValue& oObject)
{
	switch(type.getType())
	{
		case TypeMeta::TYPE_ATOMIC:
			MIRA_THROW(XNotImplemented, "Cannot convert user-defined atomic types");
			break;
		case TypeMeta::TYPE_UINT8:  atomicToJSONHelper<uint8,int>(stream,oObject); break;
		case TypeMeta::TYPE_INT8:   atomicToJSONHelper<int8,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT16: atomicToJSONHelper<uint16,int>(stream,oObject); break;
		case TypeMeta::TYPE_INT16:  atomicToJSONHelper<int16,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT32: atomicToJSONHelper<uint32,int64>(stream,oObject); break;
		case TypeMeta::TYPE_INT32:  atomicToJSONHelper<int32,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT64: atomicToJSONHelper<uint64,int64>(stream,oObject); break;
		case TypeMeta::TYPE_INT64:  atomicToJSONHelper<int64,int64>(stream,oObject); break;
		case TypeMeta::TYPE_FLOAT:  atomicToJSONHelper<float,double>(stream,oObject); break;
		case TypeMeta::TYPE_DOUBLE: atomicToJSONHelper<double,double>(stream,oObject); break;
		case TypeMeta::TYPE_BOOL:   atomicToJSONHelper<bool,bool>(stream,oObject); break;
		case TypeMeta::TYPE_STRING: atomicToJSONHelper<std::string,std::string>(stream,oObject); break;
		case TypeMeta::TYPE_ENUMERATION:  atomicToJSONHelper<int,int>(stream,oObject); break;
		case TypeMeta::TYPE_VERSION: atomicToJSONHelper<int,int>(stream,oObject); break;
		default:
			MIRA_THROW(XNotImplemented, "Cannot convert unknown atomic types");
			break;
	}
}

static void objectToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject);

static void collectionToJSON(BinaryBufferIstream& stream,
                             const TypeMeta& nestedType,
                             const MetaTypeDatabase& metadb, JSONValue& oObject);

static void pointerToJSON(BinaryBufferIstream& stream,
                          const TypeMeta& unqualifiedType,
                          const MetaTypeDatabase& metadb, JSONValue& oObject);


static void binaryToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject)
{
	checkVersion(type);

	// handle qualifiers first
	if(type.hasQualifier()) {
		TypeMeta unqualifiedType = type;
		unqualifiedType.removeQualifiers(1); // remove this qualifier

		switch(type.getQualifier())
		{
		case TypeMeta::TYPE_COLLECTION:
			collectionToJSON(stream,unqualifiedType,metadb, oObject);
			break;
		case TypeMeta::TYPE_POINTER:
			pointerToJSON(stream,unqualifiedType,metadb, oObject);
			break;
		default:
			MIRA_THROW(XIO, "Unknown meta type qualifier: " << type.getQualifier());
			break;
		}

	} else {
	// the plain type is remaining only

		switch(type.getType())
		{
		case TypeMeta::TYPE_CLASS:
			objectToJSON(stream,type,metadb, oObject);
			break;
		default:
			atomicToJSON(stream,type,oObject);
			break;
		}
	}
}

static void objectToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject)
{
	checkVersion(type);

	std::string identifier = type.getIdentifier();
	auto it = metadb.find(identifier);
	if(it==metadb.end())
		MIRA_THROW(XIO, "Cannot convert unknown class '" << identifier << "'");	

	oObject = JSONObject();

	CompoundMetaPtr compound = it->second;
	foreach(const CompoundMeta::Member& m, compound->members)
		binaryToJSON(stream,*m.type,metadb,oObject.get_obj()[m.name]);
}

static void collectionToJSON(BinaryBufferIstream& stream,
                             const TypeMeta& nestedType,
                             const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	checkVersion(nestedType);

	oObject = JSONArray();
	JSONArray& a = oObject.get_array();

	uint32 count;
	stream >> count;

	a.resize(count);

	for(uint32 i=0; i<count; ++i)
		binaryToJSON(stream,nestedType,metadb,a[i]);
}

static void pointerToJSON(BinaryBufferIstream& stream,
                          const TypeMeta& unqualifiedType,
                          const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	checkVersion(unqualifiedType);

	uint8 pointerType8U;
	stream >> pointerType8U;

	BinarySerializerMixin::PointerType pointerType = (BinarySerializerMixin::PointerType) pointerType8U;

	switch(pointerType)
	{
		case BinarySerializerMixin::NULL_POINTER:
			oObject = JSONValue();
			break;
		case BinarySerializerMixin::POINTER_REFERENCE:
		{
			int ref;
			stream >> ref; // read the ref id
			// ... and resolve it
			//pointer = this->template resolveReference<T>(ref);
			// TODO: Oh god!
			MIRA_THROW(XIO, "Not implemented yet");
			break;
		}

		case BinarySerializerMixin::NORMAL_POINTER:
			binaryToJSON(stream,unqualifiedType,metadb,oObject);
			break;

		case BinarySerializerMixin::POLYMORPHIC_POINTER: {
			// we have a "polymorphic" pointer, so get class type
			std::string typenam;
			stream >> typenam;
			typenam += " @v0"; // TODO: does this work as intended? need test

			// we can convert a polymorphic pointer only if we have the typename in our type database
			auto it = metadb.find(typenam);
			if(it==metadb.end())
				MIRA_THROW(XIO, "Cannot convert polymorphic pointer of unknown class '" << typenam << "'");

			assert(!unqualifiedType.hasQualifier());
			TypeMeta type;
			type.setClassType(typenam);
			binaryToJSON(stream,type,metadb,oObject);
			oObject.get_obj()["@class"] = json::Value(typenam);
			break;
		}
	}
}

} // namespace v0

///////////////////////////////////////////////////////////////////////////////

namespace v2 {

static void checkVersion(const TypeMeta& type)
{
	if (type.version() != 2)
		MIRA_THROW(XIO, "binaryToJSON: Version mismatch for meta type " << type.getIdentifier()
		             << ". Binary version 2, type meta version " << (int)type.version() << ".");
}

static void atomicToJSON(BinaryBufferIstream& stream, const TypeMeta& type, JSONValue& oObject)
{
	switch(type.getType())
	{
		case TypeMeta::TYPE_ATOMIC:
			MIRA_THROW(XNotImplemented, "Cannot convert user-defined atomic types");
			break;
		case TypeMeta::TYPE_UINT8:  atomicToJSONHelper<uint8,int>(stream,oObject); break;
		case TypeMeta::TYPE_INT8:   atomicToJSONHelper<int8,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT16: atomicToJSONHelper<uint16,int>(stream,oObject); break;
		case TypeMeta::TYPE_INT16:  atomicToJSONHelper<int16,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT32: atomicToJSONHelper<uint32,int64>(stream,oObject); break;
		case TypeMeta::TYPE_INT32:  atomicToJSONHelper<int32,int>(stream,oObject); break;
		case TypeMeta::TYPE_UINT64: atomicToJSONHelper<uint64,int64>(stream,oObject); break;
		case TypeMeta::TYPE_INT64:  atomicToJSONHelper<int64,int64>(stream,oObject); break;
		case TypeMeta::TYPE_FLOAT:  atomicToJSONHelper<float,double>(stream,oObject); break;
		case TypeMeta::TYPE_DOUBLE: atomicToJSONHelper<double,double>(stream,oObject); break;
		case TypeMeta::TYPE_BOOL:   atomicToJSONHelper<bool,bool>(stream,oObject); break;
		case TypeMeta::TYPE_STRING: atomicToJSONHelper<std::string,std::string>(stream,oObject); break;
		case TypeMeta::TYPE_ENUMERATION:  atomicToJSONHelper<int,int>(stream,oObject); break;
		case TypeMeta::TYPE_VERSION: atomicToJSONHelper<serialization::VersionType,int>(stream,oObject); break;
		default:
			MIRA_THROW(XNotImplemented, "Cannot convert unknown atomic types");
			break;
	}
}

static void objectToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject);

static void collectionToJSON(BinaryBufferIstream& stream,
                             const TypeMeta& nestedType,
                             const MetaTypeDatabase& metadb, JSONValue& oObject);

static void pointerToJSON(BinaryBufferIstream& stream,
                          const TypeMeta& unqualifiedType,
                          const MetaTypeDatabase& metadb, JSONValue& oObject);

static void binaryToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject)
{
	// handle qualifiers first
	if(type.hasQualifier()) {
		TypeMeta unqualifiedType = type;
		unqualifiedType.removeQualifiers(1); // remove this qualifier

		switch(type.getQualifier())
		{
		case TypeMeta::TYPE_COLLECTION:
			collectionToJSON(stream,unqualifiedType,metadb, oObject);
			break;
		case TypeMeta::TYPE_POINTER:
			pointerToJSON(stream,unqualifiedType,metadb, oObject);
			break;
		default:
			MIRA_THROW(XIO, "Unknown meta type qualifier: " << type.getQualifier());
			break;
		}

	} else {
	// the plain type is remaining only

		switch(type.getType())
		{
		case TypeMeta::TYPE_CLASS:
			objectToJSON(stream,type,metadb, oObject);
			break;
		default:
			atomicToJSON(stream,type,oObject);
			break;
		}
	}
}

static void objectToJSON(BinaryBufferIstream& stream,
                         const TypeMeta& type, const MetaTypeDatabase& metadb,
                         JSONValue& oObject)
{
	std::string identifier = type.getIdentifier();
	auto it = metadb.find(identifier);
	if(it==metadb.end())
		MIRA_THROW(XIO, "Cannot convert unknown class '" << identifier << "'");

	oObject = JSONObject();

	CompoundMetaPtr compound = it->second;
	foreach(const CompoundMeta::Member& m, compound->members) {
		// handle some special members
		if (m.name == "@version_default") {  // read version placeholder from stream and ignore it
			serialization::VersionType v;
			stream >> v;
		} else if (m.name == "@itemcount") { // read itemcount from stream and store for later use
			stream >> sItemCount;
		} else if (m.name == "@items") {     // @items is not a member, make sure items are added to current object!
			sBlockDump = false;
			binaryToJSON(stream,*m.type,metadb,oObject);
		} else if (m.name == "@items_blockdump") {
			sBlockDump = true;
			binaryToJSON(stream,*m.type,metadb,oObject);
		} else if (m.name == "@transparent") { // parent is a transparent object,
		                                       // this member should appear inline here
			binaryToJSON(stream,*m.type,metadb,oObject);
		} else {
			binaryToJSON(stream,*m.type,metadb,oObject.get_obj()[m.name]);
		}
	}
}

static void collectionToJSON(BinaryBufferIstream& stream,
                             const TypeMeta& nestedType,
                             const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	oObject = JSONArray();
	JSONArray& a = oObject.get_array();

	if (sItemCount < 0)
		MIRA_THROW(XIO, "@itemcount missing for collection!");

	uint32 count = sItemCount; // get last read itemcount and reset it
	sItemCount = -1;

	a.resize(count);

	if (sBlockDump) {
		for(uint32 i=0; i<count; ++i) {
			// this will only work for atomic types, otherwise we need either an instance of the type
			// or a type meta based on a plain dump (not based on the type's reflect method)
			atomicToJSON(stream,nestedType,a[i]); 
		}
	} else {
		for(uint32 i=0; i<count; ++i)
			binaryToJSON(stream,nestedType,metadb,a[i]);
	}
}

static void pointerToJSON(BinaryBufferIstream& stream,
                          const TypeMeta& unqualifiedType,
                          const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	uint8 pointerType8U;
	stream >> pointerType8U;

	BinarySerializerMixin::PointerType pointerType = (BinarySerializerMixin::PointerType) pointerType8U;

	switch(pointerType)
	{
		case BinarySerializerMixin::NULL_POINTER:
			oObject = JSONValue();
			break;
		case BinarySerializerMixin::POINTER_REFERENCE:
		{
			int ref;
			stream >> ref; // read the ref id
			// ... and resolve it
			//pointer = this->template resolveReference<T>(ref);
			// TODO: Oh god!
			MIRA_THROW(XIO, "Not implemented yet");
			break;
		}

		case BinarySerializerMixin::NORMAL_POINTER:
			binaryToJSON(stream,unqualifiedType,metadb,oObject);
			break;

		case BinarySerializerMixin::POLYMORPHIC_POINTER: {
			// we have a "polymorphic" pointer, so get class type
			std::string typenam;
			stream >> typenam;

			// we can convert a polymorphic pointer only, if we have the
			// typename in our type database
			auto it = metadb.find(typenam);
			if(it==metadb.end())
				MIRA_THROW(XIO, "Cannot convert polymorphic pointer of unknown class '" << typenam << "'");

			assert(!unqualifiedType.hasQualifier());
			TypeMeta type;
			type.setClassType(typenam);
			binaryToJSON(stream,type,metadb,oObject);
			oObject.get_obj()["@class"] = json::Value(typenam);
			break;
		}
	}
}

} // namespace v2

///////////////////////////////////////////////////////////////////////////////

uint8 binaryToJSON(const Buffer<uint8>& data, bool containsTypename,
                   const TypeMeta& type, const MetaTypeDatabase& metadb, JSONValue& oObject)
{
	BinaryBufferIstream stream((Buffer<uint8>*)&data);

	uint16 m;
	uint8 v;

	stream >> m;
	if (m == BINARY_VERSION_MARKER)
		stream >> v;
	else {
		v = 0;               // v0 did not store a version in binary data, so if we don't find it, it is v0
		stream.seekg(0);
	}

	if (v > 2)
		MIRA_THROW(XIO, "binaryToJSON: Found binary format version " << (int)v << ". Not implemented.");

	if(containsTypename) {
		std::string typenam;
		stream >> typenam;
	}

	if (v <= 1) {
		v0::checkVersion(type); // versions are also checked on all nested type metas
#ifndef NDEBUG
		// give a NOTICE in debug builds, else nothing (as it quickly turns into major spam,
		// and there is no efficient way to make sure it is displayed only once)
		MIRA_LOG(NOTICE) << "BinaryJSONConverter: Found outdated binary format"
		                    ", version " << (int)v << " (and type meta version 0). Using legacy converter.";
#endif
		v0::binaryToJSON(stream,type,metadb,oObject);
	} else if (v == 2) {
		v2::checkVersion(type); // version only checked here once in (default) v2 case (efficiency)
		v2::binaryToJSON(stream,type,metadb,oObject);
	}

	return v;
}

///////////////////////////////////////////////////////////////////////////////

namespace v0 {

static void atomicToBinary(const JSONValue& iObject, const TypeMeta& type,
                           BinaryBufferOstream& stream)
{
	switch(type.getType())
	{
		case TypeMeta::TYPE_ATOMIC:
			MIRA_THROW(XNotImplemented, "Cannot convert user-defined atomic types");
			break;
		case TypeMeta::TYPE_UINT8:  stream << (uint8)iObject.get_int(); break;
		case TypeMeta::TYPE_INT8:   stream << (int8)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT16: stream << (uint16)iObject.get_int(); break;
		case TypeMeta::TYPE_INT16:  stream << (int16)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT32: stream << (uint32)iObject.get_int(); break;
		case TypeMeta::TYPE_INT32:  stream << (int32)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT64: stream << (uint64)iObject.get_int64(); break;
		case TypeMeta::TYPE_INT64:  stream << (int64)iObject.get_int64(); break;
		case TypeMeta::TYPE_FLOAT:  stream << (float)iObject.get_real(); break;
		case TypeMeta::TYPE_DOUBLE: stream << iObject.get_real(); break;
		case TypeMeta::TYPE_BOOL:   stream << iObject.get_bool(); break;
		case TypeMeta::TYPE_STRING: stream << iObject.get_str(); break;
		case TypeMeta::TYPE_ENUMERATION:  stream << iObject.get_int(); break;
		case TypeMeta::TYPE_VERSION: stream << iObject.get_int(); break;
		default:
			MIRA_THROW(XNotImplemented, "Cannot convert unknown atomic types");
			break;
	}
}

static void objectToBinary(const JSONValue& iObject,
                           const TypeMeta& type, const MetaTypeDatabase& metadb,
                           BinaryBufferOstream& stream, int defaultVersion = -1);

static void collectionToBinary(const JSONValue& iObject,
                               const TypeMeta& nestedType, const MetaTypeDatabase& metadb,
                               BinaryBufferOstream& stream, int defaultVersion = -1);

static void pointerToBinary(const JSONValue& iObject,
                            const TypeMeta& unqualifiedType, const MetaTypeDatabase& metadb,
                            BinaryBufferOstream& stream, int defaultVersion = -1);

static void JSONToBinary(const JSONValue& iObject, const TypeMeta& type,
                         const MetaTypeDatabase& metadb, BinaryBufferOstream& stream,
                         int defaultVersion = -1)
{
	checkVersion(type);

	// handle qualifiers first
	if(type.hasQualifier()) {
		TypeMeta unqualifiedType = type;
		unqualifiedType.removeQualifiers(1); // remove this qualifier

		switch(type.getQualifier())
		{
		case TypeMeta::TYPE_COLLECTION:
			collectionToBinary(iObject,unqualifiedType,metadb,stream,defaultVersion);
			break;
		case TypeMeta::TYPE_POINTER:
			pointerToBinary(iObject,unqualifiedType,metadb,stream,defaultVersion);
			break;
		default:
			MIRA_THROW(XIO, "Unknown meta type qualifier: " << type.getQualifier());
			break;
		}

	} else {
	// the plain type is remaining only

		switch(type.getType())
		{
		case TypeMeta::TYPE_CLASS:
			objectToBinary(iObject,type,metadb, stream,defaultVersion);
			break;
		default:
			atomicToBinary(iObject,type,stream);
			break;
		}
	}
}

static void objectToBinary(const JSONValue& iObject,
                           const TypeMeta& type, const MetaTypeDatabase& metadb,
                           BinaryBufferOstream& stream, int defaultVersion)
{
	checkVersion(type);

	std::string identifier = type.getIdentifier();
	auto it = metadb.find(identifier);
	if(it==metadb.end())
		MIRA_THROW(XIO, "Cannot convert unknown class '" << identifier << "'");

	CompoundMetaPtr compound = it->second;
	foreach(const CompoundMeta::Member& m, compound->members)
	{
		auto f = iObject.get_obj().find(m.name);
		if ((f == iObject.get_obj().end()) && (m.name == "@version") && (defaultVersion >= 0)) {
			MIRA_LOG(WARNING) << "JSONToBinary: '@version' member not found in JSON object. "
			                     "Setting default version " << defaultVersion << ".";
			stream << defaultVersion;
			continue;
		}
		const JSONValue& v = f->second;
		JSONToBinary(v,*m.type,metadb,stream,defaultVersion);
	}
}

static void collectionToBinary(const JSONValue& iObject,
                               const TypeMeta& nestedType, const MetaTypeDatabase& metadb,
                               BinaryBufferOstream& stream, int defaultVersion)
{
	checkVersion(nestedType);

	const JSONArray& a = iObject.get_array();

	uint32 count = a.size();
	stream << count;

	for(uint32 i=0; i<count; ++i)
		JSONToBinary(a[i],nestedType,metadb,stream,defaultVersion);
}

static void pointerToBinary(const JSONValue& iObject,
                            const TypeMeta& unqualifiedType, const MetaTypeDatabase& metadb,
                            BinaryBufferOstream& stream, int defaultVersion)
{
	checkVersion(unqualifiedType);

	// null pointer
	if (iObject.is_null())
	{
		stream << (uint8)BinarySerializerMixin::NULL_POINTER;
		return;
	}
	// polymorphic pointer
	if (iObject.type() == json_spirit::obj_type && iObject.get_obj().count("@class"))
	{
		stream << (uint8)BinarySerializerMixin::POLYMORPHIC_POINTER;
		std::string typenam = iObject.get_obj().find("@class")->second.get_str();
		stream << typenam;
		auto it = metadb.find(typenam);
		if(it==metadb.end())
			MIRA_THROW(XIO, "Cannot convert polymorphic pointer of unknown class '" << typenam << "'");

		assert(!unqualifiedType.hasQualifier());
		TypeMeta type;
		type.setClassType(typenam);
		JSONToBinary(iObject,type,metadb,stream,defaultVersion);
		return;
	}

	// TODO reference pointers

	// if we reach here we have a normal pointer
	stream << (uint8)BinarySerializerMixin::NORMAL_POINTER;
	JSONToBinary(iObject, unqualifiedType, metadb, stream,defaultVersion);
}

} // namespace v0

///////////////////////////////////////////////////////////////////////////////

namespace v2 {

static void atomicToBinary(const JSONValue& iObject, const TypeMeta& type,
                           BinaryBufferOstream& stream)
{
	switch(type.getType())
	{
		case TypeMeta::TYPE_ATOMIC:
			MIRA_THROW(XNotImplemented, "Cannot convert user-defined atomic types");
			break;
		case TypeMeta::TYPE_UINT8:  stream << (uint8)iObject.get_int(); break;
		case TypeMeta::TYPE_INT8:   stream << (int8)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT16: stream << (uint16)iObject.get_int(); break;
		case TypeMeta::TYPE_INT16:  stream << (int16)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT32: stream << (uint32)iObject.get_int(); break;
		case TypeMeta::TYPE_INT32:  stream << (int32)iObject.get_int(); break;
		case TypeMeta::TYPE_UINT64: stream << (uint64)iObject.get_int64(); break;
		case TypeMeta::TYPE_INT64:  stream << (int64)iObject.get_int64(); break;
		case TypeMeta::TYPE_FLOAT:  stream << (float)iObject.get_real(); break;
		case TypeMeta::TYPE_DOUBLE: stream << iObject.get_real(); break;
		case TypeMeta::TYPE_BOOL:   stream << iObject.get_bool(); break;
		case TypeMeta::TYPE_STRING: stream << iObject.get_str(); break;
		case TypeMeta::TYPE_ENUMERATION:  stream << iObject.get_int(); break;
		case TypeMeta::TYPE_VERSION: stream << (serialization::VersionType)iObject.get_int(); break;
		default:
			MIRA_THROW(XNotImplemented, "Cannot convert unknown atomic types");
			break;
	}
}

static void objectToBinary(const JSONValue& iObject,
                           const TypeMeta& type, const MetaTypeDatabase& metadb,
                           BinaryBufferOstream& stream);

static void collectionToBinary(const JSONValue& iObject,
                               const TypeMeta& nestedType, const MetaTypeDatabase& metadb,
                               BinaryBufferOstream& stream);

static void pointerToBinary(const JSONValue& iObject,
                            const TypeMeta& unqualifiedType, const MetaTypeDatabase& metadb,
                            BinaryBufferOstream& stream);

static void JSONToBinary(const JSONValue& iObject, const TypeMeta& type,
                         const MetaTypeDatabase& metadb, BinaryBufferOstream& stream)
{
	// handle qualifiers first
	if(type.hasQualifier()) {
		TypeMeta unqualifiedType = type;
		unqualifiedType.removeQualifiers(1); // remove this qualifier

		switch(type.getQualifier())
		{
		case TypeMeta::TYPE_COLLECTION:
			collectionToBinary(iObject, unqualifiedType, metadb, stream);
			break;
		case TypeMeta::TYPE_POINTER:
			pointerToBinary(iObject, unqualifiedType, metadb, stream);
			break;
		default:
			MIRA_THROW(XIO, "Unknown meta type qualifier: " << type.getQualifier());
			break;
		}

	} else {
	// the plain type is remaining only

		switch(type.getType())
		{
		case TypeMeta::TYPE_CLASS:
			objectToBinary(iObject,type,metadb, stream);
			break;
		default:
			atomicToBinary(iObject,type,stream);
			break;
		}
	}
}

static void objectToBinary(const JSONValue& iObject,
                           const TypeMeta& type, const MetaTypeDatabase& metadb,
                           BinaryBufferOstream& stream)
{
	std::string identifier = type.getIdentifier();
	auto it = metadb.find(identifier);
	if(it==metadb.end())
		MIRA_THROW(XIO, "Cannot convert unknown class '" << identifier << "'");

	CompoundMetaPtr compound = it->second;

	foreach(const CompoundMeta::Member& m, compound->members)
	{
		// handle some special members 
		if (m.name == "@version_default") {  // write version placeholder to stream
 			stream << (serialization::VersionType)0; 
		} else if (m.name == "@itemcount") { // count items and write to stream
			const JSONArray& a = iObject.get_array();
			uint32 count = a.size();
			stream << count;
		} else if (m.name == "@items") {     // @items is not a member, make sure items are read from current object!
			sBlockDump = false;
			JSONToBinary(iObject,*m.type,metadb,stream);
		} else if (m.name == "@items_blockdump") {
			sBlockDump = true;
			JSONToBinary(iObject,*m.type,metadb,stream);
		} else if (m.name == "@transparent") {  // parent is a transparent object,
		                                        // this member should appear inline here
			JSONToBinary(iObject,*m.type,metadb,stream);
		} else { 
			auto f = iObject.get_obj().find(m.name);
			const JSONValue& v = f->second;
			JSONToBinary(v,*m.type,metadb,stream);
		}
	}
}

static void collectionToBinary(const JSONValue& iObject,
                               const TypeMeta& nestedType, const MetaTypeDatabase& metadb,
                               BinaryBufferOstream& stream)
{
	const JSONArray& a = iObject.get_array();

	uint32 count = a.size();

	if (sBlockDump) {
		for(uint32 i=0; i<count; ++i) {
			// this will only work for atomic types, otherwise we need either an instance of the type
			// or a type meta based on a plain dump (not based on the type's reflect method)
			atomicToBinary(a[i],nestedType,stream);
		}
	} else {
		for(uint32 i=0; i<count; ++i)
			JSONToBinary(a[i], nestedType, metadb, stream);
	}
}

static void pointerToBinary(const JSONValue& iObject,
                            const TypeMeta& unqualifiedType, const MetaTypeDatabase& metadb,
                            BinaryBufferOstream& stream)
{
	// null pointer
	if (iObject.is_null())
	{
		stream << (uint8)BinarySerializerMixin::NULL_POINTER;
		return;
	}
	// polymorphic pointer
	if (iObject.type() == json_spirit::obj_type && iObject.get_obj().count("@class"))
	{
		stream << (uint8)BinarySerializerMixin::POLYMORPHIC_POINTER;
		std::string typenam = iObject.get_obj().find("@class")->second.get_str();
		stream << typenam;
		auto it = metadb.find(typenam);
		if(it==metadb.end())
			MIRA_THROW(XIO, "Cannot convert polymorphic pointer of unknown class '" << typenam << "'");

		assert(!unqualifiedType.hasQualifier());
		TypeMeta type;
		type.setClassType(typenam);
		JSONToBinary(iObject, type, metadb, stream);
		return;
	}

	// TODO reference pointers

	// if we reach here we have a normal pointer
	stream << (uint8)BinarySerializerMixin::NORMAL_POINTER;
	JSONToBinary(iObject, unqualifiedType, metadb, stream);
}


} // namespace v2

///////////////////////////////////////////////////////////////////////////////

void JSONToBinary(const JSONValue& iObject, const TypeMeta& type,
                  const MetaTypeDatabase& metadb, Buffer<uint8>& oData,
                  bool includeTypename, int defaultVersion)
{
	BinaryBufferOstream stream(&oData);

	uint8 v = type.version();

	if ((v != 0) && (v != 2))
		MIRA_THROW(XIO, "JSONToBinary: Requested type meta version " << (int)v << ". Not implemented.");

	if (v == 2) {
		if (defaultVersion != -1)
			MIRA_THROW(XIO, "JSONToBinary: defaultVersion not supported for v2 meta type.");

		stream << (uint16)BINARY_VERSION_MARKER;
		stream << v;
	}

	if (includeTypename)
		stream << type.getTypename();

	if (v == 0)
		v0::JSONToBinary(iObject, type, metadb, stream, defaultVersion);
	else if (v == 2)
		v2::JSONToBinary(iObject, type, metadb, stream);
}

void JSONToBinaryV1(const JSONValue& iObject, const TypeMeta& type,
                    const MetaTypeDatabase& metadb, Buffer<uint8>& oData,
                    bool includeTypename, int defaultVersion)
{
	BinaryBufferOstream stream(&oData);

	stream << (uint16)BINARY_VERSION_MARKER;
	stream << (uint8)1;

	if (includeTypename)
		stream << type.getTypename();

	v0::JSONToBinary(iObject, type, metadb, stream, defaultVersion);
}

///////////////////////////////////////////////////////////////////////////////

}} // namespace

