/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CallStack.C
 *    Implementation of OS dependent parts in CallStack.h
 *
 * @author Christian Martin
 * @date   2009/12/26
 */

#include <error/CallStack.h>

#include <utils/ToString.h>

#include <error/Logging.h>
#include <error/SystemError.h>
#include <error/private/SymbolTableWin.h>

#include <boost/format.hpp>
#include <boost/algorithm/string/trim.hpp>

using namespace boost;
using namespace std;

#include <Tlhelp32.h>

#ifdef MIRA_WINDOWS

namespace mira {

//////////////////////////////////////////////////////////////////////////////

namespace Private {

//////////////////////////////////////////////////////////////////////////////

vector<STACKFRAME64> backtrace(size_t maxSize)
{
	// Capture the current execution context
	CONTEXT tContext;
	memset(&tContext, 0, sizeof(CONTEXT));
	tContext.ContextFlags = CONTEXT_ALL;
	RtlCaptureContext(&tContext);

	// Initialize the stack frame request
	STACKFRAME64 tFrame;
	memset(&tFrame, 0x00, sizeof(tFrame));

	DWORD tMachineType = 0;
#ifdef _M_IX86
	tMachineType = IMAGE_FILE_MACHINE_I386;
	tFrame.AddrPC.Offset    = tContext.Eip;
	tFrame.AddrPC.Mode      = AddrModeFlat;
	tFrame.AddrFrame.Offset = tContext.Ebp;
	tFrame.AddrFrame.Mode   = AddrModeFlat;
	tFrame.AddrStack.Offset = tContext.Esp;
	tFrame.AddrStack.Mode   = AddrModeFlat;
#elif _M_X64
	tMachineType = IMAGE_FILE_MACHINE_AMD64;
	tFrame.AddrPC.Offset    = tContext.Rip;
	tFrame.AddrPC.Mode      = AddrModeFlat;
	tFrame.AddrFrame.Offset = tContext.Rsp;
	tFrame.AddrFrame.Mode   = AddrModeFlat;
	tFrame.AddrStack.Offset = tContext.Rsp;
	tFrame.AddrStack.Mode   = AddrModeFlat;
#elif _M_IA64
	tMachineType = IMAGE_FILE_MACHINE_IA64;
	tFrame.AddrPC.Offset     = tContext.StIIP;
	tFrame.AddrPC.Mode       = AddrModeFlat;
	tFrame.AddrFrame.Offset  = tContext.IntSp;
	tFrame.AddrFrame.Mode    = AddrModeFlat;
	tFrame.AddrBStore.Offset = tContext.RsBSP;
	tFrame.AddrBStore.Mode   = AddrModeFlat;
	tFrame.AddrStack.Offset  = tContext.IntSp;
	tFrame.AddrStack.Mode    = AddrModeFlat;
#else
# error "Platform not supported!"
#endif

	// Get the stack backtrace
	vector<STACKFRAME64> tStack;
	while (true) {
		BOOL tRes = StackWalk64(tMachineType, GetCurrentProcess(), GetCurrentThread(),
		                        &tFrame, &tContext, NULL,
		                        SymFunctionTableAccess64, SymGetModuleBase64, NULL);
		if (!tRes)
			break;

		if (tFrame.AddrPC.Offset == tFrame.AddrReturn.Offset) {
			MIRA_LOG(ERROR) << "StackWalk64 - Endless callstack.";
			break;
		}

		tStack.push_back(tFrame);
		if (tStack.size() > maxSize) // The current function is not in the stack !!!
			break;
	}

	return(tStack);
}

//////////////////////////////////////////////////////////////////////////////

} // namespace Private

//////////////////////////////////////////////////////////////////////////////

string CallStack::Symbol::sourceLocation() const
{
	SymbolTable::SourceLocation tLoc = SymbolTable::instance().sourceLocation(address);
	if (tLoc.filename().empty())
		return "";
	else
		return tLoc.filename() + ":" + boost::lexical_cast<string>(tLoc.line());
}

///////////////////////////////////////////////////////////////////////////////

CallStack CallStack::backtrace(size_t maxSize, size_t stackStart)
{
	maxSize+=stackStart;
	vector<STACKFRAME64> tStack = Private::backtrace(maxSize);
	return CallStack(tStack, tStack.size(), stackStart);
}

void CallStack::obtainSymbols() const
{
	if (mStack.size() == 0) // we have no valid call stack
		return;

	if (!mSymbols.empty()) // we already have the symbols
		return;

	// parse each symbol
	for(size_t i = 0; i < mStackSize; ++i) {
		Symbol tSymbol;

		DWORD64 tAddr = mStack[i].AddrPC.Offset;

		string tSymName;
		int64 tSymOffset = 0;
		SymbolTable::instance().getSymbolNameFromAddr(tAddr, tSymName, tSymOffset);

		string tUndecoratedName;
		char tUndecoratedNameBuf[1024];
		if (UnDecorateSymbolName(tSymName.c_str(), (char*)&tUndecoratedNameBuf, 
		                         sizeof(tUndecoratedNameBuf), 
		                         UNDNAME_COMPLETE | UNDNAME_NO_ACCESS_SPECIFIERS) != 0)
			tUndecoratedName = string((char*)&tUndecoratedNameBuf);
		else
			tUndecoratedName = tSymName;

		string tModName;
		SymbolTable::instance().getModuleNameFromAddr(tAddr, tModName);

		stringstream tAddrStr;
		tAddrStr << boost::format("0x%x") % tAddr;

		tSymbol.address       = tAddr;
		tSymbol.addressStr    = tAddrStr.str();
		tSymbol.offset        = tSymOffset;
		tSymbol.name          = tSymName;
		tSymbol.nameDemangled = tUndecoratedName;
		tSymbol.file          = tModName;

		mSymbols.push_back(tSymbol);
	}
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
