/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PoolAllocator.h
 *    STL conform memory pool allocator.
 *
 * @author Erik Einhorn
 * @date   2011/11/01
 */

#ifndef _MIRA_POOLALLOCATOR_H_
#define _MIRA_POOLALLOCATOR_H_

#include <assert.h>
#include <malloc.h> // for malloc and free

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#endif

#include <platform/Types.h>
#include <utils/Foreach.h>
#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Provides a pool allocator that is compatible to STL allocators.
 * This allocator is suitable whenever MANY smaller objects are allocated
 * (e.g. several million objects with a size of 32 bytes).
 *
 * This allocator always allocates pools of objects instead of each object
 * separately. This reduces the management overhead for each object. If the
 * current memory pool is exhausted, another pool is allocated. There are
 * two modes for the allocation of new pools:
 * -# each memory pool has the same constant size that is specified within
 *    the constructor
 * -# the size of each newly created pool is doubled (starting with the size
 *    specified in the constructor)
 * This behavior can be configured using the optional flag in the constructor.
 *
 * This allocator uses the "Simple Segregated Storage" pattern, i.e. it
 * manages a list of deallocated objects (free list) within the allocated
 * memory blocks, i.e. no extra memory is reserved for handling the free list.
 *
 * @note Note that this allocator is able to allocate only one object with each
 * call to its allocate method, which however is no real restriction in
 * practice.
 *
 * @note This allocator is 4-5 times faster than the boost pool
 * allocators and fast pool allocator, which should not be used.
 */
template <typename T>
class PoolAllocator : boost::noncopyable
{
public:

	// STL conform typedefs

	/// Element type
	typedef T value_type;

	/// Quantities of elements
	typedef std::size_t size_type;

	/// Difference between two pointers
	typedef std::ptrdiff_t difference_type;

	/// Pointer to element
	typedef T* pointer;

	/// Reference to element
	typedef T& reference;

	/// Reference to element
	typedef const T* const_pointer;

	/// Constant reference to element
	typedef const T& const_reference;


public:

	/**
	 * Creates a new PoolAllocator.
	 * The (initial) size of the pools (in number of objects) is specified as
	 * first parameter. The second parameter specifies the mode for the
	 * allocation of new pools:
	 * -  If it is set to false, each memory pool has the same constant size
	 *    that is specified by the first parameter.
	 * -  If it is set to true, the size of each newly created pool is doubled,
	 *    starting with the size specified by the first parameter
	 * The itemSize parameter allows to specify the size in bytes of each
	 * item manually, by default sizeof(T) is used.
	 */
	PoolAllocator(size_type poolCapacity=32, bool growPools=true, size_type itemSize=sizeof(T)) :
		mFreeList(NULL), mNextCapacity(poolCapacity), mAllocatedBytes(0), mGrowPools(growPools) {
		mItemSize = itemSize > sizeof(FreeListItem*) ? itemSize : sizeof(FreeListItem*);
		newBlock();
	}

	/// Destructor
	~PoolAllocator() {
		// free our memory blocks
		foreach(uint8* p, mMemoryBlocks)
			free(p);
	}

	PoolAllocator(PoolAllocator&& other) :
		mFreeList(NULL), mItemSize(0), mNextCapacity(0), mAllocatedBytes(0), mGrowPools(false) {
		swap(other);
	}

	PoolAllocator& operator=(PoolAllocator&& other) {
		swap(other);
		return *this;
	}


	void swap(PoolAllocator& other)
	{
		std::swap(mMemoryBlocks, other.mMemoryBlocks);
		std::swap(mFreeList, other.mFreeList);
		std::swap(mItemSize, other.mItemSize);
		std::swap(mNextCapacity, other.mNextCapacity);
		std::swap(mAllocatedBytes, other.mAllocatedBytes);
		std::swap(mGrowPools, other.mGrowPools);
	}

public: // Interface to get memory usage
	size_type allocatedMemory() const
	{
		return mAllocatedBytes;
	}

	size_type usedMemory() const
	{
		return mAllocatedBytes - unusedMemory();
	}

	size_type unusedMemory() const
	{
		size_type freeBytes = 0;
		FreeListItem* curr = mFreeList;
		while(curr)
		{
			freeBytes += mItemSize;
			curr = curr->next;
		}
		return freeBytes;
	}

public:

	// STL conform interface:

	/// Returns the address of x.
	pointer address ( reference x ) const {
		return &x;
	}

	/// Returns the address of x.
	const_pointer address ( const_reference x ) const {
		return &x;
	}

	/**
	 * Returns the maximum number of elements of type T (the template parameter)
	 * that could be allocated by a call to member allocate.
	 * @note Note that this allocator can allocate one object only with each
	 * call to allocate (max_size() returns 1)!
	 */
	size_type max_size() const MIRA_NOEXCEPT_OR_NOTHROW {
		return 1;
	}

	/**
	 * Returns the size of each item in bytes.
	 */
	size_type itemSize() const MIRA_NOEXCEPT_OR_NOTHROW {
		return mItemSize;
	}

	/**
	 * Allocate a block of storage and returns the pointer, note that the object
	 * is not constructed.
	 * @note Note that this allocator can allocate one object only with each
	 * call to allocate - trying to allocate more than one object will fail
	 * (return a NULL pointer)!
	 */
	pointer allocate (size_type n, const void* hint = 0) {
		if(n!=1)
			return NULL;

		uint8* memory = NULL;

		// we have no more free chunks, reserve a new block
		if(mFreeList==NULL)
			newBlock();

		assert(mFreeList!=NULL);

		// take first element of free list
		memory = reinterpret_cast<uint8*>(mFreeList);
		// next element becomes new head
		mFreeList = mFreeList->next;

		return (pointer) memory;
	}

	/// Constructs an object of type T on the location p using its copy
	/// constructor to initialize its value to val.
	void construct (pointer p, const_reference val) {
		T* obj = new(p) T(val); // placement new
	}

	/// Destroy an object. Notice that this does not deallocate space for the element.
	void destroy(T* p) {
		p->~T();
	}

	/// Releases a block of storage previously allocated with member allocate 
	/// and not yet released.
	void deallocate (pointer p, size_type n) {
		if(p==NULL)
			return;
		assert(n==1);
		// the deallocated memory will become a free list item
		FreeListItem* item = reinterpret_cast<FreeListItem*>(p);
	
		// and it becomes the new head of that list (next points to the current head)
		item->next = mFreeList;
		mFreeList = item;
	}

	/**
	 * Allocates and constructs a new object and returns its pointer.
	 * This method is provided for convenience but may be up to 10% faster
	 * than the STL conform equivalent:
	 * \code
	 *     T* obj = a.allocate(1);
	 *     a.construct(obj, T());
	 * \endcode
	 */
	T* construct() {
		T* obj = allocate(1);
		return new(obj) T();
	}

	/**
	 * Allocates a new feature and returns its pointer
	 * This method is provided for convenience. It is equivalent to:
	 * \code
	 *     T* obj = a.allocate(1);
	 *     a.construct(obj, val);
	 * \endcode
	 */
	T* construct(const_reference val) {
		T* obj = allocate(1);
		return new(obj) T(val);
	}

	/**
	 * Destroys the object and deallocates the memory.
	 *
	 * Provided for convenience and as counterpart to the two above
	 * construct methods.
	 */
	void destruct(pointer p) {
		if(p==NULL)
			return;
		destroy(p);
		deallocate(p,1);
	}

private:

	// creates a new big memory block
	void newBlock() {
		const size_type blockSize = mNextCapacity * mItemSize;
		uint8* currentBlock = (uint8*) malloc(blockSize);
		if(currentBlock==NULL)
			MIRA_THROW(XBadAlloc, "Failed to allocate "
					<< blockSize << " bytes of memory");

		// build free list within the allocated block
		FreeListItem* oldHead = mFreeList;

		// head of new free list
		uint8* p = currentBlock;
		FreeListItem* l = reinterpret_cast<FreeListItem*>(p);
		mFreeList = l;
		p+=mItemSize;

		// each element links to the next one
		for(size_type i=1; i<mNextCapacity; ++i, p+=mItemSize)
		{
			l->next = reinterpret_cast<FreeListItem*>(p);
			l = l->next;
		}
		l->next = oldHead;

		mMemoryBlocks.push_back(currentBlock);
		mAllocatedBytes += blockSize;

		if(mGrowPools)
			mNextCapacity = mNextCapacity*2;
	}

private:

	std::list<uint8*> mMemoryBlocks;

	struct FreeListItem {
		FreeListItem* next;
	};
	FreeListItem* mFreeList;

	size_type mItemSize;
	size_type mNextCapacity;
	size_type mAllocatedBytes;
	bool mGrowPools;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
