/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Serializer.h
 *    Contains the Serializer template, a base class for all serializers.
 *
 * @author Erik Einhorn
 * @date   2010/07/03
 */

#ifndef _MIRA_SERIALIZER_H_
#define _MIRA_SERIALIZER_H_

#include <set>
#include <map>

#ifndef Q_MOC_RUN
#include <boost/type_traits/is_fundamental.hpp>
#include <boost/type_traits/is_pointer.hpp>
#endif

#include <serialization/RecursiveMemberReflector.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 *
 * Is a special reflector that is used for serialization.
 *
 * It usually is used as a base class for concrete Serializers like the
 * XMLSerializer or the BinarySerialzer.
 *
 * It implements the required RecursiveMemberReflectorBase visiting methods.
 * If you derive from this class and if you overwrite these methods, you MUST
 * call the methods of the Serializer base class!
 *
 * The Serializer implements Object-Tracking, i.e. it is aware of all objects
 * that were serialized yet. If it is called for a pointer that was already
 * serialized, it stores a pointer reference to the previously stored object
 * instead, otherwise it stores the object the pointer points to. If it detects
 * a pointer to a previously stored object, it calls the pointerReference()
 * method which should be overwritten in the derived concrete serializer.
 * This method gets two parameters. The first parameter is the full id to the
 * object that was already stored. The id consists of the id of the member and
 * all names of the enclosing objects, e.g. in the following example the int
 * member "i" will have the full id: "test.object1.member"
 *
 * \code
 * class Class1
 * {
 *     void reflect(...)
 *     {
 *         member("member", i);
 *     }
 *
 *     int i
 * }
 *
 * class Class2 {
 *     Class1 c;
 *
 *     void reflect(...)
 *     {
 *         member("object1", c);
 *     }
 * }
 *
 * Class2 obj.
 * serialize(..., "test", obj);
 * \endcode
 *
 * The derived concrete Serializer should store the reference somehow to be
 * able to restore it by the Deserializer.
 *
 * <b>Implements from base classes:</b>
 * trackObject(), pushObjectTrackingStore(), popObjectTrackingStore(),
 * pointer(), pointerNormal(), pointerPolymorphic(),
 * pointerAbstract()
 *
 * <b>Methods that should be implemented:</b>
 * pointerReference(), pointerNull(), pointerWithoutClassType(),
 * pointerWithClassType()
 *
 * @see @ref SerializationPage
 */
template <typename Derived>
class Serializer : public RecursiveMemberReflector<Derived>
{
	typedef RecursiveMemberReflector<Derived> Base;

public:

	typedef boost::mpl::bool_<true> isReadOnly;
	typedef boost::mpl::bool_<true> isObjectTrackingSupported;

public:

	/// The constructor.
	Serializer()
	{}

	/**
	 * Returns either the version number from value of environment variable
	 * 'MIRA_FORCE_SERIALIZE_VERSION', or -1 (= do not force a version).
	 */
	static int forcedSerializeVersion()
	{
		// make sure the resolve is only ever done once
		static int version = Base::checkForcedVersion("MIRA_FORCE_SERIALIZE_VERSION");
		return version;
	}

public:

	typedef typename Base::VersionType VersionType;
	typedef typename Base::ClassVersionMap ClassVersionMap;

	/// implements ReflectorInterface (for documentation see ReflectorInterface)
	void desireClassVersions(const ClassVersionMap& versions) { mDesiredClassVersions = versions; }

protected:

	VersionType queryDesiredClassVersion(VersionType version, const std::string& type,
	                                     bool acceptDesiredVersion = false) {
		auto it = mDesiredClassVersions.find(type);
		if (it == mDesiredClassVersions.end())
			return version; // no entry = default case, just use version

		if (!acceptDesiredVersion && it->second != version) {
			MIRA_LOG(WARNING) << "Desiring version for type '" << type <<
			                     "': " << (int)(it->second) << ", but desiring a version "
			                     "is not supported by the type's serialization. "
			                     "Using type's current version " << (int)version << " instead.";
			return version;
		}

		if (it->second > version) {
			MIRA_LOG(WARNING) << "Desiring version for type '" << type <<
			                     "': " << (int)(it->second) << ", but current version "
			                     "(=maximum implemented version) is: " << (int)version <<
			                     ". Using that current version instead.";
			return version;
		}

		return it->second;
	}

	template <typename T>
	VersionType queryDesiredClassVersion(VersionType version,
	                                     bool acceptDesiredVersion = false) {
		return queryDesiredClassVersion(version, typeName<T>(), acceptDesiredVersion);
	}

	ClassVersionMap mDesiredClassVersions;

public:

	/**
	 * Serializes the specified object value under the given name.
	 *
	 * If you overwrite this method in a derived class you usually MUST call
	 * this method of the base class.
	 */
	template <typename T>
	void serialize(const std::string& name, const T& value,
	               const std::string& comment="")
	{
		// reset object tracking etc.
		mObjects.clear();
		mObjectIDToName.clear();

		// cast away the constness, this is safe since all serializer
		// will read from the object only but have to use the
		// reflect interface which is non-const, since the same interface
		// is used for deserialization
		T& value_const = const_cast<T&>(value);
		this->This()->member(name.c_str(), value_const, comment.c_str());
	}

public:

	// implement methods of RecursiveMemberReflector

	template<typename T>
	void trackObject(T& member)
	{
		// only if tracking is enabled
		if(this->template isTrackingEnabled<T>())
			addObject(member);
	}

	template<typename T>
	void pointer(T* &pointer)
	{
		// #####################################################################
		// If you get a compiler error here, you tried to serialize a pointer on
		// a fundamental data type like int, float, etc. For performance reasons
		// this is not allowed. Use a wrapper for those fundamental types
		// to serialize the pointer
		// #####################################################################
		static_assert(!boost::is_fundamental<T>::value,
				"Pointers on fundamental types cannot be serialized");

		// #####################################################################
		// If you get a compiler error here, you tried to serialize a pointer on
		// a pointer. This is not allowed and can not be deserialized. Try to
		// change your code to avoid pointers on pointers!
		// #####################################################################
		static_assert(!boost::is_pointer<T>::value,
				"Pointers on pointers cannot be serialized");

		// check if we have a NULL pointer
		if(pointer==NULL) {
			this->This()->pointerNull();
			return;
		}

		Base::pointer(pointer);
	}

	template<typename T>
	void pointerNormal(T* &pointer, int typeId) {
		// catch if we found an already stored pointer
		if(checkForPointerReference<T>(pointer, typeId))
			return;

		this->This()->pointerWithoutClassType();

		// otherwise let our RecursiveMemberReflector handle the pointer
		Base::pointerNormal(pointer, typeId);
	}

	template<typename T>
	void pointerPolymorphic(T* &pointer, int typeId) {
		// catch if we found an already stored pointer
		if(checkForPointerReference<T>(pointer, typeId))
			return;

		// otherwise obtain the class identifier ...
		const Class& c = pointer->getClass();
		this->This()->pointerWithClassType(c.getIdentifier());

		// ... and let our RecursiveMemberReflector handle the pointer
		Base::pointerPolymorphic(pointer, typeId);
	}


	template<typename T>
	void pointerAbstract(T* &pointer, int typeId) {
		// #####################################################################
		// If you get a compiler error here, you tried to serialize an abstract
		// object that was not inherited from mira::Object. To resolve this issue
		// you should inherit your class from mira::Object using the class
		// factory
		// #####################################################################
		static_assert(sizeof(T)==0, "You tried to serialize an abstract class "
		              "that is not an mira::Object");
	}

	/** @name Special Pointer Handlers For Overwriting */
    //@{

	/**
	 * Is called by the pointer() method if an object, a pointer references,
	 * was already serialized and a reference to that object must be stored
	 * instead of the whole object. This method can be implemented by
	 * derived concrete serializers to store pointer references.
	 */
	void pointerReference(int referencedObjectID)
	{
	}

	/**
	 * Is called by the pointer() method to indicate that a pointer that
	 * is to be serialized is a NULL-pointer. It can be implemented by
	 * derived concrete serializers to handle the special NULL-pointer case.
	 */
	void pointerNull()
	{
	}

	/**
	 * Is called if a pointer points to a non-polymorphic type and hence the
	 * object can be stored without specifying a class type. It can be
	 * implemented by derived concrete serializers to indicate that no pointer
	 * reference and no class type needs to be stored and the full object will
	 * follow afterwards.
	 * (this is used by the BinarySerializer)
	 */
	void pointerWithoutClassType()
	{
	}

	/**
	 * Is called if a pointer points to a polymorphic type and hence the object
	 * must be stored with specifying a class type. It can be implemented by
	 * derived concrete serializers to indicate that no pointer reference is
	 * stored, but that the full object is serialized afterwards and to store
	 * the class type that is needed for proper deserialization.
	 */
	void pointerWithClassType(const std::string& type)
	{
	}

	//@}

protected:

	/// Returns true, if object tracking is enabled for the type T
	template<typename T>
	bool isTrackingEnabled() const {
		return Derived::isObjectTrackingSupported::value &&
			   IsObjectTrackable<T>::value &&
			   (this->template isReflectedAsPointer<T>() ||
				    std::is_base_of<mira::Object, T>::value);
	}

protected:

	/**
	 * Is used to store the type and the address of all previously
	 * serialized objects in a map.
	 * The type is an id that is obtained by the typeId().
	 * typeId() gives a unique id that is unique for a single run
	 * of the program. It may change between different starts
	 * of the application and is only used to check for the
	 * same types here. And to resolve the following case
	 *
	 * \code
	 * class Class1
	 * {
	 *     void reflect(...)
	 *     {
	 *         member("member", i);
	 *     }
	 *
	 *     int i
	 * }
	 *
	 * class Class2 {
	 *     Class1 c;
	 *
	 *     void reflect(...)
	 *     {
	 *         member("object1", c);
	 *     }
	 * }
	 *
	 * \endcode
	 *
	 * When serialized, Class2::c has the same address as
	 * Class1::i. If we would use the address only, we cannot
	 * distinguish between both, therefore we use the typeId
	 * which will be different, since Class2::c is of type Class1
	 * and Class1:i is of type int.
	 */
	struct AObject
	{
		void* address;
		int   type;
		int   objectID;

		AObject(void* iAddress, int iType) :
			address(iAddress), type(iType) {}

		bool operator<(const AObject &rhs) const
		{
			assert(address != NULL);
			assert(rhs.address != NULL);
			if( address < rhs.address )
				return true;
			if( address > rhs.address )
				return false;
			// if addresses are the same then we use the typeId
			// to distinguish
			return type < rhs.type;
		}

		AObject & operator=(const AObject & rhs)
		{
			address = rhs.address;
			type = rhs.type;
			objectID = rhs.objectID;
			return *this;
		}
	};

	typedef std::set<AObject> ObjectSet;
	ObjectSet mObjects; // the objects that we have stored already

	typedef std::map<int, std::string> ObjectIDToNameMap;
	ObjectIDToNameMap mObjectIDToName;

	typedef std::pair<ObjectSet, ObjectIDToNameMap> TrackingState;
	std::stack<TrackingState> mTrackingStack;

public:

	void pushObjectTrackingStore()
	{
		mTrackingStack.push(std::make_pair(mObjects, mObjectIDToName));
	}

	void popObjectTrackingStore()
	{
		std::tie(mObjects, mObjectIDToName) = mTrackingStack.top();
		mTrackingStack.pop();
	}

protected:

	/**
	 * Returns the full human readable object id / name for the given
	 * internal objectID. The human readable id only is available, if
	 * the Reflector has set useHumanReadableIDs to true.
	 */
	const std::string& getHumanReadableFullID(int objectID) const
	{
		assert(this->usesHumanReadableIDs());
		std::map<int, std::string>::const_iterator it = mObjectIDToName.find(objectID);
		assert(it!=mObjectIDToName.end());
		return it->second;
	}

private:

	/**
	 * Checks, if the object the pointer points on was already stored.
	 * If so, calls pointerReference() and returns true. Otherwise false.
	 */
	template<typename T>
	bool checkForPointerReference(T* pointer, int typeId)
	{
		if(this->template isTrackingEnabled<T>()) {
			// check if we have serialized the object already
			AObject o(serialization::void_upcast(pointer), typeId);
			typename ObjectSet::iterator i = mObjects.find(o);
			if(i!=mObjects.end()) {
				// pointer was already serialized, so store reference only
				// the id the referenced object is the value in the map
				this->This()->pointerReference(i->objectID);
				return true;
			}
		}
		return false;
	}

	/**
	 * Adds a new object to the object tracking system.
	 */
	template<typename T>
	void addObject(T& member)
	{
		assert(this->template isTrackingEnabled<T>());

		// create new object entry for the object's address and type
		AObject o( serialization::void_upcast(&member), typeId<T>());
		o.objectID = mObjects.size();

		// try to add the object
		std::pair<typename ObjectSet::const_iterator, bool> res = mObjects.insert(o);

		// make sure that an object of THAT type does not exists at this
		// address yet (otherwise we have a pointer conflict)
		if(!res.second) {
			// we have serialized the object already
			const ReflectMemberMeta& meta = this->getCurrentMemberMeta();
			std::string objectName = meta.getName();
			if(this->usesHumanReadableIDs())
				objectName = this->getCurrentMemberFullID();

			MIRA_THROW(XIO, "Pointer conflict: the object '" << objectName <<
					       "' at address " << &member <<
					       " was serialized already (probably by a pointer). "
					       "Try to resolve this conflict by serializing the "
					       "object BEFORE the pointer that points to that "
					       "object.");
		}

		// store the full id/name of the object for the object id
		if(this->usesHumanReadableIDs())
			mObjectIDToName.insert(std::make_pair(o.objectID, this->getCurrentMemberFullID()));
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
