/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertySerializer.h
 *    Serializer that handles properties and creates PropertyNodes.
 *
 * @author Erik Einhorn, Christof Schröter
 * @date   2010/12/31
 */

#ifndef _MIRA_PROPERTYSERIALIZER_H_
#define _MIRA_PROPERTYSERIALIZER_H_

#include <assert.h>

#include <serialization/PropertyReflector.h>
#include <serialization/PropertyNode.h>

#include <tuple>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * A special PropertyReflector that creates a PropertyNode for each reflected
 * property. Moreover, it updates existing property nodes, if a property is
 * updated and reflected again.
 */
class PropertySerializer : public PropertyReflector<PropertySerializer>
{
	typedef PropertyReflector<PropertySerializer> Base;

public:

	PropertySerializer() : mCurrentNode(NULL)
	{}

public:

	/**
	 * Is called by invokePropertyMemberOverwrite() and will call
	 * the actual Base::invokePropertyMemberOverwrite() method that
	 * recursively examines all properties.
	 * This method is a helper which is overloaded below, to avoid
	 * recursive examination for properties with Getters and Setters which
	 * is not implemented yet.
	 */
	template <typename T>
	void continueWithBaseInvoke(T& member, const ReflectMemberMeta& meta,
	                            PropertyHint&& hint, bool isReadOnly,
	                            bool isVolatile) {
		Base::invokePropertyMemberOverwrite(member, meta, std::move(hint),
		                                    isReadOnly, isVolatile);
	}

	/**
	 * Specialization of the above method for properties with getters and
	 * setters. For those, a recursive reflection is not yet implemented.
	 */
	template <typename Getter, typename Setter>
	void continueWithBaseInvoke(Accessor<Getter,Setter>& accessor,
	                            const ReflectMemberMeta& meta, PropertyHint&& hint,
	                            bool isReadOnly, bool isVolatile) {
	}

	// implemented from base class

	/**
	 * Invokes the PropertyReflector on the specified member.
	 * If the corresponding property node for the member exists already, its
	 * content and sub-nodes will be updated. Otherwise, a new property node
	 * is created. All sub-properties of the member are invoked recursively.
	 */
	template<typename T>
	void invokePropertyMemberOverwrite(T& member, const ReflectMemberMeta& meta,
	                                   PropertyHint&& hint, bool isReadOnly,
	                                   bool isVolatile)
	{
		// current node becomes parent node
		PropertyNode* parentNode = mCurrentNode;

		if (meta.id)
		{
			PropertyNode* node = NULL;

			if (parentNode)
				node = parentNode->findChildNode(meta.id);

			if (node) {
				TypedPropertyNodeImpl<T>* typedNode =
								dynamic_cast<TypedPropertyNodeImpl<T>*>(node);
				// if node already exists and has the right type, just update it
				if (typedNode) {
					typedNode->update(member);

					typedNode->mUpdated = true;
					foreach (auto c, typedNode->children())
						c->mUpdated = false;
				} else {
					// We must NOT delete node here, to be able to signal the removal to
					// listeners first.
					// After setting the node to NULL, a new node will be created next.
					// That will result in parentNode having 2 children with same name and id
					// temporarily, but the older one will be removed in the cleanup below, as
					// it was not updated.
					node = NULL;
				}
			}
			if (!node) // if we cannot use an existing node, create a new one
				node = createNode(parentNode, member, meta, hint, isReadOnly, isVolatile);

			// set the node as current node
			mCurrentNode=node;
		}

		// now do the real reflector invoke (but only if T is a type that
		// is worth to continue. If it is a GetterSetter, we stop here)
		continueWithBaseInvoke(member,meta, std::move(hint), isReadOnly, isVolatile);

		// remove child nodes that have not been updated during serialization --> are obsolete

		// first, collect all child nodes to be removed (index + iterator + PropertyNode)
		int index = 0;
		std::vector<std::tuple<int, PropertyNode::NodeList::iterator, PropertyNode*>> toRemove;
		for (PropertyNode::NodeList::iterator it = mCurrentNode->children().begin(); it != mCurrentNode->children().end(); ++it, ++index) {
			if (!(*it)->mUpdated)
				toRemove.push_back(std::make_tuple(index, it, *it));
		}

		// second, go through them back to front (making sure iterators remain valid)
		// remove contiguous subsequences as batch (listeners can profit from handling this as one merged update)
		int count = 1;
		for (auto i = toRemove.rbegin(); i != toRemove.rend(); ++i) {
			auto inext = std::next(i);
			if ((inext != toRemove.rend()) && (std::get<0>(*i) == std::get<0>(*inext)+1)) { // next one will be the neighbour index -> collect sequence
				++count;
				continue;
			}

			mCurrentNode->removeChildren(std::get<0>(*i), std::get<1>(*i), count);
			count = 1;
		}
	
		// finally (after nodes are removed from the parent and all listeners updated), actually delete these nodes
		for (auto i = toRemove.rbegin(); i != toRemove.rend(); ++i) {
			PropertyNode* p = std::get<2>(*i);
			delete p;
		}

		// go back to the parent node (but only if it is not NULL, otherwise we
		// did not have a root node, so keep the top most node that we have
		// created, it will be returned by reflectProperties)
		if(parentNode)
			mCurrentNode=parentNode;
	}


	void itemName(const std::string& name) {
		if(mCurrentNode) {
			// replace items only
			if(mCurrentNode->mID.compare(0,4,"item")==0)
				mCurrentNode->setName(name);
		}
	}

public:

	// public entry points

	/**
	 * Reflects the properties of the specified 'object'. The properties will
	 * be stored as a sub-node of 'root'. The name of that sub-node will be
	 * 'name'.
	 */
	template <typename T>
	void reflectProperties(PropertyNode* root, const std::string& name, T& object) {
		mCurrentNode=root;
		this->property(name.c_str(), object, "");
	}

	/**
	 * Same as above, but additionally, a unique id can be specified for the
	 * object. In the above method, the id will be equal to the name.
	 */
	template <typename T>
	void reflectProperties(PropertyNode* root, const std::string& name,
	                       const std::string& id, T& object) {
		mCurrentNode=root;
		this->property(name.c_str(), id, object, "");
	}

	/**
	 * Reflects the properties of the specified 'object'. In contrast to the
	 * above methods, the created property sub-node is not added to any
	 * parent node. Instead it is returned by the method.
	 * The name of the created property node can be specified by 'name'.
	 */
	template <typename T>
	PropertyNode* reflectProperties(const std::string& name, T& object) {
		mCurrentNode=NULL;
		this->property(name.c_str(), object, "");
		assert(mCurrentNode!=NULL);
		return mCurrentNode;
	}

	/**
	 * Same as above, but additionally, a unique id can be specified for the
	 * object. In the above method, the id will be equal to the name.
	 */
	template <typename T>
	PropertyNode* reflectProperties(const std::string& name,
	                                const std::string& id, T& object) {
		mCurrentNode=NULL;
		this->property(name.c_str(), id, object, "");
		assert(mCurrentNode!=NULL);
		return mCurrentNode;
	}


private:

	template<typename T>
	PropertyNode* createNode(PropertyNode* parentNode, T& member, const ReflectMemberMeta& meta,
	                         PropertyHint& hint, bool& isReadOnly, bool& isVolatile)
	{
		if (parentNode) { // inherit read only and volatile flags
			isReadOnly |= parentNode->isReadOnly();
			isVolatile |= parentNode->isVolatile();
		}

		PropertyNode* node = new TypedPropertyNodeImpl<T>(meta.id, meta.name,
		                                                  meta.comment, member,
		                                                  isReadOnly, isVolatile);

		// set the given hint to the node (by swapping the contents)
		std::swap(node->mHint, hint);

		if (parentNode)
			parentNode->addChild(node);

		return node;
	}

public:

	PropertyNode* mCurrentNode;
};

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline void TypedPropertyNodeImpl<T>::synchronize()
{
	if (!this->parent())
		MIRA_THROW(XInvalidParameter, "PropertyNode::synchronize() requires a valid parent node!");

	PropertySerializer s;
	s.reflectProperties(this->parent(), this->name().c_str(),
	                    this->id(), *this->value());
}


// specialization for properties of pointers
template <typename T>
inline void TypedPropertyNodeImpl<T*>::synchronize()
{
	if (!this->parent())
		MIRA_THROW(XInvalidParameter, "PropertyNode::synchronize() requires a valid parent node!");

	PropertySerializer s;
	s.reflectProperties(this->parent(), this->name().c_str(),
	                    this->id(), *this->pointer());
}

// specialization for properties with setter
template <typename Getter, typename Setter>
inline void TypedPropertyNodeImpl<Accessor<Getter,Setter>>::synchronize()
{
	if (!this->parent())
		MIRA_THROW(XInvalidParameter, "PropertyNode::synchronize() requires a valid parent node!");

	PropertySerializer s;
	s.reflectProperties(this->parent(), this->name().c_str(),
	                    this->id(), *this->accessor());
}

///@cond INTERNAL

// specializations for STL containers

namespace serialization { // our private namespace

///////////////////////////////////////////////////////////////////////////////

template<typename Reflector, typename Container> struct ReflectReadSetItems;

/**
 * Specialization for Property Serializer which creates read-only properties
 * for set elements instead of members (which would be ignored by this reflector).
 */
template<typename Container>
struct ReflectReadSetItems<PropertySerializer, Container>
{
	typedef typename Container::value_type type;

	static void reflect(PropertySerializer& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(const type& v, c)
		{
			r.roproperty("item", "item["+toString(id)+"]", v, "",
			             PropertyHint(), REFLECT_CTRLFLAG_VOLATILE);
			++id;
		}
	}
};

template<typename Reflector, typename Container> struct ReflectReadMapItems;

/**
 * Specialization for Property Serializer which creates read-only 'key' properties
 * for map keys and 'item' properties for items, instead of pairs (which do not seem
 * to have a proper editor delegate?)
 */
template<typename Container>
struct ReflectReadMapItems<PropertySerializer, Container>
{
	typedef typename Container::value_type  value_type;

	static void reflect(PropertySerializer& r, Container& c)
	{
		// store each item
		int id=0;
		foreach(value_type& p, c)
		{
			r.roproperty("key", "key["+toString(id)+"]", p.first, "",
			             PropertyHint(), REFLECT_CTRLFLAG_VOLATILE);

			// the values can be reflected/deserialized with tracking
			MIRA_PROPERTY_WITH_ID(r, "item", "item["+toString(id)+"]", p.second, "",
			                      PropertyHint(), REFLECT_CTRLFLAG_VOLATILE);

			++id;
		}
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif /* _MIRA_PROPERTYSERIALIZER_H_ */
