/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DefaultInitializer.h
 *    Provides the DefaultInitializer.
 *
 * @author Erik Einhorn
 * @date   2011/08/17
 */

#ifndef _MIRA_DEFAULTINITIALIZER_H_
#define _MIRA_DEFAULTINITIALIZER_H_

#include <serialization/AbstractReflector.h>
#include <serialization/IgnoreMissing.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Initializes the members of this instance using the default values specified
 * within the reflect method. This macro should be used within the constructor.
 */
#define MIRA_INITIALIZE_THIS mira::DefaultInitializer::initialize(*this)

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup SerializationModule
 * Reflector that visits the reflect method of objects in order to initialize
 * them using the specified default values.
 *
 * Usage:
 * \code
 * #include <serialization/DefaultInitializer.h>
 *
 * class MyClass
 * {
 * public:
 *     MyClass() {
 *         // initialize our members using their default values
 *         MIRA_INITIALIZE_THIS;
 *     }
 *
 *     template <typename Reflector>
 *     void reflect(Reflector& r)
 *     {
 *         // default value of mMember is 123.45f
 *         r.member("MyMember", mMember, "", 123.45f);
 *     }
 * };
 *
 * \endcode
 */
class DefaultInitializer : public AbstractReflector<DefaultInitializer>
{
public:

	/**
	 * Invokes the reflector and initializes the members of 'object' with the
	 * specified default values.
	 */
	template <typename T>
	static void initialize(T& object)
	{
		DefaultInitializer r;
		r.invoke(object);
	}

public:

	template<typename T, typename U>
	void member(const char* name, T& member, const char* comment,
	            const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		initializeMember(member,defaultValue);
	}

	template<typename T, typename U>
	void member(const char* name, const T& member, Setter<T> setter,
	            const char* comment, const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		initializeMember(setter,defaultValue);
	}

	template<typename T, typename U>
	void member(const char* name, Getter<T> getter, Setter<T> setter,
	            const char* comment, const U& defaultValue,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		initializeMember(setter,defaultValue);
	}

	template<typename T, typename U>
	void property(const char* name, T& member, const char* comment,
	              const U& defaultValue, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		initializeMember(member,defaultValue);
	}

	template<typename T, typename U>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		initializeMember(setter,defaultValue);
	}

	template<typename T, typename U>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, const U& defaultValue,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {
		initializeMember(setter,defaultValue);
	}

public:

	// the methods without default values, that should be inherited from ReflectorInterfaces
	// (however, they are not inherited, since we have template methods of the same
	//  name defined above)
	template<typename T>
	void member(const char* name, T& member, const char* comment,
	            ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void member(const char* name, const std::string& id, T& member,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void member(const char* name, const T& member, Setter<T> setter,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void member(const char* name, Getter<T> getter, Setter<T> setter,
	            const char* comment, ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void property(const char* name, T& member, const char* comment,
	              PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void property(const char* name, const std::string& id, T& member,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
		          ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void property(const char* name, const T& member, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}
	template<typename T>
	void property(const char* name, Getter<T> getter, Setter<T> setter,
	              const char* comment, PropertyHint&& hint = PropertyHint(),
	              ReflectCtrlFlags flags = REFLECT_CTRLFLAG_NONE) {}

private:


	template<typename T, typename U>
	void initializeMember(T& member, const U& defaultValue) {
		// #############################################################
		// If you get a compiler error here, your default value does not
		// match to your member type and cannot be casted. Make sure
		// that the data type of your default value can be casted to
		// the data type of your member!
		// #############################################################
		member=defaultValue;
	}

	// Specialization for serialization::IgnoreMissing
	template<typename T>
	void initializeMember(T& member, const serialization::IgnoreMissing&) {}

};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
