/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Colormap.h
 *	This file contains classes for different color colormaps
 *	(like Jet, HSV, etc.)
 *
 * @author Erik Einhorn
 * @date   2010/10/21
 */

#ifndef _MIRA_COLORMAP_H_
#define _MIRA_COLORMAP_H_

#include <factory/Factory.h>
#include <image/Color.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for color colormaps.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT Colormap : public Object
{
	MIRA_OBJECT(Colormap)
public:

	/**
	 * @brief Returns the number of elements in the colormap.
	 * Some colormaps like ContinuousColormap can
	 * be resized dynamically.
	 */
	virtual std::size_t size() const = 0;

	/**
	 * Returns the color in the colormap at the given index.
	 */
	virtual Color::RGB get(std::size_t idx) const = 0;

	/**
	 * Returns the color in the colormap at the given index.
	 */
	Color::RGB operator()(std::size_t idx) {return get(idx);}

	/**
	 * Returns the color in the colormap at the given index.
	 */
	Color::RGB operator[](std::size_t idx) {return get(idx);}

public:

	/**
	 * Iterator that can iterate over the whole color colormap similar to
	 * STL iterators on containers. Each iterator points to a certain color
	 * in the colormap.
	 */
	class iterator
	{
	public:
		typedef Color::RGB value_type;
		typedef int difference_type;

		typedef const Color::RGB* pointer;
		typedef const Color::RGB& reference;
		typedef std::bidirectional_iterator_tag iterator_category;

	protected:
		friend class Colormap;
		iterator(const Colormap* colormap, std::size_t position) :
		mColormap(colormap), mPosition(position) {}

	public:
		iterator() : mColormap(NULL), mPosition(0) {}
		iterator(const iterator& other) :
		mColormap(other.mColormap), mPosition(other.mPosition) {}

	public:

		/// advance the iterator
		iterator& operator++() { // pre-increment
			++mPosition;
			return *this;
		}

		/// advance the iterator
		iterator operator++(int) { // post-increment
			iterator old = *this;
			mPosition++;
			return old;
		}

		/// go to previous color
		iterator& operator--() { // pre-decrement
			--mPosition;
			return *this;
		}

		/// go to previous color
		iterator operator--(int) { // post-decrement
			iterator old = *this;
			mPosition--;
			return old;
		}

		bool operator==(const iterator& other) {
			return mColormap==other.mColormap && mPosition==other.mPosition;
		}

		bool operator!=(const iterator& other) {
			return !operator==(other);
		}

	public:

		/// Dereference to the color, the iterator is pointing to
		reference operator*() const {
			assert(mColormap!=NULL);
			mCache = mColormap->get(mPosition);
			return mCache;
		}

		/// Dereference to the color, the iterator is pointing to
		pointer operator->() const {
			assert(mColormap!=NULL);
			mCache = mColormap->get(mPosition);
			return &mCache;
		}

	private:
		const Colormap* mColormap;
		std::size_t mPosition;
		/// necessary for correct implementation of the iterators * and -> operator
		mutable Color::RGB mCache;
	};

	typedef iterator const_iterator;

public:

	/// Gives an iterator to the beginning of the colormap
	iterator begin() {return iterator(this,0);}

	/// Gives a const iterator to the beginning of the colormap
	const_iterator begin() const {return const_iterator(this,0);}

	/// Gives an iterator to the end of the colormap
	iterator end() {return iterator(this,size());}

	/// Gives an const iterator to the end of the colormap
	const_iterator end() const {return const_iterator(this,size());}
};

///////////////////////////////////////////////////////////////////////////////

/// base class for discrete color colormaps
class MIRA_BASE_EXPORT DiscreteColormap : public Colormap
{
	MIRA_OBJECT(DiscreteColormap)
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for continuous color colormaps.
 * In contrast to discrete color colormaps, where the number of colors is
 * limited, continuous color colormaps are able to generate colors for an
 * arbitrary value between 0 and 1 using its getf() method. Hence they are
 * able to generate very fine grained color gradients.
 *
 * To be compatible with other discrete color colormaps, a continuous colormap
 * can also be set to a certain size with a fixed number of colors using
 * the constructor or the resize() method. The Colors can then be accessed by
 * an integer index using the get() method. However, the access to the
 * continuous colors is always possible via the getf() method.
 *
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT ContinuousColormap : public Colormap
{
	MIRA_OBJECT(ContinuousColormap)
public:
	ContinuousColormap(std::size_t size=256) : mSize(size) {}

public:

	/// Sets a certain size of this colormap.
	void resize(std::size_t pSize) {mSize=pSize;}

	/// Returns the set size.
	virtual std::size_t size() const {return mSize;}

	/// Returns the color at the specified index (must be >= 0 and < size)
	virtual Color::RGB get(std::size_t idx) const {
		return getf((float)idx / (float)size());}

public:

	/**
	 * Accesses the continuous colors. The specified floating point value must
	 * be within the interval [0,1].
	 */
	virtual Color::RGB getf(float f) const = 0;

private:
	std::size_t mSize;
};

/**
 * Internal base class for linear gradient color maps.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT GradientColormapBase : public ContinuousColormap
{
public:

	GradientColormapBase(std::size_t size) : ContinuousColormap(size) {}

	virtual Color::RGB getf(float f) const;

protected:

	/**
	 * Sets a color key at the given position with the given color.
	 * The given position must be in the range 0 to 1.
	 */
	void setColorAt(float position, const Color::RGB& color);

	void setColorAt(float position, int r, int g, int b) {
		setColorAt(position, Color::RGB(r/255.0f,g/255.0f,b/255.0f));
	}


	/// Removes all color keys.
	void clear();

private:
	std::map<float, Color::RGB> mColorKeys;
};

/**
 * A class for creating continuous color maps based on linear gradients
 * between predefined points. Each point represents a color at a specific
 * position within the color gradient.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT GradientColormap : public GradientColormapBase
{
public:

	GradientColormap(std::size_t size=256) : GradientColormapBase(size) {}

	/**
	 * Sets a color key at the given position with the given color.
	 * The given position must be in the range 0 to 1.
	 */
	void setColorAt(float position, const Color::RGB& color) {
		GradientColormapBase::setColorAt(position, color);
	}

	/// Removes all color keys.
	void clear() {
		GradientColormapBase::clear();
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Internal base class for formulae based color maps.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT FormulaeColormapBase : public ContinuousColormap
{
public:

	FormulaeColormapBase(std::size_t size) : ContinuousColormap(size),
		mRFormula(7),mGFormula(5),mBFormula(15) {}

	virtual Color::RGB getf(float f) const;

protected:

	/**
	 * Sets a color key at the given position with the given color.
	 * The given position must be in the range 0 to 1.
	 */
	void setFormulae(int rformula, int gformula, int bformula);

private:
	int mRFormula, mGFormula, mBFormula;
};

/**
 * A class for creating formula based color maps.
 * For each RGB color component a formula is given, that specifies,
 * how the component is computed. The advantage of these color maps is, that
 * most of the formulae are at least 2 times differentiable and therefore
 * no changes in the color gradients are visible (which is the case in linear
 * gradient color maps).
 *
 * The following predefined formulas are available. They correspond to the
 * rgbformulae of gnuplot:
 * <pre>
 *     0: 0               1: 0.5             2: 1
 *     3: x               4: x^2             5: x^3
 *     6: x^4             7: sqrt(x)         8: sqrt(sqrt(x))
 *     9: sin(90x)       10: cos(90x)       11: |x-0.5|
 *    12: (2x-1)^2       13: sin(180x)      14: |cos(180x)|
 *    15: sin(360x)      16: cos(360x)      17: |sin(360x)|
 *    18: |cos(360x)|    19: |sin(720x)|    20: |cos(720x)|
 *    21: 3x             22: 3x-1           23: 3x-2
 *    24: |3x-1|         25: |3x-2|         26: (3x-1)/2
 *    27: (3x-2)/2       28: |(3x-1)/2|     29: |(3x-2)/2|
 *    30: x/0.32-0.78125 31: 2*x-0.84       32: 4x;1;-2x+1.84;x/0.08-11.5
 *    33: |2*x - 0.5|    34: 2*x            35: 2*x - 0.5
 *    36: 2*x - 1
 * </pre>
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT FormulaeColormap : public FormulaeColormapBase
{
public:

	FormulaeColormap(std::size_t size=256) : FormulaeColormapBase(size) {}

	FormulaeColormap(int rformula, int gformula, int bformula, std::size_t size=256) : FormulaeColormapBase(size) {
		setFormulae(rformula, gformula, bformula);
	}

	/**
	 * Sets the formulae for the color components.
	 */
	void setFormulae(int rformula, int gformula, int bformula) {
		FormulaeColormapBase::setFormulae(rformula, gformula, bformula);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Base class for tabular color colormaps.
 * Tabular color colormaps are discrete color colormaps, where each color is
 * read from a predefined table that must be specified in the constructor.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT TabularColormap : public DiscreteColormap
{
	MIRA_OBJECT(TabularColormap)
public:

	/**
	 * Creates a new TabularColormap using the specified color table that
	 * is specified as first parameter. The size of that color table is
	 * specified as second parameter.
	 */
	TabularColormap(const uint8* pData, std::size_t pSize) :
		mData(pData), mSize(pSize) {}

	virtual std::size_t size() const {return mSize;}

	virtual Color::RGB get(std::size_t idx) const {
		return Color::RGB((float)mData[idx*3]/255.0f,
		                  (float)mData[idx*3+1]/255.0f,
		                  (float)mData[idx*3+2]/255.0f);
	}

private:
	const uint8* mData;
	std::size_t mSize;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A continuous grayscale colormap.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT GrayscaleColormap : public ContinuousColormap
{
	MIRA_OBJECT(GrayscaleColormap)
public:

	GrayscaleColormap(std::size_t size=256) : ContinuousColormap(size) {}

	static Color::RGB sGetf(float f)
	{
		if(f<=0.0f)
			return Color::RGB(0,0,0);
		if(f<=1.0f)
			return Color::RGB(f,f,f);

		return Color::RGB(1.0f,1.0f,1.0f);
	}

	virtual Color::RGB getf(float f) const {
		return GrayscaleColormap::sGetf(f);
	}

};

///////////////////////////////////////////////////////////////////////////////

/**
 * A continuous HSV colormap.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT HSVColormap : public ContinuousColormap
{
	MIRA_OBJECT(HSVColormap)
public:

	HSVColormap(std::size_t size=256) : ContinuousColormap(size) {}

	static Color::RGB sGetf(float f)
	{
		/*
		 0.0000 - 1   0   0
		 0.1667 - 1   1   0
		 0.3333 - 0   1   0
		 0.5000 - 0   1   1
		 0.6667 - 0   0   1
		 0.8333 - 1   0   1
		 1.0000 - 1   0   0
		 */

		if(f<=0.0f)
			return Color::RGB(1,0,0);
		if(f<=0.1667f)
			return Color::RGB(1,f*6,0);
		if(f<=0.3333f)
			return Color::RGB(1-(f-0.1667f)*6,1,0);
		if(f<=0.5000f)
			return Color::RGB(0,1,(f-0.3333f)*6);
		if(f<=0.6667f)
			return Color::RGB(0,1-(f-0.5000f)*6,1);
		if(f<=0.8333f)
			return Color::RGB((f-0.6667f)*6,0,1);
		if(f<=1.0000f)
			return Color::RGB(1,0,1-(f-0.8333f)*6);

		return Color::RGB(1,0,0);
	}

	virtual Color::RGB getf(float f) const
	{
		return HSVColormap::sGetf(f);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A continuous Jet colormap.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT JetColormap : public ContinuousColormap
{
	MIRA_OBJECT(JetColormap)
public:

	JetColormap(std::size_t size=256) : ContinuousColormap(size) {}

	static Color::RGB sGetf(float f)
	{
		/*
		 0.000 - 0   0   0.5
		 0.125 - 0   0   1.0
		 0.375 - 0   1.0 1.0
		 0.625 - 1.0 1.0 0.0
		 0.875 - 1.0 0.0 0.0
		 1.000 - 0.5 0.0 0.0
		 */

		if(f<=0.0f)
			return Color::RGB(0,0,0.5f);
		if(f<=0.125f)
			return Color::RGB(0,0,0.5f+f*4.0f);
		if(f<=0.375f)
			return Color::RGB(0,(f-0.125f)*4.0f,1.0f);
		if(f<=0.625f) {
			float alpha = (f-0.375f)*4.0f;
			return Color::RGB(alpha,1.0f,1.0f-alpha);
		}
		if(f<=0.875f)
			return Color::RGB(1.0f,1.0f-(f-0.625f)*4.0f,0.0f);
		if(f<=1.0f)
			return Color::RGB(1.0f-(f-0.875f)*4,0.0f,0.0f);

		return Color::RGB(0.5f,0,0);
	}

	virtual Color::RGB getf(float f) const
	{
		return JetColormap::sGetf(f);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A continuous Red-Blue colormap.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT RedBlueColormap: public ContinuousColormap {
	MIRA_OBJECT(RedBlueColormap)
public:

	RedBlueColormap(std::size_t size = 256) :
		ContinuousColormap(size) {
	}

	static Color::RGB sGetf(float f)
	{
		if (f < 0.0f)
			return Color::RGB(0.0f, 0.0f, 1.0f);
		if (f < 1.0f)
			return Color::RGB(f, 0.0f, 1.0f - f);
		return Color::RGB(1.0f, 0.0f, 0.0f);
	}

	virtual Color::RGB getf(float f) const {
		return RedBlueColormap::sGetf(f);
	}
};

///////////////////////////////////////////////////////////////////////////////

/**
 * The default pm3d gnuplot palette (black-blue-red-yellow)
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT Pm3dColormap: public FormulaeColormapBase {
	MIRA_OBJECT(Pm3dColormap)
public:
	Pm3dColormap(std::size_t size=256);
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A discrete colormap with 10 complementary colors.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT Complementary10Colormap : public TabularColormap
{
	MIRA_OBJECT(Complementary10Colormap)
public:
	Complementary10Colormap() : TabularColormap(colormap, 10) {}
private:
	const static uint8 colormap[3*10];
};

///////////////////////////////////////////////////////////////////////////////

/**
 * A discrete colormap with 6 complementary colors.
 * @ingroup ImageModule
 */
class MIRA_BASE_EXPORT Complementary6Colormap : public TabularColormap
{
	MIRA_OBJECT(Complementary6Colormap)
public:
	Complementary6Colormap() : TabularColormap(colormap, 6) {}
private:
	const static uint8 colormap[3*6];
};

///////////////////////////////////////////////////////////////////////////////
}

MIRA_NO_PUBLIC_DEFAULT_CONSTRUCTOR( mira::TabularColormap )

#endif
