/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LogCustomizableFormatter.h
 *    A customizable log formatter.
 *
 * @author Tim Langner
 * @date   2010/07/09
 */

#ifndef _MIRA_LOGCUSTOMIZABLEFORMATTER_H_
#define _MIRA_LOGCUSTOMIZABLEFORMATTER_H_

#ifndef Q_MOC_RUN
#include <boost/date_time/local_time/local_time.hpp>
#endif

#include <error/LoggingCore.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A customizable formatter. Different formatters can be chained together
 * to match the users needs.
 * e.g.
 * @code
 * LogCustomizableFormatter cf;
 * cf << "At "
 *    << LogCustomizableFormatter::Time() << " \""
 *    << LogCustomizableFormatter::Message() << "\" occurred.";
 * @endcode
 * will format the log output like this:
 *
 * At 2010-Jul-08 14:00:50.498986 "This is a log message" occurred.
 *
 * More formatters can be added by deriving from LogCustomizableFormatter::CustomFormatBase
 * and overwrite the format function.
 * The new formatter can be chained by adding it via the << operator
 * to the LogCustomizableFormatter instance.
 *
 * @ingroup LoggingModule
 */
class LogCustomizableFormatter : public LogFormatterBase
{
public:

	/**
	 * Abstract base class for all CustomFormats.
	 * Derive from it if you like to add your own custom format.
	 */
	class CustomFormatBase
	{
	public:
		virtual ~CustomFormatBase() {}
		/**
		 * Called by LogCustomizableFormatter whenever a new log entry is getting formatted.
		 * Writes the formatted data to ioStream.
		 * @param record The log entry
		 * @param ioStream The stream the formatted data gets written to.
		 */
		virtual void format(const LogRecord& record, std::ostringstream& ioStream) = 0;
	};

	/**
	 * Generic format, used to write various data to the log format
	 */
	template< typename T>
	class Generic : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		Generic(const T& iData) : data(iData) {}

		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << data;
		}

		T data;
	};

	/**
	 * Time format, used to write the time of the log entry to the log format.
	 */
	class Time : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		/**
		 * @param iFormat The format for the time to string conversion.
		 * (see http://www.boost.org/doc/libs/1_35_0/doc/html/date_time/date_time_io.html#date%5Ftime.format%5Fflags )
		 */
		Time(const std::string& iFormat = boost::local_time::local_time_facet::default_time_format)
		{
			using namespace boost::posix_time;
			// using a time facet for time2string conversion here
			// we could create a new time facet every time we format a new log entry
			// but we gain some performance by using a shared_ptr to it.
			// time_facet uses an internal ref counter that we need to set to one
			// because locale deletes the pointer after the ref counter gets 0
			timeFacet.reset(new time_facet(iFormat.c_str(),
			                               time_facet::period_formatter_type(),
			                               time_facet::special_values_formatter_type(),
			                               time_facet::date_gen_formatter_type(), 1));
		}

		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			std::ostringstream s;
			// imbue the stream with the facet
			// the ref counter of the facet gets incremented on imbue and
			// decremented when the stream goes out of scope, but we are holding a ptr to
			// it and have also set the ref counter to 1 in the constructor so the
			// facet stays valid after leaving this method.
			s.imbue(std::locale(s.getloc(), timeFacet.get()));
			s << record.time;
			ioStream << s.str();
		}

		boost::shared_ptr<boost::posix_time::time_facet> timeFacet;
	};

	/**
	 * Uptime format, used to write the up-time of the logging core to the log format.
	 */
	class Uptime : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		/**
		 * @param iFormat The format for the time conversion.
		 * (see http://www.boost.org/doc/libs/1_35_0/doc/html/date_time/date_time_io.html#date%5Ftime.format%5Fflags )
		 */
		Uptime(const std::string& iFormat = boost::local_time::local_time_facet::default_time_duration_format)
		{
			using namespace boost::posix_time;
			// using a time facet for time2string conversion here
			// we could create a new time facet every time we format a new log entry
			// but we gain some performance by using a shared_ptr to it.
			// time_facet uses an internal ref counter that we need to set to one
			// because locale deletes the pointer after the ref counter gets 0
			timeFacet.reset(new time_facet(iFormat.c_str(),
			                               time_facet::period_formatter_type(),
			                               time_facet::special_values_formatter_type(),
			                               time_facet::date_gen_formatter_type(), 1));
			timeFacet->time_duration_format(iFormat.c_str());
		}

		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			std::ostringstream s;
			// imbue the stream with the facet
			// the ref counter of the facet gets incremented on imbue and
			// decremented when the stream goes out of scope, but we are holding a ptr to
			// it and have also set the ref counter to 1 in the constructor so the
			// facet stays valid after leaving this method.
			s.imbue(std::locale(s.getloc(), timeFacet.get()));
			s << record.uptime;
			ioStream << s.str();
		}

		boost::shared_ptr<boost::posix_time::time_facet> timeFacet;
	};

	/**
	 * Line format, used to write the line of the log entry to the log format.
	 */
	class Line : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << record.line;
		}
	};

	/**
	 * File format, used to write the file of the log entry to the log format.
	 */
	class File : public CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << record.file;
		}
	};

	/**
	 * Function format, used to write the function name of the log entry to the log format.
	 */
	class Function : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << MIRA_LOGGER.parseFunction(record).functionName;
		}
	};

	/**
	 * Class format, used to write the class name of the log entry to the log format.
	 */
	class Class : public CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& stream)
		{
			stream << MIRA_LOGGER.parseFunction(record).className;
		}
	};

	/**
	 * Namespace format, used to write the name space of the log entry to the log format.
	 */
	class NameSpace : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << MIRA_LOGGER.parseFunction(record).nameSpace;
		}
	};

	/**
	 * ThreadID format, used to write the thread id of the log entry to the log format.
	 */
	class Thread : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << record.threadID;
		}
	};

	/**
	 * Message format, used to write the message of the log entry to the log format.
	 */
	class Message : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << record.message;
		}
	};

	/**
	 * Severity format, used to write the severity level of the log entry to the log format.
	 */
	class Severity : public LogCustomizableFormatter::CustomFormatBase
	{
	public:
		void format(const LogRecord& record, std::ostringstream& ioStream)
		{
			ioStream << severityLevelStr[(int) record.level];
		}
	};

	virtual std::string format(const LogRecord& record)
	{
		std::ostringstream s;
		for (size_t i = 0; i < mFormatter.size(); ++i)
			mFormatter[i]->format(record, s);
		return s.str();
	}

	LogCustomizableFormatter& operator<<(const char* data)
	{
		mFormatter.push_back(boost::shared_ptr<LogCustomizableFormatter::CustomFormatBase>(
				new LogCustomizableFormatter::Generic<std::string>(std::string(data))));
		return *this;
	}

	LogCustomizableFormatter& operator<<(std::string & data)
	{
		mFormatter.push_back(boost::shared_ptr<LogCustomizableFormatter::CustomFormatBase>(
				new LogCustomizableFormatter::Generic<std::string>(data)));
		return *this;
	}

	template<typename T>
	LogCustomizableFormatter& operator<<(const T& f)
	{
		mFormatter.push_back(boost::shared_ptr<LogCustomizableFormatter::CustomFormatBase>(new T(f)));
		return *this;
	}

protected:
	std::vector<boost::shared_ptr<LogCustomizableFormatter::CustomFormatBase> > mFormatter;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
