/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TimePropertyDelegate.C
 *    Implementation of TimePropertyDelegate.h
 *
 * @author Tim Langner
 * @date   2011/02/02
 */

#include <utils/Time.h>

#include <widgets/PropertyEditor.h>

#include <serialization/PropertySerializer.h>

#include <gui/private/TimePropertyDelegate.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_DateTimeEdit::PropertyDelegate_DateTimeEdit(PropertyNode* iProperty,
                                                             QWidget* parent) :
	QDateTimeEdit(parent), property(iProperty)
{
	setDisplayFormat("dd.MM.yyyy HH:mm:ss");
	const TypedPropertyNode<Time>* p = property->toTyped<Time>();
	assert(p!=NULL);
	Time t = p->get();
	Duration s = t.time_of_day();

	QDate qd((int)t.date().year(), (int)t.date().month(), (int)t.date().day());
	QTime qt(s.hours(), s.minutes(), s.seconds());
	setDate(qd);
	setTime(qt);
	connect(this, SIGNAL(dateTimeChanged(const QDateTime&)),
	        this, SLOT(slotSetValue(const QDateTime&)));
}

void PropertyDelegate_DateTimeEdit::slotSetValue(const QDateTime& dt)
{
	QDate d = dt.date();
	QTime t = dt.time();
	Time nt(Date(d.year(), d.month(), d.day()),
	        Duration(t.hour(), t.minute(), t.second()));
	TypedPropertyNode<Time>* p = property->toTyped<Time>();
	p->set(nt);
}

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_Time : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_Time)
public:

	virtual SupportedTypes supportedTypes() const
	{
		return makeSupportedType<Time>(SupportedType::TEXT |
		                               SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property)
	{
		const TypedPropertyNode<Time>* p = property->toTyped<Time>();
		assert(p!=NULL);
		Time t = p->get();
		return boost::posix_time::to_simple_string(t);
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent)
	{
		return new PropertyDelegate_DateTimeEdit(property, parent);
	}
};

///////////////////////////////////////////////////////////////////////////////

PropertyDelegate_SpinBoxTimeSpan::PropertyDelegate_SpinBoxTimeSpan(PropertyNode* iProperty,
                                                                   QWidget* parent) :
	QSpinBox(parent), property(iProperty)
{
	const TypedPropertyNode<Duration>* p = property->toTyped<Duration>();
	setFrame(false);

	Duration step = p->getHint<Duration>("step", Duration::milliseconds(100));
	Duration min = p->getHint<Duration>("minimum", Duration::milliseconds(std::numeric_limits<int>::min()));
	Duration max = p->getHint<Duration>("maximum", Duration::milliseconds(std::numeric_limits<int>::max()));
	setSingleStep(step.totalMilliseconds());
	setMinimum(min.totalMilliseconds());
	setMaximum(max.totalMilliseconds());
	setValue(p->get().totalMilliseconds());
	connect(this, SIGNAL(valueChanged(int)), this, SLOT(slotSetValue(int)));
}

void PropertyDelegate_SpinBoxTimeSpan::slotSetValue(int value)
{
	TypedPropertyNode<Duration>* p = property->toTyped<Duration>();
	p->set(Duration::milliseconds(value));
}

///////////////////////////////////////////////////////////////////////////////

class PropertyDelegate_TimeSpan : public PropertyEditor::Delegate
{
	MIRA_OBJECT(PropertyDelegate_TimeSpan)
public:

	virtual SupportedTypes supportedTypes() const
	{
		return makeSupportedType<Duration>(SupportedType::TEXT |
		                                   SupportedType::EDITOR);
	}

	virtual std::string getText(const PropertyNode* property)
	{
		const TypedPropertyNode<Duration>* p = property->toTyped<Duration>();
		assert(p!=NULL);
		Duration t = p->get();
		return boost::posix_time::to_simple_string(t);
	}

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent)
	{
		return new PropertyDelegate_SpinBoxTimeSpan(property, parent);
	}
};

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_REGISTER(mira::gui::PropertyDelegate_Time,
                    mira::PropertyEditor::Delegate);
MIRA_CLASS_REGISTER(mira::gui::PropertyDelegate_TimeSpan,
                    mira::PropertyEditor::Delegate);
