/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RectVectorVisualization.C
 *    Visualization for std::vector or std::list containers of Rects.
 *    Theoretically, any STL conform iterable container can be substituted.
 *
 * @author Christian Reuther
 * @date   2017/11/30
 */

#include <serialization/Serialization.h>

#include <visualization/Visualization3DBasic.h>
#include <visualization/3d/RectObject.h>
#include <serialization/adapters/OGRE/OgreColourValue.h>

#include <widgets/OgreUtils.h>
#include <OGRE/OgreSceneManager.h>

namespace mira { namespace gui {

///////////////////////////////////////////////////////////////////////////////

template <typename T>
class RectVectorVisualization3DBase : public Visualization3DBasic<T>
{
	typedef Visualization3DBasic<T> Base;

public:
	RectVectorVisualization3DBase() : mSceneManager(NULL), mSceneNode(NULL), mColor(Ogre::ColourValue::Black)
	{
		// Empty
	}

	virtual ~RectVectorVisualization3DBase()
	{
		foreach(auto r, mRects)
			delete r;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, Base);
		r.property("Color", mColor, setter<Ogre::ColourValue>(&RectVectorVisualization3DBase<T>::setColor, this),
				"The color of the grid", Ogre::ColourValue::Black);
	}

public:
	void setColor(const Ogre::ColourValue& c)
	{
		mColor = c;
		foreach(auto r, mRects)
			r->setColor(mColor);
	}

public:
	virtual void setupScene(Ogre::SceneManager* mgr, Ogre::SceneNode* node)
	{
		mSceneManager = mgr;
		mSceneNode = node;
	}

	virtual void setEnabled(bool enabled)
	{
		Base::setEnabled(enabled);
		foreach(auto r, mRects)
			r->setVisible(enabled);
	}

	void dataChanged(ChannelRead<T> rects)
	{
		std::size_t i = 0;

		// Update (and potentially create new) rects
		foreach(const auto& rect, rects->value())
		{
			RectObject* obj = NULL;
			if(i >= mRects.size())
			{
				obj = new RectObject(mSceneManager, mSceneNode);
				obj->setColor(mColor);
				obj->setVisible(this->isEnabled());
				mRects.push_back(obj);
			}
			else
			{
				obj = mRects[i];
			}

			obj->setRect(Rect2f(rect.x0(), rect.y0(), rect.width(), rect.height()));
			++i;
		}

		// Delete all remaining rects. TODO Would be better to set a "deleted" flag.
		if(i < mRects.size())
		{
			for(std::size_t j = i; j < mRects.size(); ++j)
			{
				RectObject* obj = mRects[j];
				delete obj;
			}
			mRects.resize(i);
		}
	}

private:
	Ogre::SceneManager* mSceneManager;
	Ogre::SceneNode* mSceneNode;
	std::vector<RectObject*> mRects;
	Ogre::ColourValue mColor;
};

///////////////////////////////////////////////////////////////////////////////

class Rect2iVectorVisualization3D : public RectVectorVisualization3DBase<std::vector<Rect2i>>
{
	MIRA_META_OBJECT(Rect2iVectorVisualization3D,
		("Name", "Rect2iVectorVisualization3D")
		("Description", "Visualization of a vector of 2D int rects")
		("Category", "Geometry"))
};

class Rect2fVectorVisualization3D : public RectVectorVisualization3DBase<std::vector<Rect2f>>
{
	MIRA_META_OBJECT(Rect2fVectorVisualization3D,
		("Name", "Rect2fVectorVisualization3D")
		("Description", "Visualization of a vector of 2D float rects")
		("Category", "Geometry"))
};

class Rect2dVectorVisualization3D : public RectVectorVisualization3DBase<std::vector<Rect2d>>
{
	MIRA_META_OBJECT(Rect2dVectorVisualization3D,
		("Name", "Rect2dVectorVisualization3D")
		("Description", "Visualization of a vector of 2D double rects")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

class Rect2iListVisualization3D : public RectVectorVisualization3DBase<std::list<Rect2i>>
{
	MIRA_META_OBJECT(Rect2iListVisualization3D,
		("Name", "Rect2iListVisualization3D")
		("Description", "Visualization of a list of 2D int rects")
		("Category", "Geometry"))
};

class Rect2fListVisualization3D : public RectVectorVisualization3DBase<std::list<Rect2f>>
{
	MIRA_META_OBJECT(Rect2fListVisualization3D,
		("Name", "Rect2fListVisualization3D")
		("Description", "Visualization of a list of 2D float rects")
		("Category", "Geometry"))
};

class Rect2dListVisualization3D : public RectVectorVisualization3DBase<std::list<Rect2d>>
{
	MIRA_META_OBJECT(Rect2dListVisualization3D,
		("Name", "Rect2dListVisualization3D")
		("Description", "Visualization of a list of 2D double rects")
		("Category", "Geometry"))
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

MIRA_CLASS_SERIALIZATION(mira::gui::Rect2iVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2fVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2dVectorVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2iListVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2fListVisualization3D, mira::Visualization3D);
MIRA_CLASS_SERIALIZATION(mira::gui::Rect2dListVisualization3D, mira::Visualization3D);
