/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 *  and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OmniCameraIntrinsic.C
 *    Implementation of intrinsic parameters for Omni cameras
 *
 * @author Michael Volkhardt
 * @date   2011/09/22
 */

#include <cameraparameters/OmniCameraIntrinsic.h>

using namespace Eigen;

namespace mira {
namespace camera {

///////////////////////////////////////////////////////////////////////////////

OmniCameraIntrinsicNormalized::OmniCameraIntrinsicNormalized() {
	centerX = 0.5;
	centerY = 0.5;
	K.resize(1., 1.0);
	K_inv.resize(1, 1.0);
	A << 1.,0.,0.,1.;
}

OmniCameraIntrinsicNormalized::OmniCameraIntrinsicNormalized(float iCenterX,
		float iCenterY, std::vector<float> iK,
		std::vector<float> iK_inv, Eigen::Matrix2f const& iA) {

	centerX = iCenterX;
	centerY = iCenterY;

	K = iK;
	K_inv = iK_inv;

	A = iA;
	assert(A(1,1) == 1.);
}

///////////////////////////////////////////////////////////////////////////////

OmniCameraIntrinsic::OmniCameraIntrinsic() {
	centerX = 0;
	centerY = 0;
	K.resize(1, 1.0);
	K_inv.resize(1, 1.0);
	A << 1.,0.,0.,1.;
}

OmniCameraIntrinsic::OmniCameraIntrinsic(float iCenterX, float iCenterY,
		std::vector<float> const& iK,
		std::vector<float> const& iK_inv, Eigen::Matrix2f const& iA) {
	centerX = iCenterX;
	centerY = iCenterY;

	K = iK;
	K_inv = iK_inv;

	A = iA;
	assert(A(1,1) == 1.);
}

OmniCameraIntrinsic::OmniCameraIntrinsic(
		const OmniCameraIntrinsicNormalized& iNormIntrinParm,
		const Size2i& iImageSize) {

	centerX = iNormIntrinParm.centerX * iImageSize.width();
	centerY = iNormIntrinParm.centerY * iImageSize.height();

	K = iNormIntrinParm.K;
	K_inv= iNormIntrinParm.K_inv;
	A = iNormIntrinParm.A;
	assert(A(1,1) == 1.);
}

///////////////////////////////////////////////////////////////////////////////
#if CV_MAJOR_VERSION >= 3
FisheyeCameraIntrinsicNormalized::FisheyeCameraIntrinsicNormalized() {
	K = cv::Matx33d();
	D = cv::Vec4d();
}

FisheyeCameraIntrinsicNormalized::FisheyeCameraIntrinsicNormalized(cv::Matx33d iK, cv::Vec4d iD) {
	K = iK;
	D = iD;
}

///////////////////////////////////////////////////////////////////////////////

FisheyeCameraIntrinsic::FisheyeCameraIntrinsic() {
    K = cv::Matx33d();
    D = cv::Vec4d();
}

FisheyeCameraIntrinsic::FisheyeCameraIntrinsic(cv::Matx33d iK, cv::Vec4d iD) {
	K = iK;
	D = iD;
}

FisheyeCameraIntrinsic::FisheyeCameraIntrinsic(
		const FisheyeCameraIntrinsicNormalized& iNormalizedIntrinsic,
		const Size2i& iImageSize) {

	K(0,0) = iNormalizedIntrinsic.K(0,0) * iImageSize.height();
	K(0,2) = iNormalizedIntrinsic.K(0,2) * iImageSize.height();
	K(1,1) = iNormalizedIntrinsic.K(1,1) * iImageSize.width();
	K(1,2) = iNormalizedIntrinsic.K(1,2) * iImageSize.width();
	K(2,2) = 1.0;

	D = iNormalizedIntrinsic.D;
}
///////////////////////////////////////////////////////////////////////////////
#endif
}
} // namespace
