/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OgreMeshLoader.C
 *    Loader for Ogre meshes.
 *
 * @author Erik Einhorn
 * @date   2011/12/16
 */

#include <fstream>
#include <OGRE/Ogre.h>

#include <visualization/3d/MeshLoader.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class OgreMeshLoader : public MeshLoader
{
	MIRA_META_OBJECT(OgreMeshLoader,
	                ("Extension", "Ogre Mesh,.mesh"))
public:

	Ogre::MeshPtr loadMesh(const Path& filename)
	{
		std::ifstream is(filename.c_str());
		Ogre::DataStreamPtr filedatastream(new Ogre::FileStreamDataStream(&is, false));
		Ogre::MeshPtr mesh = Ogre::MeshManager::getSingleton().createManual(
				filename.string(),
				Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
		Ogre::MeshSerializer serializer;
		serializer.importMesh(filedatastream, mesh.get() );
		return mesh;
	}
};

class OgreZipMeshLoader : public MeshLoader
{
	MIRA_META_OBJECT(OgreZipMeshLoader,
	                ("Extension", "Ogre Mesh,.zip"))
public:

	Ogre::MeshPtr loadMesh(const Path& filename)
	{
		std::string group = filename.string();
		Ogre::ResourceGroupManager* rmgr = Ogre::ResourceGroupManager::getSingletonPtr();
		rmgr->addResourceLocation(filename.string(), "Zip", group);
		Ogre::StringVectorPtr v = rmgr->findResourceNames(filename.string(), "*.mesh");
		if(v->empty())
			MIRA_THROW(XIO, "No mesh found in archive '" << filename.string() << "'");
		rmgr->initialiseResourceGroup(group);
		rmgr->loadResourceGroup(group);

		Ogre::MeshPtr mesh = Ogre::MeshManager::getSingleton().load(v->at(0),group);
		assert(mesh.get()!=NULL);

		return mesh;
	}
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_REGISTER(mira::OgreMeshLoader, mira::MeshLoader)
MIRA_CLASS_REGISTER(mira::OgreZipMeshLoader, mira::MeshLoader)
