/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AxesCovObject.C
 *   Visualization of covariance for axes
 *
 * @author Erik Einhorn
 * @date   2011/07/07
 */
#include <visualization/3d/AxesCovObject.h>

#include <OGRE/OgreSceneNode.h>

#include <visualization/3d/MeshObject.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

AxesCovObject::AxesCovObject(Ogre::SceneManager* sceneManager,
              	               AxesObject* axesObject) :
	mAxesObject(axesObject), mVisible(true), mSigma(1.0f)
{
	float alpha = 0.5f;
	float length = 0.8f * mAxesObject->getLength();

	Ogre::SceneNode* node = mAxesObject->getNode();
	mCovRotX.reset(new MeshObject("Cone.mesh", sceneManager, node));
	mCovRotX->setVisible(mVisible);
	mCovRotX->setColor(Ogre::ColourValue(1.0f,0.0f,0.0f,alpha));
	mCovRotX->setScale(Ogre::Vector3(0.0f,length,0.0f));

	mCovRotY.reset(new MeshObject("Cone.mesh", sceneManager, node));
	mCovRotY->setVisible(mVisible);
	mCovRotY->setColor(Ogre::ColourValue(0.0f,1.0f,0.0f,alpha));
	mCovRotY->setScale(Ogre::Vector3(0.0f,length,0.0f));

	if(mAxesObject->isShowingZAxis()) {
		mCovRotZ.reset(new MeshObject("Cone.mesh", sceneManager, node));
		mCovRotZ->setVisible(mVisible);
		mCovRotZ->setColor(Ogre::ColourValue(0.0f,0.0f,1.0f,alpha));
		mCovRotZ->setScale(Ogre::Vector3(0.0f,length,0.0f));
	}
}

///////////////////////////////////////////////////////////////////////////////

void AxesCovObject::setVisible(bool visible)
{
	mVisible = visible;
	if(mCovRotX)
		mCovRotX->setVisible(visible);
	if(mCovRotY)
		mCovRotY->setVisible(visible);
	if(mCovRotZ)
		mCovRotZ->setVisible(visible);

}

void AxesCovObject::setSigma(float sigma)
{
	mSigma = sigma;
}

///////////////////////////////////////////////////////////////////////////////

void AxesCovObject::setCovariance(float var)
{
	const float eps = 1.0e-2f;

	Eigen::Matrix3f cov;
	cov.setZero();
	cov(0,0) = eps;
	cov(1,1) = eps;
	cov(2,2) = var;
	setCovariance(cov);
}

void AxesCovObject::setCovariance(const Eigen::Matrix3f& cov)
{
	float length = 0.8f * mAxesObject->getLength();

	// if we have a zero - cov matrix, escape here
	if(cov.trace() < std::numeric_limits<float>::epsilon())
	{
		mCovRotX->setOrientation(Ogre::Quaternion(Ogre::Degree(90),Ogre::Vector3::UNIT_Z));
		mCovRotX->getNode()->rotate(Ogre::Vector3::UNIT_Y,Ogre::Radian(0.0f));
		mCovRotX->setScale(Ogre::Vector3(0.0f,length,0.0f));
		mCovRotX->setPosition(Ogre::Vector3(length,0,0));

		mCovRotY->setOrientation(Ogre::Quaternion(Ogre::Degree(180),Ogre::Vector3::UNIT_Z));
		mCovRotY->getNode()->rotate(Ogre::Vector3::UNIT_Y,Ogre::Radian(0.0f));
		mCovRotY->setScale(Ogre::Vector3(0.0f,length,0.0f));
		mCovRotY->setPosition(Ogre::Vector3(0,length,0));

		mCovRotZ->setOrientation(Ogre::Quaternion(Ogre::Degree(-90),Ogre::Vector3::UNIT_X));
		mCovRotZ->getNode()->rotate(Ogre::Vector3::UNIT_Y,Ogre::Radian(0.0f));
		mCovRotZ->setScale(Ogre::Vector3(0.0f,length,0.0f));
		mCovRotZ->setPosition(Ogre::Vector3(0,0,length));
		return;
	}

	// X AXIS
	{
		Eigen::Matrix2f Ca;
		Ca << cov(2,2), -cov(2,1),
		     -cov(1,2),  cov(1,1);

		Eigen::JacobiSVD<Eigen::Matrix2f> svd(Ca, Eigen::ComputeFullV);
		Eigen::Vector2f w = svd.singularValues()*mSigma;
		float alpha = std::acos(svd.matrixV()(0,0));

		mCovRotX->setOrientation(Ogre::Quaternion(Ogre::Degree(90),Ogre::Vector3::UNIT_Z));
		mCovRotX->getNode()->rotate(Ogre::Vector3::UNIT_Y,Ogre::Radian(alpha));

		mCovRotX->setScale(Ogre::Vector3(w(0)*length,length,w(1)*length));
		mCovRotX->setPosition(Ogre::Vector3(length,0,0));
	}

	// Y AXIS
	{
		Eigen::Matrix2f Ca;
		Ca << cov(2,2), -cov(2,0),
		     -cov(0,2),  cov(0,0);

		Eigen::JacobiSVD<Eigen::Matrix2f> svd(Ca, Eigen::ComputeFullV);
		Eigen::Vector2f w = svd.singularValues()*mSigma;
		float alpha = std::acos(svd.matrixV()(0,0));

		mCovRotY->setOrientation(Ogre::Quaternion(Ogre::Degree(180),Ogre::Vector3::UNIT_Z));
		mCovRotY->getNode()->rotate(Ogre::Vector3::UNIT_Y,Ogre::Radian(-alpha));

		mCovRotY->setScale(Ogre::Vector3(w(0)*length,length,w(1)*length));
		mCovRotY->setPosition(Ogre::Vector3(0,length,0));
	}

	// Z AXIS
	if(mCovRotZ) {

		Eigen::Matrix2f Ca;
		Ca << cov(1,1), -cov(1,0),
		     -cov(0,1),  cov(0,0);

		Eigen::JacobiSVD<Eigen::Matrix2f> svd(Ca, Eigen::ComputeFullV);
		Eigen::Vector2f w = svd.singularValues()*mSigma;
		float alpha = std::acos(svd.matrixV()(0,0));

		mCovRotZ->setOrientation(Ogre::Quaternion(Ogre::Degree(-90),Ogre::Vector3::UNIT_X));
		mCovRotZ->getNode()->rotate(Ogre::Vector3::UNIT_Y,Ogre::Radian(alpha));

		mCovRotZ->setScale(Ogre::Vector3(w(0)*length,length,w(1)*length));
		mCovRotZ->setPosition(Ogre::Vector3(0,0,length));
	}
}

///////////////////////////////////////////////////////////////////////////////

}
