/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PTPSyncTestUnits.C
 *    Utility units for PTP sync test 
 *
 * @author Christof Schröter
 * @date   2021/06/03
 */

#include <fw/Unit.h>

namespace mira {
namespace fw {
namespace test {
namespace ptp {

///////////////////////////////////////////////////////////////////////////////

// <unit id="Caller" class="mira::fw::test::ptp::CallUnit" />

class CallUnit : public Unit
{
MIRA_OBJECT(CallUnit)

public:
	CallUnit() : Unit(Duration::milliseconds(100)) {}

protected:

	virtual void initialize()
	{
		mCallChannel = publish<int>("Call");
		subscribe<int>("Echo", &CallUnit::onEcho, this);
		mWaitForEcho = 0;

		mCountChannel = publish<int>("result/Count");
		mMaxDelayChannel = publish<int>("result/MaxDelay");
		mMaxTSDiffChannel = publish<int>("result/MaxTSDiff");

		mCount = 0;
		mMaxDelay = 0;
		mMaxTSDiff = 0;
	}

	virtual void process(const Timer& timer);

protected:

	void onEcho(ChannelRead<int> read);

private:

	Channel<int> mCallChannel;
	int mWaitForEcho;
	Time mCallTime;

	int mCount;
	int mMaxDelay;
	int mMaxTSDiff;
	Channel<int> mCountChannel;
	Channel<int> mMaxDelayChannel;
	Channel<int> mMaxTSDiffChannel;
};

void CallUnit::process(const Timer& timer)
{
	if (mWaitForEcho > 0)
		return;

	mCallTime = Time::now();
	mWaitForEcho = -mWaitForEcho + 1;
	mCallChannel.post(mWaitForEcho, mCallTime);
}

void CallUnit::onEcho(ChannelRead<int> read)
{
	if (*read != mWaitForEcho) {
		MIRA_LOG(ERROR) << "Unexpected echo " << *read << " (mWaitForEcho=" << mWaitForEcho << ")";
		return;
	}

	Time now = Time::now();
	int delay = (now - mCallTime).totalMicroseconds();
	int tsdiff = (read->timestamp - mCallTime).totalMicroseconds();
//	std::cout << now << ": call@" << mCallTime << " - echo@" << read->timestamp
//	          << " --> delay=" << delay << "us"
//	          << ", ts diff=" << tsdiff << "us" << std::endl;
	mWaitForEcho = -mWaitForEcho;

	mCountChannel.post(++mCount);
	if (std::abs(delay) > std::abs(mMaxDelay))
		mMaxDelay = delay;
	mMaxDelayChannel.post(mMaxDelay);

	if (std::abs(tsdiff) > std::abs(mMaxTSDiff))
		mMaxTSDiff = tsdiff;
	mMaxTSDiffChannel.post(mMaxTSDiff);
}

///////////////////////////////////////////////////////////////////////////////

// <unit id="Echo" class="mira::fw::test::ptp::EchoUnit" />

class EchoUnit : public MicroUnit
{
MIRA_OBJECT(EchoUnit)

protected:

	virtual void initialize()
	{
		mEchoChannel = publish<int>("Echo");
		subscribe<int>("Call", &EchoUnit::onCall, this);
	}

protected:

	void onCall(ChannelRead<int> read);

private:

	Channel<int> mEchoChannel;
};

void EchoUnit::onCall(ChannelRead<int> read)
{
	mEchoChannel.post((int)(*read), read->timestamp);
}

///////////////////////////////////////////////////////////////////////////////

}}}} /// namespaces

MIRA_CLASS_SERIALIZATION(mira::fw::test::ptp::CallUnit, mira::Unit);
MIRA_CLASS_SERIALIZATION(mira::fw::test::ptp::EchoUnit, mira::MicroUnit);
