/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Framework.h
 *    The framework that holds all manager classes and provides startup and
 *    shutdown of all framework related services.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/09/06
 */

#ifndef _MIRA_FRAMEWORK_H_
#define _MIRA_FRAMEWORK_H_

#include <serialization/adapters/std/list>
#include <serialization/adapters/std/map>
#include <serialization/adapters/boost/shared_ptr.hpp>

#include <utils/ProgramOptions.h>
#include <platform/Types.h>
#include <rpc/RPCManager.h>
#include <thread/ScopedAccess.h>
#include <utils/Singleton.h>
#include <xml/XMLDomReflect.h>
#include <xml/XMLDomModifier.h>
#include <factory/LibraryRegistry.h>

#include <fw/Authority.h>
#include <fw/AuthorityManager.h>
#include <fw/Channel.h>
#include <fw/ChannelManager.h>
#include <fw/NameRegistry.h>

#include <loader/Loader.h>

/// Macro for accessing the framework instance
#define MIRA_FW mira::Framework::instance()

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward decls.
struct IntSignal;
class ErrorService;
class Framework;
class FrameworkTransformer;
class RemoteModule;
class UnitManager;

///@cond INTERNAL
namespace Private {

///////////////////////////////////////////////////////////////////////////////

/**
 * Helper class that is used as base class for Framework, to perform an
 * early initialization of certain components, e.g. setup of logging
 * mechanism etc. Since this setup is placed in the constructor of this
 * helper class, it is executed before any other steps of the initialization
 * sequence while Framework is constructed.
 */
class MIRA_FRAMEWORK_EXPORT FrameworkStartup
{
public:
	FrameworkStartup(int argc, char** argv);
	FrameworkStartup(const std::vector<std::string>& args);

protected:

	void initialize();
	ProgramOptions mOptions;
	typedef std::vector<std::string> CommandLineStrings;
	CommandLineStrings mCommandLineVariables;
	CommandLineStrings mCommandLineVariablesLegacy;
	CommandLineStrings mCommandLineKnownFrameworks;
};

///////////////////////////////////////////////////////////////////////////////

class FrameworkAuthority : public Authority
{
public:
	FrameworkAuthority(Framework* framework, const ResourceName& ns,
	                   const std::string& name, Flags flags = Authority::NORMAL)
		: Authority(ns, name, flags), mFramework(framework)
	{}

public:
	boost::shared_ptr<PropertyNode> getProperties();

protected:
	Framework* mFramework;
};

///////////////////////////////////////////////////////////////////////////////

}
///@endcond

///////////////////////////////////////////////////////////////////////////////

/**
 * @ingroup FWModule
 * This class represents the core element of a modular application.
 * The Framework allows communication between different application parts,
 * so called authorities, via data channels and remote procedure calls.
 * The framework hides implementation details as well as sources,
 * destinations of data and the way of data transmission from the developer of
 * an authority.
 * This allows the user to run the application on his local PC or
 * distributed on PCs in a network without changing a single line of code.
 */
class MIRA_FRAMEWORK_EXPORT Framework : public Private::FrameworkStartup,
                                        public ExplicitSingleton<Framework>
{
public:

	/**
	 * Constructor that takes command line arguments and a flag.
	 * @param argc Number of command line arguments.
	 * @param argv The command line arguments.
	 * @param startImmediately If true the methods load() and start() are
	 *        called after construction.
	 */
	Framework(int argc, char** argv, bool startImmediately = false);
	/**
	 * Constructor that takes command line arguments as a vector and a flag.
	 * @param args The command line arguments.
	 * @param startImmediately If true the methods load() and start() are
	 *        called after construction.
	 */
	Framework(const std::vector<std::string>& args, bool startImmediately = false);

	virtual ~Framework();

	/// Reflect method for serialization
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		serialization::VersionType version = r.version(2, this);

		if (version >= 2)
			r.property("RemoteModule", mRemoteModule, "The remote module", boost::shared_ptr<RemoteModule>());

		r.method("getUptime", &Framework::getUptime, this, "Get the time since the framework was started");

		r.method("terminateProcess", &Framework::requestTermination, this,
		         "Terminate framework (and thus, the process). Use with care!",
		         "exitcode", "the exitcode returned by the process", 0u);

		r.interface("IVariableRegistry");
		r.method("getVariables", &Framework::getVariables, this,
		         "Query variables");

		r.interface("ILibraryRegistry");
		r.method("getLibraries",
		          boost::function<LibraryRegistry::Register()>(LibraryRegistry::getRegister),
		         "Query libraries");
		r.method("getLoadedLibraries",
		         boost::function<LibraryRegistry::Register()>([](){
		         	LibraryRegistry::Register reg = LibraryRegistry::getRegister();
		         	for (auto it = reg.begin(); it != reg.end(); ) {
		         		if (it->second.mIsLoaded)
		         			++it;
		         		else {
		         			auto succ = it; ++succ;
		         			reg.erase(it);
		         			it = succ;
		         		}
		         	}
		         	return reg;
		         }),
		         "Query only loaded libraries");

		r.interface("IConfigurationLoader");
		r.method("loadConfig",
		         boost::function<void(const XMLDom&)>(
		         	[&](const XMLDom& xml) { this->load(const_cast<XMLDom&>(xml)); } ),
		         "Load a configuration",
		         "xml", "XML document", XMLDom());

	}

public:
	boost::shared_ptr<PropertyNode> getProperties()	{
		return mProperties;
	}

public:

	/**
	 * Loads the specified XML configuration and applies all registered
	 * Loaders on it. This will load libraries, instantiate and configure
	 * units, etc.
	 * This method will do nothing if the xml document is empty.
	 */
	void load(XMLDom& xml);

	/**
	 * Opens the specified XML configuration file and applies all registered
	 * Loaders on it. This will load libraries, instantiate and configure
	 * units, etc.
	 * This method will do nothing if configFile is empty.
	 */
	void load(const std::string& configFile);

	/**
	 * Loads configuration files given on command line if any by calling above
	 * load(const std::string& configFile) function.
	 */
	void load();

	/**
	 * Starts the framework and it's remote component (if one was configured
	 * or created) and returns immediately.
	 */
	void start();

	/**
	 * Return true if framework is started
	 */
	bool isStarted() const
	{
		return mIsStarted;
	}

	/**
	 * Return duration since started
	 */
	Duration getUptime() const
	{
		return Time::now() - mStartTime;
	}

	/**
	 * Executes the framework and blocks until the framework is terminated,
	 * e.g. by pressing Ctrl+C.
	 * Returns the exit code as specified by requestTermination().
	 */
	virtual int exec();

	/**
	 * Requests the termination of the framework and hence
	 * the whole application.
	 * Is called e.g. when user presses Ctrl+C.
	 * You can optionally specify an exit code that will be returned as
	 * exit code by exec(). Furthermore, this code is returned by the mira
	 * executables (mira, miragui, miracenter, etc.).
	 */
	virtual void requestTermination(int exitcode=0);

	/**
	 * Returns whether the termination of the framework is reqested.
	 */
	virtual bool isTerminationRequested() const;

	/**
	 * Returns the exit code as specified by requestTermination(), or 0
	 * if requestTermination() was not called.
	 * @see isTerminationRequested().
	 */
	int getTerminationExitCode() const;

	/**
	 * Calls the above load() and start() methods according to the command
	 * line parameters that were passed in the constructor and finally calls
	 * exec(). Hence, this method will launch a fully operating framework
	 * and blocks until the framework is terminated, e.g. by pressing Ctrl+C.
	 * Returns the exit code as specified by requestTermination(), or
	 * 0 if no error or exception has occurred, otherwise -1.
	 */
	int run();

public:

	/** @name Obtain access to framework components */
	//@{

	/// Returns the reference to the name registry
	NameRegistry& getNameRegistry() { return mNames; }

	/// Returns the reference to the manager singleton for registered authorities
	AuthorityManager& getAuthorityManager() { return mAuthorityManager; }

	/// Returns the reference to the manager singleton for channels
	ChannelManager& getChannelManager() { return mChannelManager; }

	/// Returns the pointer to the transform framework
	boost::shared_ptr<FrameworkTransformer> getTransformer() { return mTransformer; }

	/// Returns the reference to the configuration file loader
	ConfigurationLoader& getConfigurationLoader() { return mConfigurationLoader; }

	/// Returns the reference to the unit manager
	boost::shared_ptr<UnitManager> getUnitManager() { return mUnitManager; }

	/// Returns the reference to the manager singleton for registered RPC services
	RPCManager& getRPCManager() { return mRPCManager; }

	/// Returns the pointer to the persistent error service
	boost::shared_ptr<ErrorService> getErrorService() { return mErrorServiceModule; }

	/// Returns the pointer to the remote module
	boost::shared_ptr<RemoteModule> getRemoteModule() { return mRemoteModule; }

	//@}

	/**
	 * Returns the ID of this framework. 
	 * The ID can be specified via command line. If no ID is
	 * specified an UUID based one will be generated.
	 */
	std::string getID() const { return mAuthority->getID(); }

	/**
	 * Return the fully qualified global id of this framework (includes namespace)
	 * @return Global ID
	 */
	std::string getGlobalID() const { return mAuthority->getGlobalID(); }

	/**
	 * Returns the list of variables that are registered via command line
	 * or config file.
	 */
	XMLVariablesMap& getVariables() { return mXMLPreprocessor.variables; }

	XMLDomPreprocessor& getXMLDomPreprocessor() { return mXMLPreprocessor; }

	typedef ProtecteeMixin<MetaTypeDatabase> ProtecteeDatabase;
	/**
	 * Return the meta database that contains all known meta information in
	 * this framework
	 */
	ScopedAccess<ProtecteeDatabase> getMetaDatabase()
	{
		return ScopedAccess<ProtecteeDatabase>(&mMetaDatabase);
	}

	bool isInExec() const { return mInExec; }

protected:

	friend class Private::FrameworkStartup;

	void initialize();
	void finalize();
	void ctrlCHandler(const IntSignal& sig);

	static void errorHandler(const IntSignal& sig);
	static bool enterLeaveErrorHandler(bool enter);

protected:

	bool mTerminationRequested;
	int  mTerminationExitCode;
	bool mIsStarted;
	bool mInExec;
	bool mRemoteDisabled;

	// The authority of the framework (used for all services of the framework
	// as well as providing a process thread for submodules)
	boost::shared_ptr<Private::FrameworkAuthority> mAuthority;

	RootPropertyNode mPropertiesRoot;
	boost::shared_ptr<PropertyNode> mProperties;

	// our components:
	NameRegistry mNames;
	boost::shared_ptr<RemoteModule> mRemoteModule;

	// DO NOT CHANGE THE ORDER of the following components, the order of
	// destruction is important for producing correct error messages when
	// destroying the framework and errors occur. If the order is not correct,
	// it could result in misleading error messages.
	RPCManager mRPCManager;          // create RPCManager and ChannelManager first
	ChannelManager mChannelManager;  // to bring up the communication components
	                                 // that may be used by the components below.
	AuthorityManager mAuthorityManager;
	boost::shared_ptr<FrameworkTransformer> mTransformer;

	boost::shared_ptr<UnitManager> mUnitManager;

	ConfigurationLoader mConfigurationLoader;

	boost::shared_ptr<ErrorService> mErrorServiceModule;
	std::string mName;

	XMLDomPreprocessor mXMLPreprocessor;
	XMLDom mConfigDom;
	ProtecteeDatabase mMetaDatabase;

	Time mStartTime;
};

///////////////////////////////////////////////////////////////////////////////

}

#include <fw/impl/Authority.hpp>
#include <fw/impl/ConcreteChannel.hpp>

#endif
