/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file AbstractAuthority.h
 *    Abstract base class of authorities
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/10/04
 */

#ifndef _MIRA_ABSTRACTAUTHORITY_H_
#define _MIRA_ABSTRACTAUTHORITY_H_

#include <set>

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#endif

#include <fw/AuthorityDescription.h>
#include <fw/Status.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// forward declaration
class PropertyNode;
// forward declaration
class MIRA_FRAMEWORK_EXPORT StatusManager;


/**
 * @ingroup FWModule
 * This is the abstract base class of local and remote authority classes.
 * It provides access to properties of the authority like name, id, namespace
 * and status informations.
 * The methods of this class are implemented by Authority and RemoteAuthority.
 */
class MIRA_FRAMEWORK_EXPORT AbstractAuthority : boost::noncopyable
{
public:

	virtual ~AbstractAuthority() {}

	/**
	 * @name Detailed Information Retrieval
	 * These methods allow to obtain detailed information on the Authority.
	 */
	//@{

	/**
	 * Return the resolved namespace of the authority
	 * @return Namespace
	 */
	std::string getNamespace() const {
		return mDescription.ns;
	}

	/**
	 * Return the fully qualified global name (includes namespace)
	 * This method returns the name without the trailing UUID added
	 * if the authority is anonymous.
	 * Use getGlobalID() to obtain the unique id of the authority.
	 * @return Global Name
	 */
	std::string getGlobalName() const {
		return mDescription.getGlobalName();
	}

	/**
	 * Return the fully qualified global id (includes namespace)
	 * @return Global ID
	 */
	std::string getGlobalID() const {
		return mDescription.getGlobalID();
	}

	/**
	 * Return the id of the authority
	 * @return ID
	 */
	std::string getID() const {
		return mDescription.id;
	}

	/**
	 * Return the name of the authority
	 * @return Name of this authority
	 */
	std::string getName() const {
		return mDescription.name;
	}

	/**
	 * Return the description of this authority.
	 */
	const AuthorityDescription& getDescription() const { return mDescription; }

	//@}

public:
	typedef std::multimap<std::string, std::string> ChannelNameMapping;

	typedef std::map<std::string, Status> ChannelStatusMap;

	/**
	 * Return a status map that contains the status of the authority as well as
	 * all registered submodules
	 */
	virtual StatusManager::StatusMap getStatusMap() = 0;

	/// Returns the main property node of this Authority.
	virtual boost::shared_ptr<PropertyNode> getProperties() = 0;

	/// Returns all published channels by this authority (and its sub-authorities)
	virtual std::set<std::string> getPublishedChannels() = 0;

	/// Returns all subscribed channels by this authority (and its sub-authorities)
	virtual std::set<std::string> getSubscribedChannels() = 0;

	/// Returns all service interfaces provided by this authority
	virtual std::set<std::string> getServiceInterfaces() = 0;

	/**
	 * Returns all published channels (pair of global and local names)
	 * by this authority (and its sub-authorities)
	 */
	virtual ChannelNameMapping getPublishedChannelNames() = 0;

	/**
	 * Returns all subscribed channels (pair of global and local names)
	 * by this authority (and its sub-authorities)
	 */
	virtual ChannelNameMapping getSubscribedChannelNames() = 0;

	/**
	 * Returns status for all subscribed channels (ok, no publisher, no data, ...)
	 * by this authority (and its sub-authorities)
	 */
	virtual ChannelStatusMap getSubscribedChannelStatus() = 0;

public:

	/// Starts/Resumes the authority.
	virtual void start() = 0;

	/// Stops/Pauses the authority. To continue call start().
	virtual void stop() = 0;

	/**
	 * Returns true, if there was a failure while processing the immediate
	 * handlers.
	 * This is an unrecoverable failure that will disallow the further execution
	 * of the authority.
	 */
	virtual bool hasUnrecoverableFailure() const = 0;

	/**
	 * Returns true, if the authority was started (by calling start()).
	 * It will return false, if the authority was not yet started or stopped.
	 */
	virtual bool isStarted() const = 0;

	/// Return if the authority is running
	virtual bool isRunning() const = 0;

protected:

	AuthorityDescription mDescription;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
