/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TemplateClasses.h
 *
 * @author Ronny Stricker
 * @date   2011/03/14
 */

#ifndef _MIRA_TEMPLATECLASSES_H_
#define _MIRA_TEMPLATECLASSES_H_

#include <serialization/Serialization.h>
#include <transform/Pose.h>

using namespace std;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

template <class CLASSTYPE>
class BaseTemplateClass : public mira::Object
{
MIRA_OBJECT(BaseTemplateClass<CLASSTYPE>)
public:

	BaseTemplateClass() : i(0) {}
	BaseTemplateClass(bool) : i(12345) {}

	virtual void check() {
		BOOST_CHECK_EQUAL(i, 12345);
	}

	virtual bool isDerived() {return false;}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
	}

public:
	CLASSTYPE i;
};

template <class CLASSTYPE>
class DerivedTemplateClass : public BaseTemplateClass<CLASSTYPE>
{
MIRA_OBJECT(DerivedTemplateClass<CLASSTYPE>)
public:

	DerivedTemplateClass() : j(0) {}
	DerivedTemplateClass(bool) : BaseTemplateClass<CLASSTYPE>(true), j(54321) {}

	virtual void check() {
		BaseTemplateClass<CLASSTYPE>::check();
		BOOST_CHECK_EQUAL(j, 54321);
	}

	virtual bool isDerived() {return true;}


	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BaseTemplateClass<CLASSTYPE>);
		r.member("j", j, "");
	}


	CLASSTYPE j;
};

template <class CLASSTYPE1, class CLASSTYPE2>
class DerivedMultiTemplateClass : public BaseTemplateClass<CLASSTYPE1>
{
MIRA_OBJECT(MIRA_PPARAM(DerivedMultiTemplateClass<CLASSTYPE1,CLASSTYPE2>))
public:

	DerivedMultiTemplateClass() : j(0), k(0) {}
	DerivedMultiTemplateClass(bool) : BaseTemplateClass<CLASSTYPE1>(true),
			j(54321), k(11111) {}

	virtual void check() {
		BaseTemplateClass<CLASSTYPE1>::check();
		BOOST_CHECK_EQUAL(j, 54321);
		BOOST_CHECK_EQUAL(k, 11111);
	}

	virtual bool isDerived() {return true;}


	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BaseTemplateClass<CLASSTYPE1>);
		r.member("j", j, "");
		r.member("k", k, "");
	}


	CLASSTYPE1 j;
	CLASSTYPE2 k;
};

// Test MIRA_ABSTRACT_OBJECT for template class with multiple template params
template <class CLASSTYPE1, class CLASSTYPE2>
class AbstractMultiTemplateClass : public BaseTemplateClass<CLASSTYPE1>
{
// a class resulting from a specialization of this template is abstract!
MIRA_ABSTRACT_OBJECT(MIRA_PPARAM(AbstractMultiTemplateClass<CLASSTYPE1,CLASSTYPE2>))
public:

	AbstractMultiTemplateClass() {}
	AbstractMultiTemplateClass(bool) : BaseTemplateClass<CLASSTYPE1>(true) {}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		MIRA_REFLECT_BASE(r, BaseTemplateClass<CLASSTYPE1>);
	}

	virtual bool isDerived() {return true;}

	virtual void makeAbstract() = 0;
};

template <class CLASSTYPE1, class CLASSTYPE2>
class NonAbstractMultiTemplateClass : public AbstractMultiTemplateClass<CLASSTYPE1, CLASSTYPE2>
{
	// a class resulting from a specialization of this template is not abstract!
	MIRA_OBJECT(MIRA_PPARAM(NonAbstractMultiTemplateClass<CLASSTYPE1,CLASSTYPE2>))
public:

	NonAbstractMultiTemplateClass() : j(0) {}
	NonAbstractMultiTemplateClass(bool) : AbstractMultiTemplateClass<CLASSTYPE1,CLASSTYPE2>(true),
			j(54321), k(1.f, 2.f, 3.f) {}

	virtual void check() {
		BaseTemplateClass<CLASSTYPE1>::check();
		BOOST_CHECK_EQUAL(j, 54321);
		BOOST_CHECK_EQUAL(k.x(), 1.f);
		BOOST_CHECK_EQUAL(k.y(), 2.f);
		BOOST_CHECK_EQUAL(k.phi(), 3.f);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		typedef AbstractMultiTemplateClass<CLASSTYPE1,CLASSTYPE2> Base;
		MIRA_REFLECT_BASE(r, Base);
		r.member("j", j, "");
		r.member("k", k, "");
	}

	void makeAbstract() {} // override to make non-abstract


	CLASSTYPE1 j;
	CLASSTYPE2 k;
};

class ClassWithPolymorphicTemplateMember
{
public:

	ClassWithPolymorphicTemplateMember() : object(NULL) {}
	ClassWithPolymorphicTemplateMember(bool)
	{
		object = new DerivedTemplateClass<int>(true);
		objects.push_back(new BaseTemplateClass<int>(true));
		objects.push_back(new DerivedTemplateClass<int>(true));
		objects.push_back(new BaseTemplateClass<int>(true));
		objects.push_back(new DerivedMultiTemplateClass<int,int>(true));
		objects.push_back(new NonAbstractMultiTemplateClass<int,Pose2>(true));
	}

	~ClassWithPolymorphicTemplateMember()
	{
		delete object;
		foreach(BaseTemplateClass<int>* obj, objects)
			delete obj;
	}

	void check() {
		BOOST_REQUIRE(object!=NULL);
		// check if we have the correct derived object
		BOOST_CHECK_EQUAL(object->isDerived(), true);

		// check the object itself
		object->check();

		BOOST_REQUIRE_EQUAL(objects.size(),5);

		for(size_t i=0; i<objects.size(); ++i)
		{
			BOOST_REQUIRE(objects[i]!=NULL);
			objects[i]->check();
		}

		BOOST_CHECK_EQUAL(objects[0]->isDerived(), false);
		BOOST_CHECK_EQUAL(objects[1]->isDerived(), true);
		BOOST_CHECK_EQUAL(objects[2]->isDerived(), false);
		BOOST_CHECK_EQUAL(objects[3]->isDerived(), true);
		BOOST_CHECK_EQUAL(objects[4]->isDerived(), true);
	}


	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("object", object, "");
		r.member("objects", objects, "");
	}

private:

	BaseTemplateClass<int>* object;
	vector<BaseTemplateClass<int>*> objects;
};

}

MIRA_TEMPLATE_CLASS_SERIALIZATION((mira),BaseTemplateClass,(int), mira::Object);
MIRA_TEMPLATE_CLASS_SERIALIZATION((mira),DerivedTemplateClass,(int), mira::BaseTemplateClass<int>);
MIRA_VARTEMPLATE_CLASS_SERIALIZATION((mira),DerivedMultiTemplateClass,2,(int,int), mira::BaseTemplateClass<int>);

// Using typedef-name Pose2 as one of the template parameters to test use with MIRA_ABSTRACT_OBJECT
MIRA_VARTEMPLATE_CLASS_SERIALIZATION((mira),AbstractMultiTemplateClass,2,(int,mira::Pose2), mira::BaseTemplateClass<int>);
MIRA_VARTEMPLATE_CLASS_SERIALIZATION((mira),NonAbstractMultiTemplateClass,2,(int,mira::Pose2), mira::AbstractMultiTemplateClass<int,mira::Pose2>);
//////////////////////////////////////////////////////////////////////////////

#endif /* _MIRA_TEMPLATECLASSES_H_ */

