/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SerializationBasicTest.C
 *    Test for serialization framework.
 *
 * @author Erik Einhorn
 */

#include "BasicClasses.h"
#include "CommonTest.h"

#include <error/LogTxtStreamSink.h>
#include <serialization/adapters/std/vector>

using namespace std;
using namespace mira;


BOOST_AUTO_TEST_CASE( TestClassNormal )
{
	testAll<ClassNormal>("Basic",1);
}

BOOST_AUTO_TEST_CASE( TestClassWithSetter )
{
	testAll<ClassWithSetter>("Basic",3);
}

BOOST_AUTO_TEST_CASE( TestClassWithGetterAndSetter )
{
	testAll<ClassWithGetterAndSetter>("Basic",4);
}


BOOST_AUTO_TEST_CASE( TestClassWithVersionA )
{
	testAll<ClassWithVersionA>("Basic",5);
}


BOOST_AUTO_TEST_CASE( TestClassWithVersionAFromString )
{
	string xml =
			"<Root>"
			"	<Test version=\"3\">"
			"	<i>123</i>"
			"	<j>543</j>"
			"	</Test>"
			"</Root>";

	// will throw exception since version is not 4
	BOOST_CHECK_THROW(testXmlFromString<ClassWithVersionA>(xml), mira::XIO);
}


BOOST_AUTO_TEST_CASE( TestClassWithVersionB )
{
	testAll<ClassWithVersionB>("Basic",6);
}


BOOST_AUTO_TEST_CASE( TestClassWithVersionBFromString )
{
	string xml =
			"<Root>"
			"	<Test version=\"3\">"
			"	<i>123</i>"
			"	</Test>"
			"</Root>";

	testXmlFromString<ClassWithVersionB>(xml);
}

BOOST_AUTO_TEST_CASE( TestClassWithoutDefaultValue )
{
	string xml =
			"<Root>"
			"   <Test>"
			"	  <i>123</i>"
			"	</Test>"
			"</Root>";

	BOOST_CHECK_THROW(testXmlFromString<ClassWithoutDefaultValue>(xml), XIO);
}

BOOST_AUTO_TEST_CASE( TestClassWithDefaultValue )
{
	string xml =
			"<Root>"
			"   <Test>"
			"	  <i>123</i>"
			"	</Test>"
			"</Root>";

	MIRA_LOGGER.registerSink(LogTxtStreamSink(std::cout));
	testXmlFromString<ClassWithDefaultValue>(xml);
}

BOOST_AUTO_TEST_CASE( TestVectorWithBinaryStream )
{
	ofstream fs("test.bin",ofstream::out);
	BinaryStlOstream bos(fs);
	BinaryStreamSerializer s(bos);

	vector<int> vals;
	vals.resize(10);
	s.serialize(vals);
}


class NestedClassWithoutDefaultValue
{
public:
	NestedClassWithoutDefaultValue(int v=0) : i(v) {}
	void check() {
		BOOST_CHECK_EQUAL(i, 123);
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("i", i, "");
	}

private:
	int i;
};

class ContainerOfNestedClassWithoutDefaultValue
{
public:

	ContainerOfNestedClassWithoutDefaultValue()  {}
	void check() { n.check(); }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Nested", n, "", NestedClassWithoutDefaultValue(222));
	}

private:
	NestedClassWithoutDefaultValue n;
};

BOOST_AUTO_TEST_CASE( TestNestedClassWithoutDefaultValue )
{
	string xml =
			"<Root>"
			"   <Test>"
			"      <Nested>"
			"      </Nested>"
			"	</Test>"
			"</Root>";

	//testXmlFromString<ContainerOfNestedClassWithoutDefaultValue>(xml);
	BOOST_CHECK_THROW(testXmlFromString<ContainerOfNestedClassWithoutDefaultValue>(xml), XIO);
}

class AmbiguousReflection
{
public:

	AmbiguousReflection(): n(1) {}
	void check() { BOOST_CHECK_EQUAL(n, 0); }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
//		r.member("N", n, "", {}); // this might be the intuitive attempt by the class developer,
		                          // but the compiler rejects it as ambiguous. see #910
		r.member("N", n, "", {}, REFLECT_CTRLFLAG_NONE);
	}

private:
	int n;
};

BOOST_AUTO_TEST_CASE( Ticket910Test )
{
	string xml =
			"<Root>"
			"   <Test>"
			"<!--      <N>1</N>-->" // value for N is missing, assign default = int{}
			"   </Test>"
			"</Root>";

	testXmlFromString<AmbiguousReflection>(xml);
}

