/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCBenchmark.C
 *    Test/benchmark for RPC.
 *
 * @author Christof Schröter
 * @date   2020/10/09
 */

#include <boost/test/unit_test.hpp>

#include <serialization/Print.h>

#include <rpc/RPCManager.h>



using namespace mira;

/////////////////////////////////////////////////////////////////////////////////

// only default + copy constructor
struct TestClassNoMove
{
	TestClassNoMove() { ++sCreated; }
	TestClassNoMove(const TestClassNoMove& other) { value = other.value; ++sCopied; }
	
	static void reset() { sCreated = 0; sCopied = 0; }
	
	TestClassNoMove& operator=(const TestClassNoMove& other) { value = other.value; ++sCopied; return *this; }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Value", value, "");
		r.member("TestClassNoMove created", sCreated, "");
		r.member("TestClassNoMove copied", sCopied, "");
	}

	int value;
	static int sCreated;
	static int sCopied;
};

int TestClassNoMove::sCreated = 0;
int TestClassNoMove::sCopied = 0;

/////////////////////////////////////////////////////////////////////////////////

// default, copy and move constructor
struct TestClassMove
{
	TestClassMove() { ++sCreated; }
	TestClassMove(const TestClassMove& other) { value = other.value; ++sCopied; }
	TestClassMove(TestClassMove&& other) { std::swap(value, other.value); ++sMoved; }

	static void reset() { sCreated = 0; sCopied = 0; sMoved = 0; }

	TestClassMove& operator=(const TestClassMove& other) { value = other.value; ++sCopied; return *this; }
	TestClassMove& operator=(TestClassMove&& other) { std::swap(value, other.value); ++sMoved; return *this; }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.member("Value", value, "");
		r.member("TestClassMove created", sCreated, "");
		r.member("TestClassMove copied", sCopied, "");
		r.member("TestClassMove moved", sMoved, "");
	}

	int value;
	static int sCreated;
	static int sCopied;
	static int sMoved;
};

int TestClassMove::sCreated = 0;
int TestClassMove::sCopied = 0;
int TestClassMove::sMoved = 0;

/////////////////////////////////////////////////////////////////////////////////

void functionNoMoveByValue(TestClassNoMove t)
{
	std::cout << "freefunction_ParamNoMove_ByValue()" << std::endl;
	std::cout << print(t) << std::endl;
	#if defined(__GNUC__) && (__GNUC__ >= 50000)
		BOOST_CHECK_LE(TestClassNoMove::sCopied, 2);
	#else // gcc < 5 is less efficient
		BOOST_CHECK_LE(TestClassNoMove::sCopied, 3);
	#endif
	TestClassNoMove::reset();
}

void functionNoMoveByConstRef(const TestClassNoMove& t)
{
	std::cout << "freefunction_ParamNoMove_ByConstRef()" << std::endl;
	std::cout << print(t) << std::endl;
	BOOST_CHECK_EQUAL(TestClassNoMove::sCopied, 0);
	TestClassNoMove::reset();
}

void functionMoveByValue(TestClassMove t)
{
	std::cout << "freefunction_ParamMove_ByValue()" << std::endl;
	std::cout << print(t) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	#if defined(__GNUC__) && (__GNUC__ >= 50000)
		BOOST_CHECK_LE(TestClassMove::sMoved, 2);
	#else // gcc < 5 is less efficient
		BOOST_CHECK_LE(TestClassMove::sMoved, 3);
	#endif
	TestClassMove::reset();
}

void functionMoveByConstRef(const TestClassMove& t)
{
	std::cout << "freefunction_ParamMove_ByConstRef()" << std::endl;
	std::cout << print(t) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	BOOST_CHECK_EQUAL(TestClassMove::sMoved, 0);
	TestClassMove::reset();
}

TestClassNoMove functionReturnNoMoveByValue()
{
	TestClassNoMove t;
	t.value = 123;
	return t;
}

const TestClassNoMove& functionReturnNoMoveByConstRef()
{
	static TestClassNoMove s;
	s.value = 123;
	return s;
}

TestClassMove functionReturnMoveByValue()
{
	TestClassMove t;
	t.value = 123;
	return t;
}

const TestClassMove& functionReturnMoveByConstRef()
{
	static TestClassMove s;
	s.value = 123;
	return s;
}

/////////////////////////////////////////////////////////////////////////////////

class ServiceParams
{
public:

	ServiceParams() { mNoMove.value = 234; mMove.value = 345; }

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.interface("IBenchmarkParams");
		r.method("memFnNoMoveByValue", &ServiceParams::methodNoMoveByValue, this, "", "t", "");
		r.method("memFnNoMoveByConstRef", &ServiceParams::methodNoMoveByConstRef, this, "", "t", "");
		r.method("memFnMoveByValue", &ServiceParams::methodMoveByValue, this, "", "t", "");
		r.method("memFnMoveByConstRef", &ServiceParams::methodMoveByConstRef, this, "", "t", "");
		r.method("memFnReturnNoMoveByValue", &ServiceParams::methodReturnNoMoveByValue, this, "");
		r.method("memFnReturnNoMoveByConstRef", &ServiceParams::methodReturnNoMoveByConstRef, this, "");
		r.method("memFnReturnMoveByValue", &ServiceParams::methodReturnMoveByValue, this, "");
		r.method("memFnReturnMoveByConstRef", &ServiceParams::methodReturnMoveByConstRef, this, "");
		r.method("freeFnNoMoveByValue", functionNoMoveByValue, "", "t", "");
		r.method("freeFnNoMoveByConstRef", functionNoMoveByConstRef, "", "t", "");
		r.method("freeFnMoveByValue", functionMoveByValue, "", "t", "");
		r.method("freeFnMoveByConstRef", functionMoveByConstRef, "", "t", "");
		r.method("freeFnReturnNoMoveByValue", functionReturnNoMoveByValue, "");
		r.method("freeFnReturnNoMoveByConstRef", functionReturnNoMoveByConstRef, "");
		r.method("freeFnReturnMoveByValue", functionReturnMoveByValue, "");
		r.method("freeFnReturnMoveByConstRef", functionReturnMoveByConstRef, "");
	}

	void methodNoMoveByValue(TestClassNoMove t)
	{
		std::cout << "ServiceParams::method_ParamNoMove_ByValue()" << std::endl;
		std::cout << print(t) << std::endl;
		BOOST_CHECK_EQUAL(TestClassNoMove::sCopied, 1);
		TestClassNoMove::reset();
	}

	void methodNoMoveByConstRef(const TestClassNoMove& t)
	{
		std::cout << "ServiceParams::method_ParamNoMove_ByConstRef()" << std::endl;
		std::cout << print(t) << std::endl;
		BOOST_CHECK_EQUAL(TestClassNoMove::sCopied, 0);
		TestClassNoMove::reset();
	}

	void methodMoveByValue(TestClassMove t)
	{
		std::cout << "ServiceParams::method_ParamMove_ByValue()" << std::endl;
		std::cout << print(t) << std::endl;
		BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
		BOOST_CHECK_EQUAL(TestClassMove::sMoved, 1);
		TestClassMove::reset();
	}

	void methodMoveByConstRef(const TestClassMove& t)
	{
		std::cout << "ServiceParams::method_ParamMove_ByConstRef()" << std::endl;
		std::cout << print(t) << std::endl;
		BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
		BOOST_CHECK_EQUAL(TestClassMove::sMoved, 0);
		TestClassMove::reset();
	}

	TestClassNoMove methodReturnNoMoveByValue() const
	{
		TestClassNoMove t;
		t.value = 123;
		return t;
	}

	const TestClassNoMove& methodReturnNoMoveByConstRef() const
	{
		return mNoMove;
	}

	TestClassMove methodReturnMoveByValue() const
	{
		TestClassMove t;
		t.value = 123;
		return t;
	}

	const TestClassMove& methodReturnMoveByConstRef() const
	{
		return mMove;
	}

	TestClassNoMove mNoMove;
	TestClassMove mMove;
};


///////////////////////////////////////////////////////////////////////////////

class RPCHandler : public AbstractRPCHandler
{
public:
	virtual void queueRequest(AbstractDeferredInvokerPtr invoker)
	{
		invoker->invoke();
	}

	virtual boost::thread::id getThreadID() const
	{
		return boost::this_thread::get_id();
	}
};

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE(Benchmark)
{
	RPCManager mgr;

	AbstractRPCHandlerPtr handler(new RPCHandler);

	ServiceParams serviceParams;

	mgr.addLocalService("ServiceParams", serviceParams, handler);

	BOOST_REQUIRE_EQUAL(mgr.existsService("ServiceParams"), true);

	/////////////////////////////////////////////////////////////////

	TestClassNoMove tNoMove;
	tNoMove.value = 123;
	TestClassMove tMove;
	tMove.value = 123;

	std::cout << "service registered for TestClassNoMove" << std::endl;
	std::cout << print(tNoMove) << std::endl;
	BOOST_CHECK_EQUAL(TestClassNoMove::sCopied, 0);
	TestClassNoMove::reset();

	std::cout << "service registered for TestClassMove" << std::endl;
	std::cout << print(tMove) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	TestClassMove::reset();

	/////////////////////////////////////////////////////////////////

	mgr.call<void>("ServiceParams", "memFnNoMoveByValue", tNoMove);

	tNoMove.value = 357;
	mgr.call<void>("ServiceParams", "memFnNoMoveByConstRef", tNoMove);

	tNoMove.value = 321;
	mgr.call<void>("ServiceParams", "freeFnNoMoveByValue", tNoMove);

	tNoMove.value = 753;
	mgr.call<void>("ServiceParams", "freeFnNoMoveByConstRef", tNoMove);

	/////////////////////////////////////////////////////////////////

	mgr.call<void>("ServiceParams", "memFnMoveByValue", tMove);

	tMove.value = 357;
	mgr.call<void>("ServiceParams", "memFnMoveByConstRef", tMove);

	tMove.value = 321;
	mgr.call<void>("ServiceParams", "freeFnMoveByValue", tMove);

	tMove.value = 753;
	mgr.call<void>("ServiceParams", "freeFnMoveByConstRef", tMove);

	/////////////////////////////////////////////////////////////////

	tNoMove = mgr.call<TestClassNoMove>("ServiceParams", "memFnReturnNoMoveByValue").get();
	std::cout << "ServiceParams::method_ReturnNoMoveByValue()" << std::endl;
	std::cout << print(tNoMove) << std::endl;
	BOOST_CHECK_LE(TestClassNoMove::sCopied, 3);
	TestClassNoMove::reset();

	const TestClassNoMove& tNoMoveRef = mgr.call<TestClassNoMove>("ServiceParams", "memFnReturnNoMoveByConstRef").get();
	std::cout << "ServiceParams::method_ReturnNoMoveByConstRef()" << std::endl;
	std::cout << print(tNoMoveRef) << std::endl;
	BOOST_CHECK_LE(TestClassNoMove::sCopied, 2);
	TestClassNoMove::reset();

	tNoMove = mgr.call<TestClassNoMove>("ServiceParams", "freeFnReturnNoMoveByValue").get();
	std::cout << "freefunction_ReturnNoMoveByValue()" << std::endl;
	std::cout << print(tNoMove) << std::endl;
	BOOST_CHECK_LE(TestClassNoMove::sCopied, 3);
	TestClassNoMove::reset();

	const TestClassNoMove& tNoMoveRef2 = mgr.call<TestClassNoMove>("ServiceParams", "freeFnReturnNoMoveByConstRef").get();
	std::cout << "freefunction_ReturnNoMoveByConstRef()" << std::endl;
	std::cout << print(tNoMoveRef) << std::endl;
	BOOST_CHECK_LE(TestClassNoMove::sCopied, 2);
	TestClassNoMove::reset();

	/////////////////////////////////////////////////////////////////

	tMove = mgr.call<TestClassMove>("ServiceParams", "memFnReturnMoveByValue").get();
	std::cout << "ServiceParams::method_ReturnMoveByValue()" << std::endl;
	std::cout << print(tMove) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	BOOST_CHECK_LE(TestClassMove::sMoved, 3);
	TestClassMove::reset();

	const TestClassMove& tMoveRef = mgr.call<TestClassMove>("ServiceParams", "memFnReturnMoveByConstRef").get();
	std::cout << "ServiceParams::method_ReturnMoveByConstRef()" << std::endl;
	std::cout << print(tMoveRef) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	BOOST_CHECK_LE(TestClassMove::sMoved, 2);
	TestClassMove::reset();

	tMove = mgr.call<TestClassMove>("ServiceParams", "freeFnReturnMoveByValue").get();
	std::cout << "freefunction_ReturnMoveByValue()" << std::endl;
	std::cout << print(tMove) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	BOOST_CHECK_LE(TestClassMove::sMoved, 3);
	TestClassMove::reset();

	const TestClassMove& tMoveRef2 = mgr.call<TestClassMove>("ServiceParams", "freeFnReturnMoveByConstRef").get();
	std::cout << "freefunction_ReturnMoveByConstRef()" << std::endl;
	std::cout << print(tMoveRef) << std::endl;
	BOOST_CHECK_EQUAL(TestClassMove::sCopied, 0);
	BOOST_CHECK_LE(TestClassMove::sMoved, 2);
	TestClassMove::reset();

	/////////////////////////////////////////////////////////////////


}

