/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TypenameTest.C
 *    Typename test cases.
 *
 * @author Tim Langner
 * @date   2010/10/20
 */

#include <vector>
#include <map>

#include <boost/test/unit_test.hpp>

#include <platform/Types.h>
#include <platform/Typename.h>

#include <transform/RigidTransform.h>

#include <iostream>

using namespace mira;

template <typename>
struct MyStruct
{

};

namespace {

struct StructInAnonymousNamespace {};

}

enum MyEnum
{
	VALUE1 = 0
};

struct Elongation {};

BOOST_AUTO_TEST_CASE( TestTypename )
{
	// test basic types
	BOOST_CHECK_EQUAL(typeName<char>(), "char");
	BOOST_CHECK_EQUAL(typeName<uint8>(), "unsigned char");
	BOOST_CHECK_EQUAL(typeName<uint16>(), "unsigned short");
	BOOST_CHECK_EQUAL(typeName<uint32>(), "unsigned int");
	BOOST_CHECK_EQUAL(typeName<uint64>(), "unsigned int64");
	BOOST_CHECK_EQUAL(typeName<int8>(), "signed char");
	BOOST_CHECK_EQUAL(typeName<int16>(), "short");
	BOOST_CHECK_EQUAL(typeName<int32>(), "int");
	BOOST_CHECK_EQUAL(typeName<int64>(), "int64");
	BOOST_CHECK_EQUAL(typeName<float>(), "float");
	BOOST_CHECK_EQUAL(typeName<double>(), "double");
	BOOST_CHECK_EQUAL(typeName<long double>(), "double128");
	BOOST_CHECK_EQUAL(typeName<std::string>(), "std::string");
	BOOST_CHECK_EQUAL(typeName<std::out_of_range>(), "std::out_of_range");

	BOOST_CHECK_EQUAL(typeName<MyEnum>(), "MyEnum");

	// test reference and pointers
	BOOST_CHECK_EQUAL(typeName<int64&>(), "int64");
	typedef int const* const ConstIntConstPtr;
	BOOST_CHECK_EQUAL(typeName<ConstIntConstPtr>(), "int const*");
	BOOST_CHECK_EQUAL(typeName<MyStruct<ConstIntConstPtr> >(), 
	                  "MyStruct<int const*const>");
	BOOST_CHECK_EQUAL(typeName<ConstIntConstPtr*>(), "int const*const*");
	BOOST_CHECK_EQUAL(typeName<const MyStruct<ConstIntConstPtr>* >(), 
	                  "MyStruct<int const*const>const*");

	// test containers
	BOOST_CHECK_EQUAL(typeName<std::vector<int> >(), 
	                  "std::vector<int,std::allocator<int>>");
	BOOST_CHECK_EQUAL(typeName<std::vector<const int*> >(), 
	                  "std::vector<int const*,std::allocator<int const*>>");
	typedef std::map<std::string, RigidTransform2f> PoseMap;
	BOOST_CHECK_EQUAL(typeName<PoseMap>(), 
	                  "std::map<std::string,mira::RigidTransform<float,2>,"
	                  "std::less<std::string>,"
	                  "std::allocator<std::pair<std::string const,"
	                  "mira::RigidTransform<float,2>>>>");

	typedef std::map<int, Eigen::Matrix4f, std::less<int>,
		Eigen::aligned_allocator<Eigen::Matrix4f> > Matrix4Map;
	BOOST_CHECK_EQUAL(typeName<Matrix4Map>(), 
	                  "std::map<int,Eigen::Matrix<float,4,4,0,4,4>,"
	                  "std::less<int>,"
	                  "Eigen::aligned_allocator<Eigen::Matrix<float,4,4,0,4,4>>>");

	// test special cases like anonymous namespaces and volatile
	BOOST_CHECK_EQUAL(typeName<StructInAnonymousNamespace>(), 
	                  "(anonymous namespace)::StructInAnonymousNamespace");
	BOOST_CHECK_EQUAL(typeName<std::list<StructInAnonymousNamespace> >(), 
	                  "std::list<(anonymous namespace)::StructInAnonymousNamespace,"
	                  "std::allocator<(anonymous namespace)::StructInAnonymousNamespace>>");
	BOOST_CHECK_EQUAL(typeName<MyStruct<volatile int> >(), 
	                  "MyStruct<int volatile>");

	BOOST_CHECK_EQUAL(typeName<Elongation>(), "Elongation");


}
